/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.calendar.bean.ScheduleTableConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.jobmanagement.bean.ForwardFileTableDefine;
import com.clustercontrol.jobmanagement.bean.HistoryTableDefine;
import com.clustercontrol.jobmanagement.bean.JobDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.JobTableDefine;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.NodeDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleDetailListTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleListTableDefine;
import com.clustercontrol.jobmanagement.bean.ScheduleInfo;
import com.clustercontrol.jobmanagement.bean.ScheduleTableDefine;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.factory.HistoryFilterProperty;
import com.clustercontrol.jobmanagement.factory.JobOperationProperty;
import com.clustercontrol.jobmanagement.factory.ModifyJob;
import com.clustercontrol.jobmanagement.factory.ModifySchedule;
import com.clustercontrol.jobmanagement.factory.SelectJob;
import com.clustercontrol.jobmanagement.factory.SelectRunSchedule;
import com.clustercontrol.jobmanagement.factory.SelectSchedule;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブ管理機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * @ejb.bean name="JobController"	
 *           jndi-name="JobController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=QuartzManager"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CalendarController"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class JobControllerBean implements javax.ejb.SessionBean, CheckFacility {
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( JobControllerBean.class );
    
	/** コンテキスト情報 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * ジョブ[履歴]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.HistoryTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getHistoryTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(HistoryTableDefine.STATUS, 
				new TableColumnInfo(Messages.getString("run.status", locale), TableColumnInfo.STATE, 80, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.END_STATUS, 
				new TableColumnInfo(Messages.getString("end.status", locale), TableColumnInfo.END_STATUS, 80, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.END_VALUE, 
				new TableColumnInfo(Messages.getString("end.value", locale), TableColumnInfo.NONE, 50, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.SESSION_ID, 
				new TableColumnInfo(Messages.getString("session.id", locale), TableColumnInfo.NONE, 140, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.JOB_TYPE, 
				new TableColumnInfo(Messages.getString("type", locale), TableColumnInfo.JOB, 110, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.SCOPE, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 150, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.SCHEDULED_START_TIME, 
				new TableColumnInfo(Messages.getString("scheduled.start.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.START_RERUN_TIME, 
				new TableColumnInfo(Messages.getString("start.rerun.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(HistoryTableDefine.END_SUSPEND_TIME, 
				new TableColumnInfo(Messages.getString("end.suspend.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ジョブ[詳細]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.JobDetailTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getDetailTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(JobDetailTableDefine.TREE, 
				new TableColumnInfo("", TableColumnInfo.NONE, 30, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.STATUS, 
				new TableColumnInfo(Messages.getString("run.status", locale), TableColumnInfo.STATE, 80, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.END_STATUS, 
				new TableColumnInfo(Messages.getString("end.status", locale), TableColumnInfo.END_STATUS, 80, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.END_VALUE, 
				new TableColumnInfo(Messages.getString("end.value", locale), TableColumnInfo.NONE, 50, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.JOB_TYPE, 
				new TableColumnInfo(Messages.getString("type", locale), TableColumnInfo.JOB, 110, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.SCOPE, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 150, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.WAIT_RULE_TIME, 
				new TableColumnInfo(Messages.getString("wait.rule.time", locale), TableColumnInfo.NONE, 120, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.START_RERUN_TIME, 
				new TableColumnInfo(Messages.getString("start.rerun.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(JobDetailTableDefine.END_SUSPEND_TIME, 
				new TableColumnInfo(Messages.getString("end.suspend.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ジョブ[一覧]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.JobTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getJobTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(JobTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(JobTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(JobTableDefine.JOB_TYPE, 
				new TableColumnInfo(Messages.getString("type", locale), 	TableColumnInfo.JOB, 110, SWT.LEFT));
		tableDefine.add(JobTableDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(JobTableDefine.SCOPE, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 150, SWT.LEFT));
		tableDefine.add(JobTableDefine.WAIT_RULE, 
				new TableColumnInfo(Messages.getString("wait.rule", locale), TableColumnInfo.WAIT_RULE, 120, SWT.LEFT));
		tableDefine.add(JobTableDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(JobTableDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(JobTableDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(JobTableDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		
		return tableDefine;
	}
	
	/**
	 * ジョブ[ノード詳細]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.NodeDetailTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getNodeDetailTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(NodeDetailTableDefine.STATUS, 
				new TableColumnInfo(Messages.getString("run.status", locale), TableColumnInfo.STATE, 80, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.RETURN_VALUE, 
				new TableColumnInfo(Messages.getString("return.value", locale), TableColumnInfo.NONE, 50, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.FACILITY_NAME, 
				new TableColumnInfo(Messages.getString("facility.name", locale), TableColumnInfo.FACILITY, 150, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.START_RERUN_TIME, 
				new TableColumnInfo(Messages.getString("start.rerun.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.END_SUSPEND_TIME, 
				new TableColumnInfo(Messages.getString("end.suspend.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(NodeDetailTableDefine.MESSAGE, 
				new TableColumnInfo(Messages.getString("message", locale), TableColumnInfo.TEXT_DIALOG, 300, SWT.LEFT));
		
		return tableDefine;
	}
	
	/**
	 * ジョブ[スケジュール]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.ScheduleTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getScheduleTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(ScheduleTableDefine.SCHE_ID, 
				new TableColumnInfo(Messages.getString("schedule.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.SCHE_NAME, 
				new TableColumnInfo(Messages.getString("schedule.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.SCHEDULE, 
				new TableColumnInfo(Messages.getString("schedule", locale), TableColumnInfo.SCHEDULE, 100, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.VALID, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 100, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(ScheduleTableDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ジョブ[実行予定]ビューの左側のテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.RunScheduleListTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getRunScheduleListTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(RunScheduleListTableDefine.SCHE_ID, 
				new TableColumnInfo(Messages.getString("schedule.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(RunScheduleListTableDefine.SCHE_NAME, 
				new TableColumnInfo(Messages.getString("schedule.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(RunScheduleListTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(RunScheduleListTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(RunScheduleListTableDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(RunScheduleListTableDefine.CALENDAR_ID + 1, 
				new TableColumnInfo("", TableColumnInfo.CHECKBOX, 0, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ジョブ[実行予定詳細]ビューの左側のテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.RunScheduleDetailListTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getRunScheduleDetailListTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(RunScheduleDetailListTableDefine.TREE, 
				new TableColumnInfo("", TableColumnInfo.NONE, 30, SWT.LEFT));
		tableDefine.add(RunScheduleDetailListTableDefine.JOB_ID, 
				new TableColumnInfo(Messages.getString("job.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(RunScheduleDetailListTableDefine.JOB_NAME, 
				new TableColumnInfo(Messages.getString("job.name", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(RunScheduleDetailListTableDefine.JOB_TYPE, 
				new TableColumnInfo(Messages.getString("type", locale), TableColumnInfo.JOB, 110, SWT.LEFT));
		tableDefine.add(RunScheduleDetailListTableDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar.id", locale), TableColumnInfo.NONE, 80, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * ジョブ[実行予定]ビューの右側のテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param base 基準日
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.calendar.bean.ScheduleTableConstant#DAYS
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getRunScheduleTableDefine(Date base, Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(0, 
				new TableColumnInfo("", TableColumnInfo.NONE, 0, SWT.LEFT));
		for(int i = 0; i < ScheduleTableConstant.DAYS; i++){
			Calendar baseCalendar = Calendar.getInstance();
			baseCalendar.setTime(base);
			baseCalendar.add(Calendar.DAY_OF_MONTH, i);
			Date work = baseCalendar.getTime();
			tableDefine.add(i + 1, 
					new TableColumnInfo(
							DateFormat.getDateInstance(DateFormat.SHORT, locale).format(work), 
							TableColumnInfo.SCHEDULE_ON_OFF, 70, SWT.LEFT));			
		}
		
		return tableDefine;
	}
	
	/**
	 * ジョブ[実行予定詳細]ビューの右側のテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param base 基準日
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.calendar.bean.ScheduleTableConstant#DAYS
	 * @see com.clustercontrol.jobmanagement.bean.RunScheduleDetailTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getRunScheduleDetailTableDefine(Date base, Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(RunScheduleDetailTableDefine.TREE, 
				new TableColumnInfo("", TableColumnInfo.NONE, 0, SWT.LEFT));
		tableDefine.add(RunScheduleDetailTableDefine.JOB_ID, 
				new TableColumnInfo("", TableColumnInfo.NONE, 0, SWT.LEFT));
		for(int i = 0; i < ScheduleTableConstant.DAYS; i++){
			Calendar baseCalendar = Calendar.getInstance();
			baseCalendar.setTime(base);
			baseCalendar.add(Calendar.DAY_OF_MONTH, i);
			Date work = baseCalendar.getTime();
			tableDefine.add(i + 2, 
					new TableColumnInfo(
							DateFormat.getDateInstance(DateFormat.SHORT, locale).format(work), 
							TableColumnInfo.SCHEDULE_ON_OFF, 70, SWT.LEFT));			
		}
		
		return tableDefine;
	}
	
	/**
	 * ジョブ[ファイル転送]ビューのテーブル定義情報を返します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.jobmanagement.bean.ForwardFileTableDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getForwardFileTableDefine(Locale locale) {
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(ForwardFileTableDefine.STATUS, 
				new TableColumnInfo(Messages.getString("run.status", locale), TableColumnInfo.STATE, 80, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.END_STATUS, 
				new TableColumnInfo(Messages.getString("end.status", locale), TableColumnInfo.END_STATUS, 80, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.FILE_NAME, 
				new TableColumnInfo(Messages.getString("file", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.SRC_FACILITY_ID, 
				new TableColumnInfo(Messages.getString("forward.source", locale) + Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.SRC_FACILITY_NAME, 
				new TableColumnInfo(Messages.getString("forward.source", locale) + Messages.getString("facility.name", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.DEST_FACILITY_ID, 
				new TableColumnInfo(Messages.getString("forward.destination", locale) + Messages.getString("facility.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.DEST_FACILITY_NAME, 
				new TableColumnInfo(Messages.getString("forward.destination", locale) + Messages.getString("facility.name", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.START_RERUN_TIME, 
				new TableColumnInfo(Messages.getString("start.rerun.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));
		tableDefine.add(ForwardFileTableDefine.END_SUSPEND_TIME, 
				new TableColumnInfo(Messages.getString("end.suspend.time", locale), TableColumnInfo.NONE, 150, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * 履歴フィルタ用プロパティを返します。
	 * 
	 * @ejb.interface-method
	 * 
     * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return 履歴フィルタ用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.HistoryFilterProperty#getProperty(Locale)
	 */
	public Property getHistoryFilterProperty(Locale locale) {
	    HistoryFilterProperty history = new HistoryFilterProperty();
		return history.getProperty(locale);
	}
	
	/**
	 * セパレータ文字列を取得
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		return SelectJob.SEPARATOR;
	}
	
	/**
	 * ジョブツリー情報を取得する
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTree(boolean, Locale)
	 */
	public JobTreeItem getJobTree(boolean treeOnly, Locale locale) throws FinderException, NamingException {
	    m_log.debug("getJobTree() : treeOnly=" + treeOnly + ", locale=" + locale);
	    
		JobTreeItem item = null;
		try {
			//ジョブツリーを取得
			SelectJob select = new SelectJob();
			item = select.getJobTree(treeOnly, locale);
		} catch (FinderException e) {
			m_log.debug("getJobTree() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * 同一階層のジョブツリー情報を取得する
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param parentJobId 親ジョブID
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTreeOneLevel(String, Locale)
	 */
	public JobTreeItem getJobTreeOneLevel(String parentJobId, Locale locale) throws FinderException, NamingException {
	    m_log.debug("getJobTreeOneLevel() : parentJobId=" + parentJobId + ", locale=" + locale);
	    
		//ジョブツリーを取得
	    JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getJobTreeOneLevel(parentJobId, locale);
		} catch (FinderException e) {
			m_log.debug("getJobTreeOneLevel() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * ジョブツリー情報を登録する
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param item ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJob(JobTreeItem item) throws FinderException, NamingException, RemoveException, CreateException {
	    m_log.debug("registerJob()");
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifyJob modify = new ModifyJob();
	    try {
            modify.registerJob(item, pri.getName());
        } catch (FinderException e) {
            m_log.debug("registerJob() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            m_log.debug("registerJob() : " + e.getMessage());
            throw e;
        } catch (RemoveException e) {
            m_log.debug("registerJob() : " + e.getMessage());
            throw e;
        } catch (CreateException e) {
            m_log.debug("registerJob() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * ジョブ操作開始用プロパティを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作開始用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStartProperty(String, String, String, Locale)
	 */
	public Property getStartProperty(String sessionId, String jobId, String facilityId, Locale locale) {
	    JobOperationProperty jobPropaerty = new JobOperationProperty();
	    return jobPropaerty.getStartProperty(sessionId, jobId, facilityId, locale);
	}
	
	/**
	 * ジョブ操作停止用プロパティを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作停止用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStopProperty(String, String, String, Locale)
	 */
	public Property getStopProperty(String sessionId, String jobId, String facilityId, Locale locale) {
	    JobOperationProperty jobPropaerty = new JobOperationProperty();
	    return jobPropaerty.getStopProperty(sessionId, jobId, facilityId, locale);
	}

	/**
	 * ジョブを実行します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param jobId ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, LogOutputInfo)
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public void runJob(String jobId) throws CreateException, FinderException, NamingException {
	    m_log.debug("runJob() : jobId=" + jobId);
		
	    //ジョブ情報作成
        JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
        String sessionId = null;
        try {
            sessionId = jobController.createJobInfo(jobId, null);
        } catch (CreateException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        } catch (FinderException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        }
        
        //ジョブ実行
	    JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
	    try {
            JobRunManagement.runJob(sessionId, jobId);
        } catch (FinderException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * ジョブを実行します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, LogOutputInfo)
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public void runJob(String jobId, LogOutputInfo info) throws CreateException, FinderException, NamingException {
	    m_log.debug("runJob() : jobId=" + jobId);
	    
	    //ジョブ情報作成
	    JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
	    String sessionId = null;
	    try {
	    	sessionId = jobController.createJobInfo(jobId, info);
	    } catch (CreateException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	throw e;
	    } catch (FinderException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	throw e;
	    } catch (NamingException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	throw e;
	    }
	    
        //ジョブ実行
	    JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
	    try {
            JobRunManagement.runJob(sessionId, jobId);
        } catch (FinderException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * ジョブをスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param jobId ジョブID
	 * @param calendarId カレンダID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(java.lang.String, java.util.Date)
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#runJob(String)
	 */
	public void scheduleRunJob(String jobId, String calendarId) throws CreateException, FinderException, NamingException {
	    m_log.debug("runJob() : jobId=" + jobId + ", calendarId=" + calendarId);
	    
    	//カレンダをチェック
	    try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar = 
					CalendarControllerUtil.getLocalHome().create();
					
				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}
			
			if(!check)
				return;
			
			//ジョブ実行
			runJob(jobId);
		} catch (CreateException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("scheduleRunJob() -> " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ジョブの実行用情報を作成します。
	 * 
     * @ejb.interface-method
     *     view-type="local"
     *     
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
     *     
	 * @ejb.transaction
	 *     type="RequiresNew"
	 * 
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @return セッションID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocalHome#makeSession(java.lang.String, com.clustercontrol.monitor.message.LogOutputInfo)
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionBean#ejbHomeMakeSession(String, LogOutputInfo)
	 */
	public String createJobInfo(String jobId, LogOutputInfo info) throws CreateException, FinderException, NamingException  {
	    m_log.debug("createJobInfo() : jobId=" + jobId);

	    try {
			return JobSessionUtil.getLocalHome().makeSession(jobId, info);
		} catch (CreateException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("createJobInfo() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}

	/**
	 * ジョブ操作を行います。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws FinderException
	 * @throws CreateException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#operationJob(Property)
	 */
	public void operationJob(Property property) throws FinderException, CreateException, NamingException {
	    m_log.debug("operationJob()");
	    
	    try {
			JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
			JobRunManagement.operationJob(property);
		} catch (CreateException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("operationJob() -> " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ジョブ履歴一覧情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(int)
	 */
	public ViewListInfo getHistoryList(int histories) throws FinderException, NamingException, SQLException {
	    m_log.debug("getHistoryList()");
	    
	    ViewListInfo list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getHistoryList(histories);
		} catch (FinderException e) {
			m_log.debug("getHistoryList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (SQLException e) {
			//JobSessionDAOImplにてログ出力するため、そのままthrowする
			throw e;
		}
	    
	    return list;
	}
	
	/**
	 * ジョブ履歴一覧情報を返します。
	 * 
	 * @ejb.interface-method
 	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param property 履歴フィルタ用プロパティ
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(Property, int)
	 */
	public ViewListInfo getHistoryList(Property property, int histories) throws FinderException, NamingException, SQLException {
	    m_log.debug("getHistoryList()");
	    
	    ViewListInfo list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getHistoryList(property, histories);
		} catch (FinderException e) {
			m_log.debug("getHistoryList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (SQLException e) {
			throw e;
		}
	    
	    return list;
	}
	
	/**
	 * ジョブ詳細一覧情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @return ジョブ詳細一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getDetailList(String)
	 */
	public CommonTableTreeItem getDetailList(String sessionId) throws FinderException, NamingException {
	    m_log.debug("getDetailList() : sessionId=" + sessionId);
	    
	    CommonTableTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getDetailList(sessionId);
		} catch (FinderException e) {
			m_log.debug("getDetailList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	    
	    return item;
	}
	
	/**
	 * ノード詳細一覧情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getNodeDetailList(String, String, Locale)
	 */
	public ArrayList getNodeDetailList(String sessionId, String jobId, Locale locale) throws FinderException, NamingException, CreateException {
	    m_log.debug("getNodeDetailList() : sessionId=" + sessionId + ", jobId=" + jobId);
	    
	    ArrayList list;
		try {
			SelectJob select = new SelectJob();
			list = select.getNodeDetailList(sessionId, jobId, locale);
		} catch (FinderException e) {
			m_log.debug("getNodeDetailList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (CreateException e) {
			throw e;
		}

	    return list;
	}
	
	/**
	 * ファイル転送一覧情報を返します。
	 * 
	 * @ejb.interface-method
 	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileList(String, String)
	 */
	public ArrayList getForwardFileList(String sessionId, String jobId) throws FinderException, NamingException, CreateException {
	    m_log.debug("getNodeDetailList() : sessionId=" + sessionId + ", jobId=" + jobId);
	    
	    ArrayList list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getForwardFileList(sessionId, jobId);
		} catch (FinderException e) {
			m_log.debug("getForwardFileList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (CreateException e) {
			throw e;
		}

	    return list;
	}
	
	/**
	 * スケジュール情報を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param info スケジュール情報
	 * @throws ParseException
	 * @throws NamingException
	 * @throws SchedulerException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#addSchedule(ScheduleInfo, String)
	 */
	public void addSchedule(ScheduleInfo info) throws ParseException, NamingException, SchedulerException {
	    m_log.debug("addSchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.addSchedule(info, pri.getName());
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * スケジュール情報を変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param info スケジュール情報
	 * @throws NamingException
	 * @throws SchedulerException
	 * @throws ParseException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#modifySchedule(ScheduleInfo, String)
	 */
	public void modifySchedule(ScheduleInfo info) throws NamingException, SchedulerException, ParseException {
	    m_log.debug("modifySchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.modifySchedule(info, pri.getName());
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * スケジュール情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param scheduleId スケジュールID
	 * @throws NamingException
	 * @throws SchedulerException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
	 */
	public void deleteSchedule(String scheduleId) throws NamingException, SchedulerException {
		m_log.debug("deleteSchedule() : scheduleId=" + scheduleId);
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.deleteSchedule(scheduleId);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.debug("deleteSchedule() : " + e.getMessage());
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.debug("deleteSchedule() : " + e.getMessage());
            throw e;
        }
	}
	
	/**
	 * スケジュール一覧情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return スケジュール一覧情報（Objectの2次元配列）
	 * @throws NamingException
	 * @throws SchedulerException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectSchedule#getScheduleList()
	 */
	public ArrayList getScheduleList() throws NamingException, SchedulerException {
	    m_log.debug("getScheduleList()");
	    
	    SelectSchedule select = new SelectSchedule();
	    //FIXME FinderException以外はそのままthrow
	    ArrayList list = select.getScheduleList();

	    return list;
	}
	
	/**
	 * 実行予定一覧情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 実行予定一覧情報（Objectの2次元配列）
	 * @throws NamingException
	 * @throws SchedulerException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getList()
	 */
	public ArrayList getRunScheduleList() throws NamingException, SchedulerException {
	    m_log.debug("getRunScheduleList()");
	    
	    SelectRunSchedule select = new SelectRunSchedule();
	    //FIXME FinderException以外はそのままthrow
	    ArrayList list = select.getList();

	    return list;
	}
	
	/**
	 * 実行予定情報を返します。
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param base 基準日
	 * @return 実行予定情報（Objectの2次元配列）
	 * @throws NamingException
	 * @throws SchedulerException
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getSchedule(Date)
	 */
	public ArrayList getRunSchedule(Date base) throws NamingException, SchedulerException, FinderException {
		m_log.debug("getRunSchedule() : base=" + base);
		
		ArrayList list = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			list = select.getSchedule(base);
		} catch (NamingException e) {
			throw e;
		} catch (SchedulerException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("getRunSchedule() -> " + e.getMessage());
			throw e;
		}

		return list;
	}
	
	/**
	 * 実行予定詳細一覧情報を返します。
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param scheduleId スケジュールID
	 * @return 実行予定詳細一覧情報（Objectの2次元配列）
	 * @throws NamingException
	 * @throws FinderException
	 * @throws SchedulerException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getDetailList(String)
	 */
	public CommonTableTreeItem getRunScheduleDetailList(String scheduleId) throws NamingException, FinderException, SchedulerException {
		m_log.debug("getRunScheduleDetailList() : scheduleId=" + scheduleId);
		
		CommonTableTreeItem item = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			item = select.getDetailList(scheduleId);
		} catch (FinderException e) {
			m_log.debug("getRunScheduleDetailList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (SchedulerException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * 実行予定詳細情報を返します。
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param scheduleId スケジュールID
	 * @param base 基準日
	 * @return 実行予定詳細情報（Objectの2次元配列）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SchedulerException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getDetailSchedule(String, Date)
	 */
	public CommonTableTreeItem getRunScheduleDetail(String scheduleId, Date base) throws FinderException, NamingException, SchedulerException {
		m_log.debug("getRunScheduleDetail() : scheduleId=" + scheduleId + ", base=" + base);
		
		CommonTableTreeItem item = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			item = select.getDetailSchedule(scheduleId, base);
		} catch (FinderException e) {
			m_log.debug("getRunScheduleDetail() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (SchedulerException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * カレンダIDリストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダIDのリスト
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#getCalendarIdList()
	 */
	public ArrayList getCalendarIdList() throws CreateException, NamingException, FinderException {
		m_log.debug("getCalendarIdList()");
		
		ArrayList list = null;
		try {
			CalendarControllerLocal calendar = 
				CalendarControllerUtil.getLocalHome().create();
			
			list = calendar.getCalendarIdList();
		} catch (CreateException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("getCalendarIdList() -> " + e.getMessage());
			throw e;
		}
		
	    return list;
	}
	
	/**
	 * セッションジョブ情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}
	 * @throws NamingException
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getSessionJobInfo(String, String)
	 */
	public JobTreeItem getSessionJobInfo(String sessionId, String jobId) throws NamingException, FinderException {
	    m_log.debug("getSessionJobInfo() : sessionId=" + sessionId + ", jobId=" + jobId);
	    
	    JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getSessionJobInfo(sessionId, jobId);
		} catch (FinderException e) {
			m_log.debug("getSessionJobInfo() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
	    return item;
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="JobManagementRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			Collection ct = null;
			
			// ファシリティIDが使用されているジョブコマンドマスタを取得する。
			ct = JobCommandMasterUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
			}
			
			// ファシリティIDが使用されているジョブコマンド情報を取得する。
			ct = null;
			ct = JobCommandInfoUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					//ジョブコマンド情報を取得
					JobCommandInfoLocal command = (JobCommandInfoLocal)itr.next();
					JobInfoLocal job = command.getJobInfo();
					JobSessionJobLocal sessionJob = job.getJobSessionJob();
					//セッションジョブの実行状態が終了または変更済以外の場合、UsedFacilityExceptionをスロー
					if(sessionJob.getStatus().intValue() != StatusConstant.TYPE_END && 
							sessionJob.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
					}
				}
			}
			
			// ファシリティIDが使用されているセッションノードを取得する。
			ct = null;
			ct = JobSessionNodeUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					//セッションノードを取得
					JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
					//セッションノードの実行状態が終了または変更済以外の場合、UsedFacilityExceptionをスロー
					if(sessionNode.getStatus().intValue() != StatusConstant.TYPE_END && 
							sessionNode.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
					}
				}
			}
			
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
}
