/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.MonitorConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.snmp.factory.AddMonitorSnmp;
import com.clustercontrol.snmp.factory.AddMonitorSnmpString;
import com.clustercontrol.snmp.factory.DeleteMonitorSnmp;
import com.clustercontrol.snmp.factory.DeleteMonitorSnmpString;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmp;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmpString;
import com.clustercontrol.snmp.factory.SelectMonitorSnmp;
import com.clustercontrol.snmp.factory.SelectMonitorSnmpString;

/**
 * @ejb.bean name="MonitorSnmpController"
 *           jndi-name="MonitorSnmpController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSnmpControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorSnmpControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;

	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SNMP監視情報を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public boolean addSnmp(MonitorInfo info) throws CreateException, FinderException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMP監視情報を登録
		AddMonitor snmp = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new AddMonitorSnmp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new AddMonitorSnmpString();
		}
		else{
			return false;
		}
		return  snmp.add(info, pri.getName());
	}
	
	/**
	 * SNMP監視情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public boolean modifySnmp(MonitorInfo info) throws CreateException, FinderException, RemoveException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMP監視情報を更新
		ModifyMonitor snmp = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new ModifyMonitorSnmp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new ModifyMonitorSnmpString();
		}
		else{
			return false;
		}
		return  snmp.modify(info, pri.getName());
	}

	/**
	 * SNMP監視情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @return 削除に成功した場合、true
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public boolean deleteSnmp(String monitorId, int monitorType) throws FinderException, RemoveException, SchedulerException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException{

		// SNMP監視情報を削除
		DeleteMonitor snmp = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new DeleteMonitorSnmp();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new DeleteMonitorSnmpString();
		}
		else{
			return false;
		}
		return  snmp.delete(MonitorConstant.STRING_SNMP, monitorId);
	}
	
	/**
	 * SNMP監視情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 状態監視ID
	 * @param monitorType 監視判定タイプ
	 * @return SNMP監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public MonitorInfo getSnmp(String monitorId, int monitorType) throws CreateException, FinderException, SchedulerException, NamingException {

		// SNMP監視情報を取得
		SelectMonitor snmp = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new SelectMonitorSnmp();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new SelectMonitorSnmpString();
		}
		else{
			return null;
		}
		
		MonitorInfo info = snmp.getMonitor(MonitorConstant.STRING_SNMP, monitorId);
		return info;
	}

	/**
	 * SNMP監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public ArrayList getSnmpList() throws CreateException, FinderException, SchedulerException, NamingException {

		// SNMP監視一覧を取得
		SelectMonitorSnmp snmp = new SelectMonitorSnmp();
		ArrayList list = snmp.getMonitorList(MonitorConstant.STRING_SNMP);

		return list;
	}
	
	/**
	 * SNMP監視一覧のテーブル定義情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getSnmpListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorSnmp snmp = new SelectMonitorSnmp();
		ArrayList tableDefine = snmp.getMonitorListTableDefine(locale);

		return tableDefine;
	}
	
	/**
	 * SNMP監視 文字列監視判定一覧のテーブル定義情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 * 
	 * @version 2.1.0
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getSnmpStringValueListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorSnmpString snmp = new SelectMonitorSnmpString();
		ArrayList tableDefine = snmp.getStringValueListTableDefine(locale);
		
		return tableDefine;
	}
}
