/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.composite;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.util.Messages;

/**
 * 通知基本情報コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「通知ID」 テキストボックス</dd>
 *  <dd>「説明」 テキストボックス</dd>
 * </dl>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */

public class NotifyBasicComposite extends Composite {

	/** カラム数（タイトル）。*/
	public static final int WIDTH_TITLE = 2;

	/** カラム数（値）。*/
	public static final int WIDTH_VALUE = 8;
	
	/** 空白のカラム。*/
	public static final int WIDTH_BLANK = 5;
	
	/** カラム数（全て）。*/
	public static final int WIDTH_ALL = 15;
	
	/** 通知ID テキストボックス。 */
	private Text m_textNotifyId = null;

	/** 説明 テキストボックス。 */
	private Text m_textDescription = null;
	
	/** 変更対象通知ID。 */
	private String notifyId = null;

	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NotifyBasicComposite(Composite parent, int style, String notifyId) {
		super(parent, style);

		this.notifyId = notifyId;
		this.initialize();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);

		/*
		 * 通知ID
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.id") + " : ");
		// テキスト
		this.m_textNotifyId = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textNotifyId.setLayoutData(gridData);
		this.m_textNotifyId.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));

		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 説明
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.setLayoutData(gridData);
		this.m_textDescription.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// 空行
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_ALL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
	}

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知情報
	 */
	public void setInputData(NotifyInfo info) {
		
		if(info != null){
			if (info.getNotifyId() != null) {
				this.m_textNotifyId.setText(info.getNotifyId());
				this.m_textNotifyId.setEnabled(false);
			}
			if (info.getDescription() != null) {
				this.m_textDescription.setText(info.getDescription());
			}
		}
	}
	
	/**
	 * 引数で指定された通知情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する通知情報
	 * @return 検証結果
	 * 
	 * @see #setValidateResult(String, String)
	 */
	public ValidateResult createInputData(NotifyInfo info) {
		
		if(info != null){
			if (this.m_textNotifyId.getText() != null
					&& !"".equals((this.m_textNotifyId.getText()).trim())) {
				info.setNotifyId(this.m_textNotifyId.getText());
			} else {
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.10"));
			}
			if (this.m_textDescription.getText() != null
					&& !"".equals((this.m_textDescription.getText()).trim())) {
				info.setDescription(this.m_textDescription.getText());
			}
		}
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
        if(this.notifyId != null && !"".equals(this.notifyId.trim())){
        	this.m_textNotifyId.setEnabled(false);
        }
        else {
    		this.m_textNotifyId.setEnabled(enabled);
        }
		this.m_textDescription.setEnabled(enabled);
	}
	
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
}