/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;
import java.util.StringTokenizer;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.mail.composite.MailTemplateIdListComposite;
import com.clustercontrol.util.Messages;

/**
 * 通知（メール）作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class NotifyMailCreateDialog extends NotifyBasicCreateDialog {
	
	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 		= 1;
	
	/** カラム数（チェックボックス）。 */
	private static final int WIDTH_CHECK 			= 1;
	
	/** カラム数（メッセールアドレス）。 */
	private static final int WIDTH_MAIL_ADDRESS		= 6;
	
	/** カラム数（抑制）。 */
	private static final int WIDTH_INHIBITION_FLG	= 2;
	
	/** カラム数（空欄）。 */
	private static final int WIDTH_BLANK 			= 5;
	
    // ----- instance フィールド ----- //
	
	/** 通知タイプ 
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final Integer m_notifyType = NotifyTypeConstant.TYPE_MAIL;
	
	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;
	
	/** メールテンプレートID情報コンポジット */
	private MailTemplateIdListComposite m_compositeMailTemplateIdList = null;
	
	/** メール（重要度：通知） チェックボックス。 */ 
	private Button m_checkMailNormalInfo = null;
	/** メール（重要度：警告） チェックボックス。 */
	private Button m_checkMailNormalWarning = null;
	/** メール（重要度：危険） チェックボックス。 */
	private Button m_checkMailNormalCritical = null;
	/** メール（重要度：不明） チェックボックス。 */
	private Button m_checkMailNormalUnknown = null;
	
	/** メールアドレス（重要度：通知） テキスト。 */ 
	private Text m_textMailAddressInfo = null;
	/** メールアドレス（重要度：警告） テキスト。 */
	private Text m_textMailAddressWarning = null;
	/** メールアドレス（重要度：危険） テキスト。 */
	private Text m_textMailAddressCritical = null;
	/** メールアドレス（重要度：不明） テキスト。 */
	private Text m_textMailAddressUnknown = null;
		
	/** 抑制（通知） チェックボックス。 */ 
	private Button m_checkMailInhibitionInfo = null;
	/** 抑制（警告） チェックボックス。 */
	private Button m_checkMailInhibitionWarning = null;
	/** 抑制（異常） チェックボックス。 */
	private Button m_checkMailInhibitionCritical = null;
	/** 抑制（不明） チェックボックス。 */
	private Button m_checkMailInhibitionUnknown = null;
	
	// ----- コンストラクタ ----- //
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyMailCreateDialog(Shell parent) {
		super(parent);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyMailCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}
	
    // ----- instance メソッド ----- //
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
    protected void customizeDialog(Composite parent) {
    	
    	super.customizeDialog(parent);
    	
        // 通知IDが指定されている場合、その情報を初期表示する。
        NotifyInfo info = null; 
        if(this.notifyId != null){
        	info = new GetNotify().getNotify(this.notifyId);        	
        }
        else{
        	info = new NotifyInfo();
        }
        this.setInputData(info);
        
    }
    
	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
    protected void customizeSettingDialog(Composite parent) {
    	final Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.notify.mail.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);     
        
        /*
         * メール
         */
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notifies.mail")+ " : ");
		
        /*
         * 大グループ
         */
        // メールグループ
        Group groupMail = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupMail.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupMail.setLayoutData(gridData);        
        
        /*
         * メールテンプレートIDの設定
         */
        this.m_compositeMailTemplateIdList = new MailTemplateIdListComposite(groupMail, SWT.NONE, true);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.BEGINNING;
        gridData.grabExcessHorizontalSpace = true;
        m_compositeMailTemplateIdList.setLayoutData(gridData);        
        
        // 空行
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);  
        
        /*
         * 重要度 ごとの設定
         */
        // ラベル（重要度）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_PRIORITY;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority"));
        
        // ラベル（メール）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.event.treatment.mail"));
        
        // ラベル（メールアドレス）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_MAIL_ADDRESS;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("email.address.ssv"));
        
        // ラベル（抑制）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.event.treatment.inhibition"));
        
        // ラベル（空欄）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        //　メール　重要度：通知
        label = this.getLabelPriority(groupMail, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
        this.m_checkMailNormalInfo = this.getCheckMailNormal(groupMail);
        this.m_textMailAddressInfo = this.getTextMailAddress(groupMail);
        this.m_checkMailInhibitionInfo = this.getCheckMailInhibition(groupMail);
        
        this.m_checkMailNormalInfo.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		setEnabledForMail(m_checkMailNormalInfo.getSelection(), 
        				m_textMailAddressInfo, 
        				m_checkMailInhibitionInfo);
        	}
        });
        // ラベル（空欄）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        
        //　メール　重要度：警告
        label = this.getLabelPriority(groupMail, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
        this.m_checkMailNormalWarning = this.getCheckMailNormal(groupMail);
        this.m_textMailAddressWarning = this.getTextMailAddress(groupMail);
        this.m_checkMailInhibitionWarning = this.getCheckMailInhibition(groupMail);
        
        this.m_checkMailNormalWarning.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		setEnabledForMail(m_checkMailNormalWarning.getSelection(), 
        				m_textMailAddressWarning, 
        				m_checkMailInhibitionWarning);
        	}
        });
        // ラベル（空欄）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        //　メール　重要度：危険
        label = this.getLabelPriority(groupMail, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
        this.m_checkMailNormalCritical = this.getCheckMailNormal(groupMail);
        this.m_textMailAddressCritical = this.getTextMailAddress(groupMail);
        this.m_checkMailInhibitionCritical = this.getCheckMailInhibition(groupMail);
        
        this.m_checkMailNormalCritical.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		setEnabledForMail(m_checkMailNormalCritical.getSelection(), 
        				m_textMailAddressCritical, 
        				m_checkMailInhibitionCritical);
        	}
        });
        // ラベル（空欄）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        //　メール　重要度：不明
        label = this.getLabelPriority(groupMail, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
        this.m_checkMailNormalUnknown = this.getCheckMailNormal(groupMail);
        this.m_textMailAddressUnknown = this.getTextMailAddress(groupMail);
        this.m_checkMailInhibitionUnknown = this.getCheckMailInhibition(groupMail);
        
        this.m_checkMailNormalUnknown.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		setEnabledForMail(m_checkMailNormalUnknown.getSelection(), 
        				m_textMailAddressUnknown, 
        				m_checkMailInhibitionUnknown);
        	}
        });
        // ラベル（空欄）
        label = new Label(groupMail, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_BLANK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

    }

	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
    public NotifyInfo getInputData() {
        return this.inputData;
    }

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
    protected void setInputData(NotifyInfo notify) {

        this.inputData = notify;
        
        // 通知基本情報
        super.m_notifyBasic.setInputData(notify);
        
        // 抑制情報
        super.m_notifyInhibition.setInputData(notify);
      
        // メール情報
        if (notify.getNotifyType() != null && notify.getNotifyType().intValue() == this.m_notifyType.intValue() && notify.getNotifyInfoDetail() != null) {
            ArrayList<NotifyInfoDetail> list = notify.getNotifyInfoDetail();
            if(list != null){
                for(int index=0; index<list.size(); index++){
                	NotifyMailInfo mail = (NotifyMailInfo)list.get(index);
                	if(mail != null){
                		// 通知の場合
                		if(PriorityConstant.TYPE_INFO ==  mail.getPriority()){
                			this.setInputDataForMail(mail, this.m_checkMailNormalInfo, this.m_textMailAddressInfo, this.m_checkMailInhibitionInfo);
                            //通知のみメール通知の共通部分の設定行う
                            if (mail.getMailTemplateId() != null) {
                            	this.m_compositeMailTemplateIdList.setText(mail.getMailTemplateId());
                            }
                		}
                		// 警告の場合
                		else if(PriorityConstant.TYPE_WARNING ==  mail.getPriority()){
                			this.setInputDataForMail(mail, this.m_checkMailNormalWarning, this.m_textMailAddressWarning, this.m_checkMailInhibitionWarning);
                		}
                		// 危険の場合
                		else if(PriorityConstant.TYPE_CRITICAL ==  mail.getPriority()){
                			this.setInputDataForMail(mail, this.m_checkMailNormalCritical, this.m_textMailAddressCritical, this.m_checkMailInhibitionCritical);
                		}
                		// 不明の場合
                		else if(PriorityConstant.TYPE_UNKNOWN ==  mail.getPriority()){
                			this.setInputDataForMail(mail, this.m_checkMailNormalUnknown, this.m_textMailAddressUnknown, this.m_checkMailInhibitionUnknown);
                		}
                	}
                }
            }
        }
        
        // 有効／無効
        if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
            this.m_confirmValid.setSelection(false);
        }
        else {
        	this.m_confirmValid.setSelection(true);
        }
        
        // 入力制御
        if(this.referenceFlg){
        	this.setEnabled(false);
        }
        else{
        	this.setEnabled(true);
        }
        
    }
                
	/**
	 * 引数で指定された通知ステータス情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知メール情報
	 * @param checkMailNormal 通知チェックボックス
	 * @param textMailAddress メールアドレステキストボックス
	 * @param checkMailInhibition 抑制チェックボックス
	 */
    private void setInputDataForMail(NotifyMailInfo info, 
    		Button checkMailNormal,
    		Text textMailAddress,
    		Button checkMailInhibition
    		) {
    	// メール通知
        if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkMailNormal.setSelection(true);
        }
        // メールアドレス
        if (info.getMailAddress() != null) {
        	textMailAddress.setText(info.getMailAddress());
        }
        // 抑制
        if (info.getInhibitionFlg() != null && info.getInhibitionFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkMailInhibition.setSelection(true);
        }
    }
    

	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForMail(ArrayList, int, Button, Text, Button)
	 * @see #setValidateResult(String, String)
	 */
    protected NotifyInfo createInputData() {
        NotifyInfo info = new NotifyInfo();

        // 通知タイプの設定
    	info.setNotifyType(m_notifyType);
        
        // 通知基本情報
        validateResult = super.m_notifyBasic.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // 抑制情報
        validateResult = super.m_notifyInhibition.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // メール情報
        // 各重要度で１つも通知が選択されていない場合はエラー
    	if(!this.m_checkMailNormalInfo.getSelection() && 
    			!this.m_checkMailNormalWarning.getSelection() && 
    			!this.m_checkMailNormalCritical.getSelection() && 
    			!this.m_checkMailNormalUnknown.getSelection()){
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.13"));
                return null;
    	}      

        ArrayList<NotifyInfoDetail> list = new ArrayList<NotifyInfoDetail>();

        list = this.createInputDataForMail(list, PriorityConstant.TYPE_INFO,	this.m_checkMailNormalInfo,		this.m_textMailAddressInfo,		this.m_checkMailInhibitionInfo);
        list = this.createInputDataForMail(list, PriorityConstant.TYPE_WARNING,	this.m_checkMailNormalWarning,	this.m_textMailAddressWarning,	this.m_checkMailInhibitionWarning);
        list = this.createInputDataForMail(list, PriorityConstant.TYPE_CRITICAL,this.m_checkMailNormalCritical,	this.m_textMailAddressCritical,	this.m_checkMailInhibitionCritical);
        list = this.createInputDataForMail(list, PriorityConstant.TYPE_UNKNOWN,	this.m_checkMailNormalUnknown, 	this.m_textMailAddressUnknown,	this.m_checkMailInhibitionUnknown);
        
        if(list == null){
        	return null;
        }
        info.setNotifyInfoDetail(list);
        
        // 有効/無効
        if (this.m_confirmValid.getSelection()) {
        	info.setValidFlg(ValidConstant.TYPE_VALID);
        } else {
        	info.setValidFlg(ValidConstant.TYPE_INVALID);
        }
        
        return info;
    }

	/**
	 * 入力値を設定した通知イベント情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知メール情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected ArrayList<NotifyInfoDetail> createInputDataForMail(ArrayList<NotifyInfoDetail> list,
    		int priority,
    		Button checkMailNormal, 
    		Text textMailAddress,
    		Button checkMailInhibition
    		) {
    
    	if(list == null)
    		return null;
    	
    	NotifyMailInfo mail = new NotifyMailInfo();
    	
    	// 重要度
    	mail.setPriority(priority);
    	// メール通知
        if (checkMailNormal.getSelection()) {
        	mail.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	mail.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
        }
        // メールアドレス
        if (textMailAddress.getText() != null
                && !"".equals((textMailAddress.getText()).trim())) {
    		//セミコロンで分ける
    		StringTokenizer t = new StringTokenizer(textMailAddress.getText(), ";");
    		while (t.hasMoreTokens()) {
    			String mailAddress = t.nextToken();    			
    			//@マークで分ける
    			String attr[] = mailAddress.split("@");
    			if (attr.length != 2) {
            		this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.18"));
                    return null;
                }
    		}
        	mail.setMailAddress(textMailAddress.getText());
        } else {
        	if(checkMailNormal.getSelection()){
                this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.17"));
                return null;
        	}
        }
        
        // 抑制
        if (checkMailInhibition.getSelection()) {
        	// 抑制がチェックされているが、抑制条件が指定されていない場合
        	if(checkMailNormal.getSelection() && this.m_notifyInhibition.m_radioNoInhibition.getSelection()){
                this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.16"));
                return null;
        	}
        	mail.setInhibitionFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	mail.setInhibitionFlg(new Integer(ValidConstant.TYPE_INVALID));
        }
        
        // メールテンプレートＩＤ（重要度に関係なく全て同じものを設定する）
        String mailTemplateId = this.m_compositeMailTemplateIdList.getText();
        // メールテンプレートID
        if (mailTemplateId != null && !"".equals(mailTemplateId.trim())) {
        	mail.setMailTemplateId(mailTemplateId);
        }  
        
        list.add(mail);
    	return list;
    }
    
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
    protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	NotifyInfo info = this.getInputData();
    	if(info != null){
            if(this.notifyId == null){
            	// 作成の場合
            	result = new AddNotify().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyNotify().modify(info);
            }
    	}
    	
    	return result;
    }

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
    
	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
    protected void setEnabled(boolean enable) {

    	super.m_notifyBasic.setEnabled(enable);
    	super.m_notifyInhibition.setEnabled(enable);
    	
    	this.setEnabledForMail(enable);
    }    
    
	/**
	 * メール通知の通知関連コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 * 
	 * @see #setEnabledForMails(boolean, Button, Text, Button)
	 */
    private void setEnabledForMail(boolean enable) {

    	// 通知関連
        this.setEnabledForMails(enable, this.m_checkMailNormalInfo, 	this.m_textMailAddressInfo, 	this.m_checkMailInhibitionInfo);
        this.setEnabledForMails(enable, this.m_checkMailNormalWarning,	this.m_textMailAddressWarning,	this.m_checkMailInhibitionWarning);
        this.setEnabledForMails(enable, this.m_checkMailNormalCritical, this.m_textMailAddressCritical,	this.m_checkMailInhibitionCritical);
        this.setEnabledForMails(enable, this.m_checkMailNormalUnknown, 	this.m_textMailAddressUnknown,	this.m_checkMailInhibitionUnknown);
    }
    
    /**
	 * メール通知のコンポジットの選択可/不可を設定します。
	 *
	 * @param enable 選択可の場合、<code> true </code>
	 * @param checkMailNormal メール通知チェックボックス
	 * @param textMailAddress メールアドレステキストボックス
	 * @param checkMailInhibition 抑制チェックボックス
	 */
    private void setEnabledForMails(boolean enable, 
    		Button checkMailNormal,
    		Text textMailAddress,
    		Button checkMailInhibition
    		) {

        if (enable) {
        	checkMailNormal.setEnabled(true);
        	
        	textMailAddress.setEnabled(checkMailNormal.getSelection());
        	checkMailInhibition.setEnabled(checkMailNormal.getSelection());
        	
        }
        else{
        	checkMailNormal.setEnabled(false);
        	
        	textMailAddress.setEnabled(false);
        	checkMailInhibition.setEnabled(false);
        	
        }
    }
    
    /**
	 * 引数で指定されたコンポジットの選択可/不可を設定します。
	 * 
	 * @param enabled 選択可の場合、</code> true </code>
	 * @param textMailAddress 設定対象のメールアドレステキストボックス
	 * @param checkInhibitionFlg 設定対象の抑制チェックボックス
	 */
	public void setEnabledForMail(boolean enabled,
			Text textMailAddress,
			Button checkInhibitionFlg
	) {
		
		textMailAddress.setEnabled(enabled);
		checkInhibitionFlg.setEnabled(enabled);
	}   
     
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    protected void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
    
	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        if(!this.referenceFlg){
        	super.createButtonsForButtonBar(parent);
        }
        else{
        	// 閉じるボタン
            this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
        }
    }
    
	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
    protected Label getLabelPriority(Composite parent,
    		String text,
    		Color background
    		) {
    	
    	// ラベル（重要度）
    	Label label = new Label(parent, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.horizontalSpan = WIDTH_PRIORITY;
    	gridData.horizontalAlignment = GridData.FILL;
    	gridData.grabExcessHorizontalSpace = true;
    	label.setLayoutData(gridData);
    	label.setText(text + " : ");
    	label.setBackground(background);
        
        return label;
    }
    
	/**
	 * メールの実行チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckMailNormal(Composite parent) {
    	
        // チェックボックス（実行）
    	Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
	/**
	 * メールのメールアドレステキストボックスを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @return 生成されたテキストボックス
	 */
    protected Text getTextMailAddress(Composite parent) {
    	
        // テキスト（送信先（セミコロン区切り））
    	Text text = new Text(parent, SWT.BORDER);
    	GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_MAIL_ADDRESS;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        text.setLayoutData(gridData);
        text.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_1024));
        
        return text;
    }
    
	/**
	 * メールの抑制のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckMailInhibition(Composite parent) {
    	
        // チェックボックス（抑制する）
        Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
}