/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.commons.util;

import java.io.Serializable;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.bean.ConnectionFactoryConstant;

/**
 * キューへメッセージを送信するクラスです。
 *
 * @version 2.1.0
 * @since 2.0.0
 */
public class SendQueue {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SendQueue.class );
	
	/** キュー名。 */
    protected String m_queueName;
    /** コネクションファクトリー。 */
    protected QueueConnectionFactory m_connectionFactory;
    /** コネクション。 */
    protected QueueConnection m_connection;
    /** キュー。 */
    protected Queue m_queue;
    /** セッション。 */
    protected QueueSession m_session;
    
    /**
     * コンストラクタ。<BR>
     * キュー名を設定します。初期処理を呼び出します。
     * 
     * @param queueName キュー名
     * @throws NamingException
     * @throws JMSException
     * 
     * @see #initial()
     */
    public SendQueue(String queueName) throws NamingException, JMSException{
        m_queueName = queueName;
        this.initial();
    }
    
    /**
     * 引数で指定されたメッセージをキューへ送信します。
     * 
     * @param orgMessage メッセージ
     * @throws JMSException
     */
    public void put(Serializable orgMessage) throws JMSException{
    	
    	QueueSender sender = null;
		try {
	    	// キューセンダーを作成
	        sender = m_session.createSender(m_queue);
	        
	        // 送信メッセージ作成
	        ObjectMessage message = m_session.createObjectMessage(orgMessage);
	        
	        // メッセージ送信
	        sender.send(message);
		}
		catch (JMSException e) {
    		m_log.error("put:" + e.getMessage());
			
			this.terminate();
			throw e;
		}
		finally{
			try {
				if(sender != null)
					sender.close();
			}
			catch (JMSException e) {
			}
		}
    }
    
    /**
     * メッセージ送信の初期処理を行います。
	 * <p>
	 * <ol>
	 * <li>コネクションファクトリと宛先を取得します。</li>
	 * <li>コネクションファクトリより、JMSプロバイダとのコネクションを生成します。</li>
	 * <li>コネクションよりセッションを生成します。</li>
	 * </ol>
     * 
     * @throws JMSException
     * @throws NamingException
     */
    public void initial() throws NamingException, JMSException {
    	
    	try{
    		InitialContext context = null;
    		String serverName = System.getProperty("jboss.server.name");
    		if(serverName.equals("all")){
    			Properties properties = new Properties(); 
    			properties.put(InitialContext.INITIAL_CONTEXT_FACTORY,"org.jnp.interfaces.NamingContextFactory");
    			properties.put(InitialContext.URL_PKG_PREFIXES,"org.jboss.naming:org.jnp.interfaces");
    			properties.put(InitialContext.PROVIDER_URL,"jnp://localhost:1100");
    			
    			context = new InitialContext(properties);
    		}
    		else{
    			context = new InitialContext();
    		}
//            m_connectionFactory = (QueueConnectionFactory)context.lookup("ConnectionFactory");
    		m_connectionFactory = (QueueConnectionFactory)context.lookup(ConnectionFactoryConstant.JNDI_JVMIL);
            m_connection = m_connectionFactory.createQueueConnection();
            m_session = m_connection.createQueueSession( false, Session.AUTO_ACKNOWLEDGE );
            m_connection.start();
            m_queue = (Queue)context.lookup(m_queueName);
            
            context.close();
    	}
    	catch(NamingException e){
    		m_log.error("initial:" + e.getMessage());
			
			this.terminate();
			throw e;
    	}
    	catch(JMSException e){
    		m_log.error("initial:" + e.getMessage());
			
			this.terminate();
			throw e;
    	}
    }
    
    /**
     * メッセージ送信の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>セッションをクローズします。</li>
	 * <li>コネクションをクローズします。</li>
	 * </ol>
	 * 
     * @throws JMSException
     */
    public void terminate() throws JMSException{
    	
    	//connectionの作法として、stopしてからcloseする
    	//コネクション停止、セッションクローズに失敗した場合でも、
    	//コネクションをクローズすることはできるので、Exceptionはスローしない。
    	
		try {
			if (m_connection != null){
				m_connection.stop();
			}
		} catch (JMSException e) {
			m_log.error("terminate : " + e.getMessage(), e);
		}
    	
		try {
			if (m_session != null){
				m_session.close();
				m_session = null;
			}
		} catch (JMSException e) {
			m_log.error("terminate : " + e.getMessage(), e);
		}

		try {
			if (m_connection != null) {
				m_connection.close();
				m_connection = null;
			}
		} catch (JMSException e) {
			m_log.error("terminate : " + e.getMessage(), e);
			throw e;
		}
    }
}

