/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.text.DateFormat;
import java.util.Date;

import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.composite.CommonTableTreeViewer;
import com.clustercontrol.jobmanagement.action.GetRunScheduleDetail;
import com.clustercontrol.jobmanagement.action.GetRunScheduleDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleDetailTableDefine;
import com.clustercontrol.jobmanagement.composite.action.VerticalBarSelectionListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[実行予定詳細]ビューの右側用のコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RunScheduleDetailComposite extends Composite {
	/** テーブルビューア */
    protected CommonTableTreeViewer m_viewer = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** スケジュールID */
    protected String m_scheduleId = null;
    /** 基準日 */
    protected Date m_base = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param base 基準日
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public RunScheduleDetailComposite(Composite parent, int style, Date base) {
        super(parent, style);
        m_base = base;
        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        Label dummy = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        dummy.setLayoutData(gridData);
        dummy.setText("    ");
        
        //予定詳細テーブル作成
        TableTree tree = new TableTree(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);

        Table table = tree.getTable();
        table.setHeaderVisible(true);
        table.setLinesVisible(true);
        table.getHorizontalBar().setVisible(true);
        table.getHorizontalBar().setEnabled(true);
        table.getVerticalBar().setVisible(true);
        table.getVerticalBar().setEnabled(true);
        
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        tree.setLayoutData(gridData);
        
        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);
        
        m_viewer = new CommonTableTreeViewer(tree);
        m_viewer.createTableColumn(GetRunScheduleDetailTableDefine.get(m_base), 
        		RunScheduleDetailTableDefine.SORT_COLUMN_INDEX, 
        		RunScheduleDetailTableDefine.SORT_ORDER);
        
        m_viewer.getTableTree().getTable().getVerticalBar().addSelectionListener(
        		new VerticalBarSelectionListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 引数にしてされたスケジュールIDに該当する基準日からの実行予定詳細マーク情報を取得し、テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数にしてされたスケジュールIDに該当する基準日からの実行予定詳細マーク情報を取得します。</li>
	 * <li>テーブルビューアーに実行予定詳細マーク情報をセットします。</li>
	 * </ol>
	 * 
     * @see com.clustercontrol.jobmanagement.action.GetRunScheduleDetail#get(String, Date)
     */
    public void update(String scheduleId) {
        //予定詳細情報取得
        GetRunScheduleDetail getList = new GetRunScheduleDetail();
        CommonTableTreeItem item = null;
        if (scheduleId != null && scheduleId.length() > 0) {
            item = getList.get(scheduleId, m_base);
        }

        m_viewer.setInput(item);
        m_viewer.expandAll();
        
        m_scheduleId = scheduleId;

        Object[] args = { DateFormat.getDateInstance().format(m_base) };
        m_labelCount.setText(Messages.getString("specified.period", args));
    }
    
    /**
     * このコンポジットが利用するテーブルツリービューアを返します。
     * 
     * @return テーブルツリービューア
     */
    public TableTreeViewer getTableTreeViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTableTree().getTable();
    }

    /**
     * このコンポジットが利用するテーブルツリーを返します。
     * 
     * @return テーブルツリー
     */
    public TableTree getTableTree() {
        return m_viewer.getTableTree();
    }

	/**
	 * スケジュールIDを返します。
	 * 
	 * @return スケジュールID
	 */
	public String getScheduleId() {
		return m_scheduleId;
	}
	
	/**
	 * スケジュールIDを設定します。
	 * 
	 * @param scheduleId スケジュールID
	 */
	public void setScheduleId(String scheduleId) {
		m_scheduleId = scheduleId;
	}
	
	/**
	 * 基準日を返します。
	 * 
	 * @return 基準日
	 */
	public Date getBaseDate() {
		return m_base;
	}
	
	/**
	 * 基準日を設定します。
	 * 
	 * @param base 基準日
	 */
	public void setCalendarId(Date base) {
		m_base = base;
	}
}