/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.monitor.factory;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType;
import com.clustercontrol.performance.util.code.CollectorItemCodeTable;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.monitor.ejb.entity.MonitorPerfInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.MonitorPerfInfoPK;
import com.clustercontrol.performance.monitor.ejb.entity.MonitorPerfInfoUtil;

/**
 * リソース監視の閾値判定クラス
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class RunMonitorPerformance2 extends RunMonitorNumericValueType {
	
	protected static Log m_log = LogFactory.getLog( RunMonitorPerformance2.class );
	
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	/** 閾値情報 */
	protected MonitorPerfInfoLocal m_perf = null;
	
	/** 収集項目名 */
	private String m_itemName = null;
	
	/** 数値フォーマット */
	protected NumberFormat m_numberFormat = NumberFormat.getNumberInstance();
	
	/**
	 * コンストラクタ
	 * 
	 */
	public RunMonitorPerformance2() throws NamingException, JMSException, CreateException{
		super();
	}
	
	/**
	 * マルチスレッドを実現するCallableTaskに渡すためのインスタンスを作成するメソッド
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#runMonitorInfo()
	 * @see com.clustercontrol.monitor.run.util.CallableTask
	 */
	public RunMonitor createMonitorInstance() throws NamingException, JMSException, CreateException{
		return new RunMonitorPerformance2();
	}
	
	/**
	 * リソース監視の閾値判定を実行します。
	 * (性能管理の閾値監視ではスコープの判定方法が異なるため新規に生成)
	 * 
	 * @param monitorTypeId 監視監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws CreateException 
	 */
	@SuppressWarnings("unchecked")
	public void run(String monitorTypeId, String monitorId, CollectedDataInfo data)
	throws FinderException, RemoveException, JMSException, NamingException, CreateException {
		
		//キューコネクションを初期化する
		super.initialize();
		
		String facilityId = data.getFacilityId();
		
		if(facilityId == null || "".equals(facilityId)){
			return;
		}
		
		// 指定のファシリティID配下（ノードの場合は自分）で管理対象有効なノードが存在しない場合
		if(m_repository.getExecTargetFacilityIdList(facilityId).size() == 0){
			return;
		}
		
		// 性能値を設定
		m_value = (float)data.getValue();
		
		m_now = new Date();
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		m_priorityMap = new HashMap();
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_INFO),		new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_WARNING),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_CRITICAL),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN),	new ArrayList<String>());
		
		try 
		{
			// 監視基本情報を設定
			boolean notify = this.setMonitorInfo(m_monitorTypeId, m_monitorId);
			if(!notify){
				// 通知しない場合は処理終了
				return;
			}
			
			// 判定情報を設定
			setJudgementInfo();
			
			// 監視チェック情報を設定
			setCheckInfo();

			boolean isNode = false;
			// 監視対象として指定したファシリティがノードの場合
			if(m_repository.isNode(this.m_facilityId)){
				isNode = true;
			}

			// 監視値が収集できたかどうかを判定
			boolean result = collect(facilityId);
			
			// 監視結果より重要度を取得
			int priority = getCheckResult(result);		
			
			// 判定対象ファシリティがノードの場合
			// (監視対象ファシリティと判定対象ファシリティとは別のものを指す
			//  監視対象ファシリティ：リソース監視の設定で指定したファシリティID
			//  判定対象ファシリティ：このメソッドを呼び出す際に渡された収集済みデータのファシリティID)
			if(m_repository.isNode(facilityId)){
				if((MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||
						(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && isNode) ||
						MonitorBlockConstant.TYPE_ALL == m_monitorBlock)){
					
					// 状態監視へ通知
					if(data.getDate().getTime() == 0){
						notify(true, facilityId, priority, new Date());
					} else {
						notify(true, facilityId, priority, data.getDate());
					}
				}
			}
			// 判定対象ファシリティがスコープの場合
			else {  
				if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock || 
						MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
					
					// 状態監視へ通知
					if(data.getDate().getTime() == 0){
						notify(false, facilityId, priority, new Date());
					} else {
						notify(false, facilityId, priority, data.getDate());
					}
				}
			}
		} catch (FinderException e) {
			m_log.error("run():" + e.getMessage());
			e.printStackTrace();
			throw e;
		} catch (JMSException e) {
			e.printStackTrace();
			throw e;
		} catch (NamingException e) {
			m_log.error("run():" + e.getMessage());
			e.printStackTrace();
			throw e;
		} catch (CreateException e) {
			m_log.error("run():" + e.getMessage());
			e.printStackTrace();
			throw e;
		} finally {
			// 終了処理
			this.terminate();
		}
	}
	
	/**
	 * リソース監視情報を設定します。
	 * @see com.clustercontrol.monitor.run.factory.OperationNumericValueInfo#setMonitorAdditionInfo()
	 */
	@Override
	public void setCheckInfo() throws FinderException, NamingException{
		// 性能管理閾値監視情報を取得
		MonitorPerfInfoPK pk = new MonitorPerfInfoPK(m_monitorId, m_monitorTypeId);
		m_perf = MonitorPerfInfoUtil.getLocalHome().findByPrimaryKey(pk);
		m_itemName = CollectorItemCodeTable.getFullItemName(m_perf.getItemCode(), m_perf.getDeviceName());
	}
	
	/**
	 * メッセージIDを取得します。
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageId(int)
	 */
	@Override
	public String getMessageId(int result) {
		
		if(result == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(result == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(result == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
	
	/**
	 * スコープ用メッセージIDを取得します。
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#getMessageIdForScope(int)
	 */
	@Override
	public String getMessageIdForScope(int priority) {
		
		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
	
	/**
	 * メッセージを取得します。
	 */
	@Override
	public String getMessage(int result) {
		String valueString;
		if(Double.isNaN(m_value)){
			valueString = "NaN";
		} else {
			valueString = m_numberFormat.format(m_value);
		}
		return m_itemName + " : " + valueString;
	}

	/**
	 * メッセージを取得します。
	 */
	@Override
	public String getMessageForScope(int result){
		String valueString;
		if(Double.isNaN(m_value)){
			valueString = "NaN";
		} else {
			valueString = m_numberFormat.format(m_value);
		}
		return m_itemName + " : " + valueString;
	}
	
	/**
	 * オリジナルメッセージを取得します。
	 */
	@Override
	public String getMessageOrg(int result) {
		String valueString;
		if(Double.isNaN(m_value)){
			valueString = "NaN";
		} else {
			valueString = m_numberFormat.format(m_value);
		}
		return m_itemName + " : " + valueString;
	}
	
	/**
	 * オリジナルメッセージを取得します。
	 */
	@Override
	public String getMessageOrgForScope(int result){
		String valueString;
		if(Double.isNaN(m_value)){
			valueString = "NaN";
		} else {
			valueString = m_numberFormat.format(m_value);
		}
		return m_itemName + " : " + valueString;
	}

	@Override
	public boolean collect(String facilityId) {
		// 性能値がNaNでなければ、値は取得できている
		return !Double.isNaN(m_value);
	}
}
