//
// This file is part of the joeSNMP Java Library.
//
// joeSNMP is Copyright (C) 2002-2003 Blast Internet Services, Inc.  All rights reserved.
// joeSNMP is a derivative work, containing both original code, included code and modified
// code that was published under the GNU Lesser General Public License. Copyrights for modified 
// and included code are below.
//
// Copyright (C) 1999-2001 Oculan Corp. All rights reserved.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// See: http://www.fsf.org/copyleft/lesser.html
//
// For more information contact:
//      joeSNMP Licensing       <joesnmp-license@lists.sourceforge.net>
//      http://sourceforge.net/projects/joesnmp/
//
// Modifications:
//
// $Id: SnmpRequest.java,v 1.2 2005/03/22 19:02:32 aozarov Exp $
//
//2005 Mar 21: aozarov@hotmail.com
//     change the interaction between the request and the session to deprecate the need for 
//     the cleanup timer task
//

package org.opennms.protocols.snmp;

import java.io.IOException;

import org.opennms.protocols.snmp.asn1.AsnEncodingException;

/**
 * <P>Implements a way to track outstanding SNMP pdu request.
 * The object tracks the pdu, it's sending parameters,
 * the number of times send, and the last time sent.</P>
 *
 * <P>The SnmpRequest implements the Runnable interface and
 * its run method is invoked by the corresponding SnmpTimer 
 * when the request effectively expires. The member m_expires
 * refers the the expiretion of the request and pdu,
 * not to the failed response.</P>
 *
 * @author	<A HREF="mailto:weave@oculan.com">Brian Weaver</A>
 * @author	<A HERF="http://www.opennms.org">OpenNMS</A>
 *
 * @version	1.1.1.1
 *
 */
class SnmpRequest implements Runnable
{
    private Integer m_requestID;
    
	/**
	 * The session that created this request
	 *
	 */
	private SnmpSession	m_session;	// the seesion who sent it

	/**
	 * The pdu transemited to the SnmpPeer.
	 *
	 */
	private SnmpSyntax	m_pdu;		// the PDU

	/**
	 * The SnmpHandler to invoke for this request
	 *
	 */
	private SnmpHandler	m_handler;

	/**
	 * The number of times this request has been
	 * transmitted.
	 *
	 */
	private int m_timesSent;	// number of times sent

	/**
	 * When set the request as a whole has expired and 
	 * should no longer be process by any methods. It is
	 * effectively waiting for garbage collection.
	 *
	 */
	private boolean		m_expired;	// when set DO NOT PROCESS

    /**
     * True is this request represents a trap
     */
    private boolean m_isTrap;

	/**
	 * Used to create an SnmpRequest object. This constructor
	 * sets the fields to their default value along with the
	 * passed parameters.
	 *
	 * @param session	The sending session
	 * @param pdu		The pdu to send to the remote
	 * @param handler	The handler to invoke!
	 *
	 */
	SnmpRequest(SnmpSession session, Integer requestID, SnmpPduPacket pdu, SnmpHandler handler)
	{
		m_session   = session;
        m_requestID = requestID;
		m_pdu	    = pdu;
		m_handler   = handler;
        m_isTrap    = pdu.getCommand() == SnmpPduPacket.V2TRAP;
	}

	/**
	 * Used to create an SnmpRequest object. This constructor
	 * sets the fields to their default value along with the
	 * passed parameters.
	 *
	 * @param session	The sending session
	 * @param pdu		The pdu to send to the remote
	 * @param handler	The handler to invoke!
	 *
	 */
	SnmpRequest(SnmpSession session, SnmpPduTrap pdu, SnmpHandler handler)
	{
		m_session   = session;
		m_pdu	    = pdu;
		m_handler   = handler;
        m_isTrap    = true;
	}

	/**
	 * Used to process the timeout of an SnmpRequest. The method is 
	 * invoked by the session timer object. If the request has "expired"
	 * then no processing occurs. If the number of retries have 
	 * exceeded the session parameters then the SnmpHandler's 
	 * snmpTimeoutError() method is invoked. If an error occurs transmiting
	 * the pdu then the snmpInternalError() method is invoked. 
	 *
	 * @see SnmpHandler
	 * @see SnmpPduRequest
	 * @see SnmpSession
	 */
	public void run()
	{
		if(m_expired)
			return;

		if(m_timesSent < m_session.getPeer().getRetries())
		{
			m_timesSent++;
			try
			{
				//
				// An SNMP Trap command should only be sent ONCE!
				//
				m_session.transmit(this);
				if(m_isTrap)
                    m_expired = true;
                else
    				m_session.getTimer().schedule(this, m_session.getPeer().getTimeout());
			}
			catch(IOException err)
			{
                m_session.snmpInternalError(this, SnmpSession.ERROR_IOEXCEPTION);
			}
			catch(SnmpPduEncodingException err)
			{
                m_session.snmpInternalError(this, SnmpSession.ERROR_ENCODING);
			}
			catch(AsnEncodingException err)
			{
                m_session.snmpInternalError(this, SnmpSession.ERROR_ENCODING);
			}
            catch (RuntimeException ex)
            {
                m_session.snmpInternalError(this, SnmpSession.ERROR_RUNTIME);
            }
		}
		else
		{
            m_session.snmpTimeoutError(this);
		}
	}	
    
    public boolean isTrap()
    {
        return m_isTrap;
    }
        
    public boolean isExpired()
    {
        return m_expired;
    }
    
    public void setExpired(boolean expired)
    {
        m_expired = expired;
    }
    
    public SnmpHandler getHandler()
    {
        return m_handler;
    }
    
    public SnmpSyntax getPdu()
    {
        return m_pdu;
    }
    
    public Integer getRequestID()
    {
        return m_requestID;
    }
}

	
	
