//
// This file is part of the joeSNMP Java Library.
//
// joeSNMP is Copyright (C) 2002-2003 Blast Internet Services, Inc.  All rights reserved.
// joeSNMP is a derivative work, containing both original code, included code and modified
// code that was published under the GNU Lesser General Public License. Copyrights for modified 
// and included code are below.
//
// Original code base Copyright (C) 1999-2001 Oculan Corp.  All rights reserved.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// See: http://www.fsf.org/copyleft/lesser.html
//
// For more information contact:
//      joeSNMP Licensing       <joesnmp-license@lists.sourceforge.net>
//      http://sourceforge.net/projects/joesnmp/
//
// Modifications:
//
// $Id: SnmpSession.java,v 1.6 2005/03/23 14:16:59 aozarov Exp $
//
// 2005 Mar 21: aozarov@hotmail.com
//      Change interaction with SnmpRequest (deprecate the need for Cleanup task)
//      Added more constructors to enable setting send/receive buffers
//      Optimize access to the current requests set
//      Consider the community string from peer when sending requests
//      Remove the check that compares the community string in the received pdu
//      with the community string of the sent pdu 
// 
// 2004 Sep 20: Dimitris.Andreadis@jboss.org
//      Added local binding address
//
// 2003 Mar 20:
//      Added code to allow the joeSNMP library to answer SNMP requests
//      to be used in creating SNMP agents.
//
// 2000 May 25: Sowmya
//      Added the SnmpPortal member to abstract the communication related
//      data out of this class. Added SnmpSessionPacketHandler to
//      handle callbacks from the SnmpPortal
//
// 2000 Mar 29: Weave
//		Made some minor changes here and there, but the biggest change
//		is from ArrayList to LinkedList for the outstanding request. I 
//		realized that the session was mostly adding/deleting request
//		and linked list was faster. When the ArrayList was being used
//		the search was O(N), just like a linked list (actually N/2 on average).
//		So the performance gains from the faster add/remove should help
//		until a better structure for searching, adding, and deleting 
//		can be used.
//

package org.opennms.protocols.snmp;

import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.SocketException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.opennms.protocols.snmp.asn1.ASN1;
import org.opennms.protocols.snmp.asn1.AsnEncoder;
import org.opennms.protocols.snmp.asn1.AsnEncodingException;

/**
 * <P>The SnmpSession is the main connection between the SNMP manager
 * and the SNMP Agent. All the request flow through this class. To 
 * use the SnmpSession class a SnmpHandler class must be defined
 * to process any errors or responses through the library.</P>
 *
 * <P>Once the session is created the creator must call <EM>close()</EM> to 
 * ensure an orderly release of threads and resources.</P>
 *
 * @author <A HREF="mailto:weave@oculan.com">Brian Weaver</A>
 * @author <A HREF="http://www.opennms.org/">OpenNMS</A>
 *
 * @version 1.1.1.1 2001/11/11 17:27:22
 *
 * @see SnmpHandler
 * @see SnmpPacketHandler
 * @see SnmpPortal
 */
public class SnmpSession extends Object
{
    /**
	 * This is the command passed to the SnmpHandler
	 * if a timeout occurs. All errors are less than
	 * zero.
	 *
	 * @see SnmpHandler
	 */
	public static final int ERROR_TIMEOUT		= -1;

	/**
	 * This is the command passed to the SnmpHandler
	 * if an IOException occurs while attempting to
	 * transmit the request
	 *
	 * @see SnmpHandler
	 */
	public static final int ERROR_IOEXCEPTION	= -2;
	
	/** 
	 * This is the command passed to the SnmpHandler
	 * if an encoding exception is generated when
	 * attempting to send an SnmpPduRequest message
	 *
	 * @see SnmpHandler
	 */
	public static final int ERROR_ENCODING		= -3;

    /**
     * This is the command passed to the SnmpHandler
     * if an unexpected runtime exception was generated when
     * attempting to send an SnmpPduRequest message
     *
     * @see SnmpHandler     
     */
    public static final int ERROR_RUNTIME      = -4;
    
	/**
	 * Used to contain a list of outstanding request
	 * for the session. The list should only
	 * contain SnmpRequest objects!
	 */
	private Map m_requestsMap = new HashMap();

	/**
	 * The SNMP peer to whom this session will communicate
	 * with. The peer contains the parameters also.
	 *
	 * @see SnmpParameters
	 */
	private SnmpPeer m_peer;

	/**
	 * The timer object used to schedule the SnmpRequest
	 * timeouts. It is also used to schedule the cleanup
	 * of the m_requests list.
	 *
	 * @see org.opennms.protocols.snmp.SnmpSession.CleanupRequest
	 * @see SnmpRequest
	 */
	private SnmpTimer m_timer;

	/**
	 * The default SNMP callback handler. If this is not
	 * set and it is needed then an SnmpHandlerNotDefinedException
	 * is thrown.
	 */
	private SnmpHandler m_defHandler;

	/**
	 * ASN encoder
	 */
	AsnEncoder m_encoder;

	//
	// thread related stuff
	//

	/**
	 * Provides a synchronization point
	 */
	private Object m_sync = new Object();

	/**
	 * If the boolean variable is set then 
	 * the destroy() method must have been called
	 */
	private boolean	m_stopRun;

	/**
	 * the receiver thread 
	 */
	private SnmpPortal m_portal;

    
    private static final int EXPIRED_REQUEST_CLEANUP_FREQUENCY = 30000;
    
	/**
	 * Inner class SessionHandler implements the interface
	 * SnmpPacketHandler to handle callbacks from the SnmpPortal
	 */
	private class SessionHandler implements SnmpPacketHandler
	{
		public void processSnmpMessage(InetAddress	agent,
					       int		port,
					       SnmpInt32	version,
					       SnmpOctetString	community,
					       int		pduType,
					       SnmpPduPacket	pdu)
		{
            /*
            
			//
			// now find the request and inform
			//
			SnmpRequest req   = null;
            synchronized (m_requestsMap)
            {
            Integer key = new Integer(pdu.getRequestId());
                req = (SnmpRequest) m_requestsMap.get(key);
                if (req == null || req.isExpired())
                    return;
            }            

			if(!(req.getPdu() instanceof SnmpPduPacket))
                throw new SnmpPduEncodingException("Invalid PDU Type for session received");

            // No need for check here as there are quite few devices which don't follow
            // The rule of returning the community string that was sent to them
            // This doesn't seem a compromise of security

            SnmpPduPacket reqPdu = (SnmpPduPacket) req.getPdu();
  			int cmd = reqPdu.getCommand();

            switch(cmd)
			{
                case SnmpPduPacket.SET:
					{
						String tst = new String(community.getString());
						String wr  = reqPdu.getPeer() == null ? m_peer.getParameters().getWriteCommunity() :
                            reqPdu.getPeer().getParameters().getWriteCommunity();
                        
						if(!tst.equals(wr))
							throw new SnmpPduEncodingException("Invalid community string");
					}
                    break;

                    
				case SnmpPduPacket.GET:
				case SnmpPduPacket.GETNEXT:
				case SnmpPduPacket.INFORM:
				case SnmpPduPacket.GETBULK:
				case SnmpPduPacket.REPORT:
					{
						String tst = new String(community.getString());
						String rd  = reqPdu.getPeer() == null ? m_peer.getParameters().getReadCommunity() :
                            reqPdu.getPeer().getParameters().getReadCommunity();
						
                        if(!tst.equals(rd))
							throw new SnmpPduEncodingException("Invalid community string");
					}                    
					break;
                
				default:
					throw new SnmpPduEncodingException("Invalid PDU Type for session received");
			}
*/
            Integer key = new Integer(pdu.getRequestId());

            SnmpRequest req = null;
            synchronized (m_requestsMap)
            {
                req = (SnmpRequest) m_requestsMap.remove(key);
                if (req == null)
                    return;
                
                req.setExpired(true);
            }
            
            try
            {
                req.getHandler().snmpReceivedPdu(SnmpSession.this, pdu.getCommand(), pdu);
            }
            catch (Exception ex)
            {
                // ignore
            }
		}

		public void processSnmpTrap(InetAddress		agent, 
					    int			port, 
					    SnmpOctetString	community,
					    SnmpPduTrap		pdu) throws SnmpPduEncodingException
		{
			throw new SnmpPduEncodingException("Invalid PDU Type for session");
		}


		public void processBadDatagram(DatagramPacket p)
		{
			// do nothing - discard?
		}

		public void processException(Exception e)
		{
			// do nothing - discard?
		}
	}
	
	/**
	 * <P>Encapsulates a byte array and the number of
	 * bytes of valid data in the array. The length passed
	 * to the constructor is normally less then the length
	 * of the encapsulated array.</P>
	 *
	 */
	private static class ByteArrayInfo
	{
		/**
		 * The buffer
		 */
		private byte[]  m_buf;
		
		/**
		 * The valid length of the buffer
		 */
		private int	m_length;
		
		/**
		 * Builds an encapuslated array with the passed
		 * buffer and its <EM>valid</EM> length set to
		 * <EM>length</EM>.
		 *
		 * @param buf		The buffer.
		 * @param length	The valid length of the buffer.
		 */
		public ByteArrayInfo(byte[] buf, int length)
		{
			m_buf = buf;
			m_length = length;
		}
		
		/**
		 * returns the encapsulated array
		 */
		public byte[] array()
		{
			return m_buf;
		}
		
		/**
		 * Returns the valid length of the encapsulate array
		 */
		public int size()
		{
			return m_length;
		}
	}
	
	/**
	 * <P>This method is used to encode the passed protocol data unit
	 * and return the encoding. The encoding is performed using the 
	 * default encoder for the session and limits the size to a
	 * 16K buffer. </P>
	 *
	 * @param pdu	The pdu to encode
	 *
	 * @return The encoded pdu request
	 *
	 * @exception SnmpPduEncodingException	Thrown if an encoding exception 
	 *		occurs at the session level
	 * @exception org.opennms.protocols.snmp.asn1.AsnEncodingException Thrown 
	 *		if an encoding exception occurs in the AsnEncoder object.
	 */
	private ByteArrayInfo encode(SnmpPeer peer, SnmpPduPacket pdu) throws SnmpPduEncodingException, AsnEncodingException
	{
		SnmpParameters	parms = peer.getParameters();
		AsnEncoder encoder = peer.getParameters().getEncoder();
        
		//
		// Get the encoder and start
		// the encoding process
		//
		// get a suitable buffer (16k)
		//
		int offset = 0;
		byte[] buf= new byte[16 * 1024];

		//
		// encode the snmp version
		//
		SnmpInt32 version = new SnmpInt32(parms.getVersion());
		offset = version.encodeASN(buf, offset, encoder);

		//
		// get the correct community string. The
		// SET command uses the write community, all
		// others use the read community
		//
		SnmpOctetString community;
		if(pdu.getCommand() == SnmpPduPacket.SET)
		{
			String wrComm = parms.getWriteCommunity();
			if(wrComm == null)
				throw new SnmpPduEncodingException("Requested SET but there is no write community");
			community = new SnmpOctetString(wrComm.getBytes());
		}
		else
		{
			community = new SnmpOctetString(parms.getReadCommunity().getBytes());
		}

		//
		// encode the community strings
		//
		offset = community.encodeASN(buf, offset, encoder);
		offset = pdu.encodeASN(buf, offset, encoder);

		//
		// build the header, don't forget to mark the
		// pivot point
		//
		int pivot = offset;
		offset = encoder.buildHeader(buf, 
					      offset,
					      (byte)(ASN1.SEQUENCE | ASN1.CONSTRUCTOR),
					      pivot);

		//
		// rotate the buffer around the pivot point
		//
		SnmpUtil.rotate(buf, 0, pivot, offset);
		return new ByteArrayInfo(buf,offset);
	}
	
	/**
	 * <P>This method is used to encode the passed protocol data unit
	 * and return the encoding. The encoding is performed using the 
	 * default encoder for the session and limits the size to a
	 * 16K buffer. </P>
	 *
	 * @param pdu	The pdu to encode
	 *
	 * @return The encoded pdu request
	 *
	 * @exception SnmpPduEncodingException	Thrown if an encoding exception 
	 *		occurs at the session level
	 * @exception org.opennms.protocols.snmp.asn1.AsnEncodingException Thrown 
	 *		if an encoding exception occurs in the AsnEncoder object.
	 */
	private ByteArrayInfo encode(SnmpPduTrap pdu) throws AsnEncodingException
	{
		SnmpPeer	peer	= m_peer;
		SnmpParameters	parms	= peer.getParameters();
		//
		// get a suitable buffer (16k)
		//
		int offset = 0;
		byte[] buf= new byte[16 * 1024];

		//
		// encode the snmp version
		//
		SnmpInt32 version = new SnmpInt32(parms.getVersion());
		offset = version.encodeASN(buf, offset, m_encoder);

		//
		// get the correct community string. The
		// SET command uses the write community, all
		// others use the read community
		//
		SnmpOctetString community = new SnmpOctetString(parms.getReadCommunity().getBytes());

		//
		// encode the community strings
		//
		offset = community.encodeASN(buf, offset, m_encoder);
		offset = pdu.encodeASN(buf, offset, m_encoder);

		//
		// build the header, don't forget to mark the
		// pivot point
		//
		int pivot = offset;
		offset = m_encoder.buildHeader(buf, 
					       offset,
					       (byte)(ASN1.SEQUENCE | ASN1.CONSTRUCTOR),
					       pivot);

		//
		// rotate the buffer around the pivot point
		//
		SnmpUtil.rotate(buf, 0, pivot, offset);
		return new ByteArrayInfo(buf, offset);
	}

	/**
	 * Returns the internal timer object for the 
	 * SNMP Session.
	 *
	 * @return The internal timer object
	 *
	 */
    SnmpTimer getTimer()
	{
		return m_timer;
	}

	/**
	 * Transmits the specified SnmpRequest to the SnmpPeer defined
	 * by the session. First the SnmpPdu contained within the request
	 * is encoded using the session AsnEncoder, as defined by the
	 * SnmpParameters. Once the packet is encoded it is transmitted
	 * to the agent defined by SnmpPeer. If an error occurs an 
	 * appropiate exception is generated.
	 * This method is called by SnmpRequest.
     * 
	 * @param req	The SnmpRequest to transmit
	 *
	 * @exception SnmpPduEncodingException	Thrown if an encoding exception 
	 *		occurs at the session level
	 * @exception org.opennms.protocols.snmp.asn1.AsnEncodingException Thrown 
	 *		if an encoding exception occurs in the AsnEncoder object.
	 * @exception java.io.IOException	Thrown if an error occurs sending the 
	 *		encoded datagram
	 *
	 * @see SnmpRequest
	 * @see SnmpParameters
	 * @see SnmpPeer
	 *
	 */
	void transmit(SnmpRequest req) throws SnmpPduEncodingException, AsnEncodingException, java.io.IOException
	{
		//
		// break down the pieces into usable variables
		//
		SnmpPduPacket	pdu	= null;
		SnmpPduTrap	trap	= null;
		SnmpPeer	peer	= m_peer;
		SnmpParameters	parms	= peer.getParameters();
		
		if(req.getPdu() instanceof SnmpPduPacket)
        {
			pdu = (SnmpPduPacket) req.getPdu();
            if (pdu.getPeer() != null)
                peer = pdu.getPeer();
        }
        else if(req.getPdu() instanceof SnmpPduTrap)
			trap = (SnmpPduTrap) req.getPdu();

		//
		// verify that for a SNMPV1 session that no
		// SNMPV2 packets are transmitted!
		//
		if(pdu != null)
		{
			switch(pdu.getCommand())
			{
			case SnmpPduPacket.INFORM:
			case SnmpPduPacket.V2TRAP:
			case SnmpPduPacket.REPORT:
			case SnmpPduPacket.GETBULK:
				if(parms.getVersion() < SnmpSMI.SNMPV2)
				{
					throw new SnmpPduEncodingException("Cannot send pdu, invalid SNMP version");
				}
			}
			
			//
			// transmit the datagram
			//
			ByteArrayInfo msg = encode(peer, pdu);
		    m_portal.send(peer, msg.array(), msg.size());
		}
		else if(trap != null)
		{
			ByteArrayInfo msg = encode(trap);
			m_portal.send(m_peer, msg.array(), msg.size());
		}
		else
		{
			throw new SnmpPduEncodingException("Invalid PDU type passed to method");
		}
	}

    // Called by SnmpRequest
    void snmpInternalError(SnmpRequest request, int errorCode)
    {
        if (request.isExpired())
            return;

        Integer id = request.getRequestID();
        synchronized (m_requestsMap)
        {
            Object tmp = m_requestsMap.get(id);
            if (tmp != request)
                return;
            
            m_requestsMap.remove(id);
            request.setExpired(true);
        }

        try
        {
            request.getHandler().snmpInternalError(this,errorCode, request.getPdu());
        }
        catch(Exception e)
        {
            // ignore
        }
    }    

    // Called by SnmpRequest
    public void snmpTimeoutError(SnmpRequest request)
    {
        if (request.isExpired())
            return;
        
        Integer id = request.getRequestID();
        synchronized (m_requestsMap)
        {
            Object tmp = m_requestsMap.get(id);
            if (tmp != request)
                return;
            
            m_requestsMap.remove(id);
            request.setExpired(true);
        }

        try
        {
            request.getHandler().snmpTimeoutError(this, request.getPdu());
        }
        catch(Exception e)
        {
            // ignore
        }
    }

	/**
	 * The default SnmpSession constructor. The object
	 * is constructed with a default SnmpPeer object.
	 *
	 * @param peer	The peer agent
	 *
	 * @see SnmpPeer
	 *
	 * @exception java.net.SocketException If thrown it is from the creation
	 *		of a DatagramSocket.
	 *
	 */
	public SnmpSession(InetAddress peer) throws SocketException
	{
        this(new SnmpPeer(peer)) ;
	}

	/**
	 * Constructs the SnmpSession with the specific SnmpPeer.
	 *
	 * @param peer	The SnmpPeer used to configure this session
	 *
	 * @see SnmpPeer
	 *
	 * @exception java.net.SocketException If thrown it is from the creation
	 *		of a DatagramSocket.
	 *
	 */
	public SnmpSession(SnmpPeer peer) throws SocketException
    {
        this(peer, SnmpPortal.DEFAULT_THREADPOOL_SIZE, SnmpPortal.DEFAULT_RECEIVE_BUFFER_SIZE,
                SnmpPortal.DEFAULT_SEND_BUFFER_SIZE);
    }
    
    /**
     * Constructs the SnmpSession with the specific
     * parameters. The parameters are associated with
     * the default SnmpPeer object.
     *
     * @param peer   The peer address for agent
     * @param params The SnmpParameters to configure with this session
     *
     * @see SnmpPeer
     * @see SnmpParameters
     *
     * @exception java.net.SocketException If thrown it is from the creation
     *      of a DatagramSocket.
     *
     */
    public SnmpSession(InetAddress peer, SnmpParameters params) throws SocketException
    {
        this(peer);
        m_peer.setParameters(params);
    }

    public SnmpSession(SnmpPeer peer, int threadPoolSize, int receiveBuffer, int sendBuffer) throws SocketException
	{
        m_peer = peer;
		m_encoder  = peer.getParameters().getEncoder();
		m_portal   = new SnmpPortal(new SessionHandler(), 
					    m_encoder,
					    peer.getServerPort(),
                        peer.getServerAddress(),
                        threadPoolSize,
                        receiveBuffer,
                        sendBuffer);

        m_timer     = new SnmpTimer();
        m_timer.schedule(new CleanupRequest(), EXPIRED_REQUEST_CLEANUP_FREQUENCY);
	}

	/**
	 * Gets the default SnmpHandler for the session. If
	 * the handler has never been set via the setDefaultHandler()
	 * method then a null will be returned.
	 *
	 * @return The default SnmpHandler, a null if one has never
	 *	been registered.
	 *
	 */
	public SnmpHandler getDefaultHandler()
	{
		return m_defHandler;
	}

	/**
	 * Sets the default SnmpHandler.
	 *
	 * @param hdl	The new default handler
	 *
	 */
	public void setDefaultHandler(SnmpHandler hdl)
	{
		m_defHandler = hdl;
	}

	/**
	 * Gets the current peer object.
	 *
	 * @return The current SnmpPeer object
	 *
	 */
	public SnmpPeer getPeer( )
	{
		return m_peer;
	}

	/**
	 * Sets the passed SnmpPeer object to the one
	 * used for all new SNMP communications. This
	 * includes any outstanding retries.
	 *
	 * @param peer The SnmpPeer object for the sesison
	 *
	 */
	public void setPeer(SnmpPeer peer)
	{
		m_peer = peer;
		setAsnEncoder(peer.getParameters().getEncoder());
	}

	/**
	 * Returns the number of outstanding request for 
	 * the agent. An outstanding request is one
	 * that has no yet responded to the query.
	 *
	 * @return The number of outstanding request
	 *
	 * @throws java.lang.IllegalStateException Throw if the session has been closed.
	 */
	public int getOutstandingCount()
	{
		//
		// check to ensure that the session is still open
		//
		synchronized(m_sync)
		{
			if(m_stopRun) // session has been closed!
				throw new IllegalStateException("illegal operation, the session has been closed");
            
            synchronized(m_requestsMap)
            {
                cleanupExpiredRequests();
                return m_requestsMap.size();
            }
        }
	}

	/**
	 * Cancels the current outstanding reqeust as defined
	 * by the SnmpPduPacket's requestId method.
	 *
	 * @param requestId	The request to cancel
	 *
	 * @see SnmpPduPacket
	 *
	 * @throws java.lang.IllegalStateException Throw if the session has been closed.
	 */
	public void cancel(int requestId)
	{
        Integer requestID = new Integer(requestId);
        
		//
		// check to ensure that the session is still open
		//
		synchronized(m_sync)
		{
			if(m_stopRun) // session has been closed!
				throw new IllegalStateException("illegal operation, the session has been closed");

            synchronized(m_requestsMap)
    		{
                SnmpRequest req = (SnmpRequest) m_requestsMap.remove(requestID);
                if (req != null)
                    req.setExpired(true);
    		}
        }
	}

	/**
	 * Send the SNMP PDU to the remote agent and invokes the 
	 * specified handler when the packet is recieve. This is
	 * a non-blocking call.
	 *
	 * @param pdu		The pdu to encode and send
	 * @param handler	The handler object for this request
	 *
	 * @return The request identifier for the newly sent PDU.
	 *
	 * @exception SnmpHandlerNotDefinedException Thrown if the handler is null
	 * @exception java.lang.IllegalStateException Thrown if the session has been closed.
	 */
	public int send(SnmpPduPacket pdu, SnmpHandler handler)
	{
		if(handler == null)
			throw new SnmpHandlerNotDefinedException("No Handler Defined");

        Integer requestId = new Integer(pdu.getRequestId());
        SnmpRequest req = new SnmpRequest(this, requestId, pdu, handler);
        boolean shouldHoldRequest = !req.isTrap();
                    
        //
		// check to ensure that the session is still open
		//
		synchronized(m_sync)
		{
			if(m_stopRun) // session has been closed!
				throw new IllegalStateException("illegal operation, the session has been closed");

            if(shouldHoldRequest) // traps and responses don't get answers!
            {
                synchronized(m_requestsMap)
                {
                    SnmpRequest old = (SnmpRequest) m_requestsMap.get(requestId);
                    if (old == null || old.isExpired())
                        m_requestsMap.put(requestId, req);
                    else
                        throw new IllegalStateException("Session has an active request with the same id");
                }
            }
        }
        
		req.run();
        return shouldHoldRequest ? 0 : requestId.intValue(); 
	}

	/**
	 * Sends the SNMP PDU to the remote agent and uses the
	 * default SnmpHandler to process the request. This is
	 * a non-blocking call
	 *
	 * @param pdu	The pdu to encode and send
	 *
	 * @return The request identifier for the newly sent PDU.
	 *
	 * @exception SnmpHandlerNotDefinedException Thrown if the handler is null
	 * @exception java.lang.IllegalStateException Thrown if the session has been closed.
	 */
	public int send(SnmpPduPacket pdu) 
	{
		return send(pdu, m_defHandler);
	}

	/**
	 * Send the SNMP PDU Trap to the remote agent. This is
	 * a non-blocking call.
	 *
	 * @param pdu		The pdu to encode and send
	 * @param handler	The handler object for this request
	 *
	 * @return The request identifier for the newly sent PDU.
	 *
	 * @exception SnmpHandlerNotDefinedException Thrown if the handler is null
	 * @exception java.lang.IllegalStateException Thrown if the session has been closed.
	 */
	public int send(SnmpPduTrap pdu, SnmpHandler handler)
	{
		if(handler == null)
			throw new SnmpHandlerNotDefinedException("No Handler Defined");
		
		//
		// check to ensure that the session is still open
		//
		synchronized(m_sync)
		{
			if(m_stopRun) // session has been closed!
				throw new IllegalStateException("illegal operation, the session has been closed");
		}

		SnmpRequest req = new SnmpRequest(this, pdu, handler);
		req.run();
		return 0;
	}

	/**
	 * Sends the SNMP PDU Trap to the remote agent. This is
	 * a non-blocking call
	 *
	 * @param pdu	The pdu to encode and send
	 *
	 * @return The request identifier for the newly sent PDU.
	 *
	 * @exception SnmpHandlerNotDefinedException Thrown if the handler is null
	 * @exception java.lang.IllegalStateException Thrown if the session has been closed.
	 */
	public int send(SnmpPduTrap pdu) 
	{
		return send(pdu, m_defHandler);
	}

	/**
	 * Returns true if the <CODE>close</CODE> method has
	 * been called. The session cannot be used to send
	 * request after <CODE>close</CODE> has been executed.
	 *
	 */
	public boolean isClosed()
	{
		synchronized(m_sync)
		{
			return m_stopRun;
		}
	}

	/**
	 * Used to close the session. Once called the session should
	 * be considered invalid and unusable.
	 *
	 * @exception java.lang.IllegalStateException Thrown if the session has already
	 * 	been closed by another thread.
	 */
	public void close()
	{
		synchronized(m_sync)
		{
			//
			// only allow close to be called once
			//
			if(m_stopRun)
				throw new IllegalStateException("The session is already closed");
				
			m_stopRun = true;
		}
        
        m_timer.cancel(true);
        m_portal.close();
	}

	/**
	 * Sets the default encoder.
	 *
	 * @param encoder	The new encoder
	 *
	 */
	public void setAsnEncoder(AsnEncoder encoder)
	{
		m_encoder = encoder;
		m_portal.setAsnEncoder(encoder);
	}

	/**
	 * Gets the AsnEncoder for the session. 
	 *
	 * @return the AsnEncoder
	 */
	public AsnEncoder getAsnEncoder()
	{
		return m_encoder;
	}

	/**
	 * Allows library users to register new ASN.1 types
	 * with the SNMP library. The object must support
	 * all methods of the SnmpSyntax interface. The object
	 * is registered globally with the library and is visible
	 * to all session after it is registered.
	 *
	 * @param object The new SnmpSyntax object to register
	 */
	public static void registerSyntaxObject(SnmpSyntax object)
	{
		SnmpUtil.registerSyntax(object);
	}

    private void cleanupExpiredRequests()
    {
        if(!m_requestsMap.isEmpty())
        {
            Iterator iter = m_requestsMap.values().iterator();
            
            while(iter.hasNext())
            {
                SnmpRequest req = (SnmpRequest) iter.next();
                if(req.isExpired())
                {
//                    System.out.println("***************** cleanupExpiredRequests(" + this + ") cleaned -> "
//                            + req.getRequestID()
//                            + ", " + req.getPdu()
//                            + ", " + ((SnmpPduPacket) req.getPdu()).getPeer().getPeer()
//                            + ", " + ((SnmpPduPacket) req.getPdu()).getPeer().getParameters().getReadCommunity()
//                            );
                    iter.remove();
                }
            } 
        }
    }
    
    /**
     * This class is used to periodically cleanup the 
     * outstanding request that have expired. The cleanup
     * interval is nominally once every 5 to 10 seconds.
     * It's used like the garbage collector for the m_requests
     * list. This is used in hopes of minimizing the contention
     * for the request array
     *
     */
    private class CleanupRequest implements Runnable
    {
        private CleanupRequest()
        {
            // Do nothing
        }
        
        /**
         * Preforms the actual removal of the expired
         * SnmpRequest elements.
         *
         * @see SnmpRequest
         *
         */
        public void run()
        {
            if (m_stopRun)
                return;
            
            try
            {
                synchronized(m_requestsMap)
                {
                    cleanupExpiredRequests();
                }                    
            }
            catch (Throwable te)
            {
                // ignore
            }
                
            m_timer.schedule(this, EXPIRED_REQUEST_CLEANUP_FREQUENCY);
        }
    }
} // end of SnmpSession class
