/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.util;

import java.io.IOException;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.ldap.InitialLdapContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.util.apllog.AplLogger;

/**
 * LDAPとのコネクションを管理するクラス<BR>
 *
 * @version 2.1.0
 * @since 1.0.0
 */
public class LdapConnectionManager {
	/** ログ出力のインスタンス */
    protected static Log m_log = LogFactory.getLog( LdapConnectionManager.class );

	/** 
	 * RMIAdaptorのルックアップ名。<BR>
	 * 障害検知へ通知するために使用します。
	 */
    private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	/**
	 * オブジェクト名。<BR>
	 * 障害検知サービスのオブジェクト名を指定します。
	 */
    private static final String OBJECT_NAME = "user:name=TroubleDetection,service=TroubleDetectionService";
	/**
	 * 操作名。<BR>
	 * 障害検知へ通知する為の操作名を指定します。
	 */
    private static final String OPERATION_NAME = "putMessage";
	
    /** プロバイダへの再接続間隔 */
	private static final int PROVIDER_CONNECT_INTERVAL = 60000;
	/** 再接続回数 */
	private static final int RECONNECT_MAX = 5;
	
	/** プロバイダ接続フラグ */
    private static boolean m_provider = true;
    /** コンシューマ接続フラグ */
    private static boolean m_consumer = false;
    /** 最終接続日時 */
    private static long m_conenctTime = System.currentTimeMillis();
    /** 最終出力ログ */
    private static String m_outputLog = "";
    
	private static LdapConnectionManager m_instance = null;

	/**
	 * 唯一の自身のインスタンスを返します。<BR>
	 * <code> LdapConnectionManager </code>オブジェクトを保持します。
	 * 
	 * @return 自身のインスタンス
	 */
	public static LdapConnectionManager getConnectionManager() {
		if (m_instance==null) {
			m_instance = new LdapConnectionManager();
		}
		return m_instance;
	}

	/**
	 * コンストラクタ
	 */
	private LdapConnectionManager() {

	}
	
	/**
	 * LDAP用コンテキストを返します。
	 * <p>
	 * <ol>
	 * <li>最終接続日時からプロバイダへの再接続間隔が経過した場合、プロバイダ接続フラグを立てる。</li>
	 * <li>プロバイダ接続フラグが立っている場合、プロバイダに接続します。</li>
	 * <li>プロバイダへの接続に失敗した場合、コンシューマに接続します。</li>
	 * <li>プロバイダ接続フラグが立っていない場合、コンシューマに接続します。</li>
	 * </ol>
	 * 
	 * @return LDAP用コンテキスト
	 */
	public javax.naming.directory.DirContext getDirContext() {
		
		boolean provider = false;
		boolean consumer = false;
		synchronized (this) {
			long diff = System.currentTimeMillis() - m_conenctTime;
			if(diff >= PROVIDER_CONNECT_INTERVAL){
				provider = true;
				m_conenctTime = System.currentTimeMillis();
				m_log.debug("getDirContext() : LDAP ReConnect");
			}
			else{
				provider = m_provider;
			}
			consumer = m_consumer;
		}
		
		javax.naming.directory.DirContext ctx = null;
		try {
			//プロバイダLDAP用のコンテキストの作成
			if(provider){
				ctx = getDirContext("external/hinemos/ldap/provider");
			}
			else{
				throw new NamingException();
			}
			
			//コンシューマから切り替わった場合、ログ出力
			if(consumer){
				outputLog("LDAP03", "001");
			}
			
			provider = true;
			consumer = false;
			
		} catch (NamingException e) {
			//コンシューマLDAP用のコンテキストの作成
			try {
				ctx = getDirContext("external/hinemos/ldap/consumer");
			} catch (NamingException e1) {
			}
			
			if(ctx != null){
				//プロバイダから切り替わった場合、ログ出力
				if(provider && !consumer){
					outputLog("LDAP02", "001");
				}
				
				provider = false;
				consumer = true;
			}
		}

		if(ctx == null){
			AplLogger apllog = new AplLogger("REP", "rep");
			apllog.put("SYS", "001");
			
			//ログ出力
			if(provider){
				outputLog("LDAP02", "002");
			}
			else{
				outputLog("LDAP03", "002");
			}
			
		    provider = false;
		    consumer = false;
		}
		else{
			m_log.debug("getDirContext() : Get LDAP Connection");
		}
		
		synchronized (this) {
			m_provider = provider;
			m_consumer = consumer;
		}
		
		return ctx;
	}
	
	/**
	 * 指定されたJNDI名のLDAP用コンテキストを取得します。
	 * 
	 * @param jndiName JNDI名
	 * @return LDAP用コンテキスト
	 * @throws NamingException
	 */
	@SuppressWarnings("static-access")
	private javax.naming.directory.DirContext getDirContext(String jndiName) throws NamingException {
		javax.naming.directory.DirContext ctx = null;
		
		//LDAP用のコンテキストの作成
		for (int i = 0; i < RECONNECT_MAX; i++) {
			TimeKeeper timer = new TimeKeeper(15000); //timeout of 15 secs
			try {
				m_log.debug("getDirContext() : LDAP Connect " + jndiName);
				InitialContext iniCtx = new InitialContext();
				ctx = (InitialLdapContext)iniCtx.lookup(jndiName);
				
				if(ctx != null)
					break;
			} 
			catch (NamingException e) {
				m_log.error("getDirContext() : LDAP Connection Error " + jndiName + " : " + e.getMessage(), e);
			}
			finally {
				timer.setLdapResponse(true);
				Thread.currentThread().interrupted();
								
				// タイマースレッドを停止
				timer.interrupt();
			}
		}
		
		if(ctx == null)
			throw new NamingException();
		
		return ctx;
	}
	
	/**
	 * LDAP用コンテキストを設定します。<BR>
	 * 現状では何もしません。
	 * 
	 * @param ctx LDAP用コンテキスト
	 */
	public void setDirContext(javax.naming.directory.DirContext ctx) {
//	    try {
//            m_ctx.close();
//        } catch (NamingException e) {
//
//        }
//		m_ctx = ctx;
	}
	
	/**
	 * 障害検知へログ出力を通知します。
	 * 
	 * @param monitorId 監視項目ID
	 * @param messageId メッセージID
	 */
	private void outputLog(String monitorId, String messageId) {

		synchronized (m_outputLog) {
			String outputLog = monitorId + messageId;
			if(outputLog.equals("LDAP02" + "002")){
				m_outputLog = outputLog;
			}
			else if(outputLog.equals("LDAP03" + "002")){
				m_outputLog = outputLog;
			}
			else if(outputLog.equals(m_outputLog)){
				return;
			}
		}
		
		try{
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        Object[] args = {monitorId, messageId};
	        String[] signature = {String.class.getName(), String.class.getName()};
	        server.invoke(name, OPERATION_NAME, args, signature);

		} catch (NamingException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (NullPointerException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (MBeanException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (ReflectionException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (IOException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		}
	}
	
	class TimeKeeper extends Thread {
		long timeout; //msec
		long timerStart;
		
		boolean ldapResponse = false;
		Thread parent;
		
		TimeKeeper(long timout) {
			parent = Thread.currentThread();
			this.timeout = timout;
			this.start();
		}
		
		@SuppressWarnings("static-access")
		public void run() {
			timerStart = System.currentTimeMillis();
			
			long waitTime = timeout;
			
			while (waitTime > 0) {
				try {
					this.sleep(waitTime);
				} catch (InterruptedException ie) {
					// ignore and continue
					if (isLdapResponse()){
						break;
					}
				}
				waitTime = timeout - (System.currentTimeMillis() - timerStart);
			}
			synchronized (this) {
				if (!isLdapResponse()){
					m_log.error("LDAP Connection Timeout. [ " + timeout + " ms ]");
					parent.interrupt();
				}
			}
		}
		
		public boolean isLdapResponse() {
			return ldapResponse;
		}

		synchronized public void setLdapResponse(boolean ldapResponse) {
			this.ldapResponse = ldapResponse;
		}
	}
}
