/*
                                                                                                                                                                 
Copyright (C) 2008 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.operator;

import java.math.BigInteger;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performance.monitor.ejb.entity.CollectorSnmpMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorSnmpMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorSnmpMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorSnmpMstUtil;
import com.clustercontrol.performance.operator.Operator.InvalidValueException;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.TableEntry;

public abstract class SnmpOperator extends Operator {
	private static Log m_log = LogFactory.getLog( SnmpOperator.class );
	
	// SNMPで取得する値の型がCOUNTER32であった場合の最大値
	private final static long COUNTER32_MAX_VALUE = ((long)(Integer.MAX_VALUE))*2+1;
	private final static BigInteger COUTNER64_MAX_VALUE = BigInteger.valueOf(Long.MAX_VALUE).shiftLeft(1).add(BigInteger.valueOf(1)); 
	
	private HashMap<String, CollectorSnmpMstData> m_oidMap = new HashMap<String, CollectorSnmpMstData>();
	
	/**
	 * 収集項目コードを設定します。
	 * 同時に指定の収集項目コードの性能値算出式で使用可能な変数を設定します。
	 */
	@Override
	public void setItemCode(String itemCode) {
		// プライベート変数に収集項目コードを設定
		super.setItemCode(itemCode);
		
		try{
			// 項目コードごとに利用可能な変数名が定義されているため設定
			Collection<CollectorSnmpMstLocal> beans = CollectorSnmpMstUtil.getLocalHome().findAll();
			Iterator<CollectorSnmpMstLocal> itr = beans.iterator();

			// 保持している全ての変数IDをクリア
			getVariables().clear();
			
			// この収集項目で利用可能な変数IDをリストに保存
			while(itr.hasNext()){
				CollectorSnmpMstLocal bean = itr.next();

				if(bean.getPlatformId().equals(getPlatformId()) && 
						bean.getItemCode().equals(itemCode)){
					
					m_log.debug("add " + bean.getVariableId());
					
					getVariables().add(bean.getVariableId());
				}
			}
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
	}
	
	/**
	 * 直近の収集時の値を取得します
	 * @param variable 変数名
	 * @return 収集値
	 * @throws CollectedDataNotFoundException 
	 * @throws InvalidValueException 
	 */
	@Override
	public double getCurrentMibValue(String variable) 
	throws CollectedDataNotFoundException, InvalidValueException {
		return getValue(variable, getCurrentTable());
	}

	/**
	 * 前回収集時の値を取得します
	 * @param variable 変数名
	 * @return 収集値
	 * @throws CollectedDataNotFoundException 
	 * @throws InvalidValueException 
	 */
	@Override
	public double getPreviousMibValue(String variable) 
	throws CollectedDataNotFoundException, InvalidValueException{
		return getValue(variable, getPreviousTable());
	}
	
	/**
	 * 全変数の直近収集値の合計値を取得します
	 * @return 合計値
	 * @throws CollectedDataNotFoundException 
	 * @throws InvalidValueException 
	 */
	@Override
	public double getCurrentMibValueTotal()
	throws CollectedDataNotFoundException, InvalidValueException{
		Iterator<String> itr = getVariables().iterator();
		
		double total = 0;
		while(itr.hasNext()){
			String variable = itr.next();
			
			total = total + getCurrentMibValue(variable);
		}

		return total;
	}
	/**
	 * 全変数の前回収集値の合計値を取得します
	 * @return 合計値
	 * @throws CollectedDataNotFoundException 
	 * @throws InvalidValueException 
	 */
	@Override
	public double getPreviousMibValueTotal()
	throws CollectedDataNotFoundException, InvalidValueException{
		Iterator<String> itr = getVariables().iterator();
		
		double total = 0;
		while(itr.hasNext()){
			String variable = itr.next();
			
			total = total + getPreviousMibValue(variable);
		}

		return total;
	}
	
	private double getValue(String variable, DataTable table) 
	throws CollectedDataNotFoundException, InvalidValueException{
		CollectorSnmpMstData data = getCollectorSnmpMstData(variable);

		// OIDインデックスの定義によって処理を変える
		if ("?".equals(data.getOidIndex())){
			try {
				// OIDインデックスの定義が"?"の場合
				String oid = data.getTableOid() + "." + getDeviceIndex();
				return getMibValueDouble(table, data, oid);
			} catch (InvalidValueException e){
				throw e;
			} catch (Exception e){
				m_log.debug(e.getMessage(), e);
				// エラー処理
				throw new CollectedDataNotFoundException();
			}
		} else if ("*".equals(data.getOidIndex())){
			try {
				// OIDインデックスの定義が"*"の場合
				String tableOid = data.getTableOid();

				// 直近の取得時のデータテーブルからベースとなるOID配下の値全てを取得
				Set<TableEntry> entrys = table.getValueSetStartWith(tableOid);

				if(entrys == null){
					// エラー処理
					throw new CollectedDataNotFoundException();
				}

				// 合計を求める
				double total = 0;  // 初期値0で初期化
				Iterator<TableEntry> itr = entrys.iterator();
				while(itr.hasNext()){
					String oid = itr.next().getKey();
					total = total + getMibValueDouble(table, data, oid);
					// tableOidの値は2度データテーブルから取得する処理となるため高速化が必要な場合は見直す必要あり
				}
				
				return total;
			} catch (Exception e){
				// エラー処理
				m_log.debug(e.getMessage(), e);
				throw new CollectedDataNotFoundException();
			}
		} else {
			try {
				String oid = data.getTableOid() + "." + data.getOidIndex();
				return getMibValueDouble(table, data, oid);
			} catch (InvalidValueException e){
				throw e;
			} catch (Exception e){
				// エラー処理
				m_log.debug(e.getMessage(), e);
				throw new CollectedDataNotFoundException();
			}
		}
	}
	
	/**
	 * 
	 * mibの値をlongで取得するメソッドです。
	 * mib値の差分を計算をする必要がある場合は、このメソッドを使用して値を取得し、
	 * 差分値を計算した後、convLongToDobleメソッドでdouble値に変換してreturnしてください。
	 * 
	 * @see #convLongToDouble(long)
	 * 
	 * @param table
	 * @param data
	 * @param oid
	 * @return
	 * @throws InvalidValueException
	 */
	private long getMibValueLong(DataTable table, CollectorSnmpMstData data,
			String oid) throws InvalidValueException {
		TableEntry entry = table.getValue(oid);

		// 指定のOIDの値が存在しない場合
		if(entry == null){
			// 値取得失敗と判定し、値取得失敗時の返却値を返す
			return Long.parseLong(data.getFailureValue());
		} else {
			long value = (Long)entry.getValue();
			
//			// 負の値の場合
//			if(value < 0){
//				// 値のタイプがCounter64の型の場合
//				// javaのlong型は符号付であるため、Long.MAX_VALUEを超える正の値を扱うことができない
//				if("Counter64".equals(data.getValueType())){
//					throw new InvalidValueException();
//				}
//			}
		
			return value;
		}
	}
	
	/**
	 * 
	 * mibの値をdoubleで取得するメソッドです。
	 * 値を加算する場合、あるいはそのままの値を利用する場合は、このメソッドで値を取得してください。
	 * 
	 * @param table
	 * @param data
	 * @param oid
	 * @return
	 * @throws InvalidValueException
	 */
	private double getMibValueDouble(DataTable table, CollectorSnmpMstData data,
			String oid) throws InvalidValueException {
		
		long longValue = getMibValueLong(table, data, oid);
		return convLongToDouble(longValue);
		
	}
	
	/**
	 * 
	 * 引数で与えられたlong値を、doubleへ変換します。
	 * 
	 * @param longValue
	 * @return
	 */
	private double convLongToDouble(long longValue) {
		double ret = longValue;
		if (longValue < 0) {
			long tmp = longValue - Long.MAX_VALUE;
			ret = (double)Long.MAX_VALUE + tmp;
		}
		return ret;
	}
	
	/**
	 * 前回収集時からの差分を取得します。
	 * OIDインデックスが"*"の場合は、差分の合計を求めます（合計の差分ではない）。
	 * 
	 * @param variable 変数名
	 * @return 差分値
	 * @throws CollectedDataNotFoundException 
	 * @throws InvalidValueException 
	 */
	@Override
	public double getDifferenceValue(String variable)
	throws CollectedDataNotFoundException, InvalidValueException{
		m_log.debug("getDifferenceValue " + variable);
		
		CollectorSnmpMstData data = getCollectorSnmpMstData(variable);

		DataTable currentTable = getCurrentTable();
		DataTable previousTable = getPreviousTable();
		
		// OIDインデックスの定義によって処理を変える
		if ("?".equals(data.getOidIndex())){
			try {
				// OIDインデックスの定義が"?"の場合
				String oid = data.getTableOid() + "." + getDeviceIndex();
				return getMibValueDiff(data, currentTable, previousTable, oid);
			} catch (InvalidValueException e){
				throw e;
			} catch (Exception e){
				m_log.debug(e.getMessage(), e);
				// エラー処理
				throw new CollectedDataNotFoundException();
			}
		} else if ("*".equals(data.getOidIndex())){
			try {
				// OIDインデックスの定義が"*"の場合
				String tableOid = data.getTableOid();

				// 直近の取得時のデータテーブルからベースとなるOID配下の値全てを取得
				Set<TableEntry> entrys = currentTable.getValueSetStartWith(tableOid);

				if(entrys == null){
					// エラー処理
					throw new CollectedDataNotFoundException();
				}

				// 合計を求める
				double total = 0;  // 初期値0で初期化
				Iterator<TableEntry> itr = entrys.iterator();
				while(itr.hasNext()){
					String oid = itr.next().getKey();
					total = total + getMibValueDiff(data, currentTable, previousTable, oid);
					// tableOidの値は2度データテーブルから取得する処理となるため高速化が必要な場合は見直す必要あり
				}

				return total;
			} catch (InvalidValueException e){
				throw e;
			} catch (Exception e){
				// エラー処理
				m_log.debug(e.getMessage(), e);
				throw new CollectedDataNotFoundException();
			}
		} else {
			try {
				String oid = data.getTableOid() + "." + data.getOidIndex();
				return getMibValueDiff(data, currentTable, previousTable, oid);
			} catch (InvalidValueException e){
				throw e;
			} catch (Exception e){
				// エラー処理
				m_log.debug(e.getMessage(), e);
				throw new CollectedDataNotFoundException();
			}
		}
	}

	@Override
	protected double getDifferenceValueTotal()
	throws CollectedDataNotFoundException, InvalidValueException {
		m_log.debug("getDifferenceValueTotal");
		
		Iterator<String> itr = getVariables().iterator();
		
		double total = 0;
		while(itr.hasNext()){
			String variable = itr.next();
			
			total = total + getDifferenceValue(variable);
		}

		return total;
	}

	
	/**
	 * 直近収集と前回収集のデータテーブルから指定のOIDの値を取得しその差分を求める
	 * @throws InvalidValueException 
	 */
	private double getMibValueDiff(CollectorSnmpMstData data,
			DataTable currentTable, DataTable previousTable, String oid)
	throws InvalidValueException {
		
		// return用の変数
		double ret = 0;
		
		// 前回収集時の値を取得
		long previousValue =  getMibValueLong(previousTable, data, oid);

		// 直近収集時の値を取得
		long currentValue =  getMibValueLong(currentTable, data, oid);

		// 差分をlongで計算する
		long diff = currentValue - previousValue;
		
		
		// 差分が正の値の場合
		if(diff >= 0){
			ret = diff;
			
		// 差分が負の値の場合はdouble値へ変換する
		} else {
			
			// 値のタイプがCounter32の型の場合は上限を越えループした場合の処理
			if("Counter32".equals(data.getValueType())) {
				diff = COUNTER32_MAX_VALUE + 1 + diff;
				ret = convLongToDouble(diff);
				
			// 値のタイプがCounter64の型の場合は上限を越えループした場合の処理				
			} else if ("Counter64".equals(data.getValueType())) {
				ret = COUTNER64_MAX_VALUE.add(BigInteger.valueOf(diff + 1)).doubleValue();
				
			}
		}
		
		return ret;
	}

	/**
	 * キャッシュからOID定義を取得
	 * 
	 * CollectorSnmpMstData には下記が含まれる
	 * ・プラットフォーム
	 * ・収集項目コード
	 * ・変数名
	 * ・OIDベース
	 * ・OIDインデックス
	 * ・値の型
	 * ・ポーリング対象OID
	 * ・取得失敗時の値
	 */
	private CollectorSnmpMstData getCollectorSnmpMstData(String variable) {
		CollectorSnmpMstData data = m_oidMap.get(variable);

		// 登録されていない場合は登録
		if (data == null){
			registerOidMap(variable);

			data = m_oidMap.get(variable);

			// 登録できなかった場合
			if(data == null){
				// エラー処理
				throw new IllegalStateException();
			}
		}
		return data;
	}
	
	private void registerOidMap(String variable){
		try {
			m_log.debug(getPlatformId() + ", " +  getItemCode() + ", " + variable);
			
			CollectorSnmpMstLocal bean = CollectorSnmpMstUtil.getLocalHome()
			.findByPrimaryKey(new CollectorSnmpMstPK(getPlatformId(), getItemCode(), variable));

			CollectorSnmpMstData data = new CollectorSnmpMstData(
					bean.getPlatformId(), 
					bean.getItemCode(), 
					bean.getVariableId(),
					bean.getTableOid(), 
					bean.getOidIndex(),
					bean.getValueType(), 
					bean.getPollingOid(), 
					bean.getFailureValue());
			
			m_oidMap.put(variable, data);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
	}
}
