/*
 
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.rrdtool;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.rmi.RemoteException;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.ejb.session.CollectorController;
import com.clustercontrol.performance.ejb.session.CollectorControllerHome;
import com.clustercontrol.performance.rrdtool.csv.CSVColumn;
import com.clustercontrol.performance.rrdtool.util.CollectedDataInfoDateComparator;
import com.clustercontrol.performance.rrdtool.util.Config;
import com.clustercontrol.performance.rrdtool.util.LoginManager;

/**
 * Hinemos Addon for RRDTool<br>
 * CSVファイル出力処理 実行クラス<br>
 * 
 * @since 3.0.0
 */
public class PerformanceCSVExport {

	// main関数に与えられる引数の格納変数
	private static String collectorID = null;
	private static String facilityID = null;
	private static String listFilePath = null;
	private static String outputFilePath = null;
	private static String startDateEpochStr = null;
	private static String endDateEpochStr = null;

	// main関数に与えられる引数を変換した変数
	private static long startDateEpoch = 0;
	private static long endDateEpoch = 0;

	// 実行ログ出力用クラス
	private static Log log = LogFactory.getLog(PerformanceCSVExport.class);

	/**
	 * CSV出力実行関数<br>
	 * 
	 * @param args
	 */
	public static void main(String args[]) {

		// 出力項目定義格納変数
		ArrayList<CSVColumn> csvCols = new ArrayList<CSVColumn>();

		// 収集設定変数
		CollectorProperty property = null;

		// 不正出力項目項目存在フラグ
		boolean invalidColFlg = false;

		// 収集性能値の格納変数
		ArrayList<CollectedDataInfo> collectedDataInfos = null;

		// 出力項目の性能値存在フラグ
		HashMap<CSVColumn, Boolean> csvColFlg = null;

		// マネージャとのセッション情報
		NamingContext m_ctx = null;
		CollectorController collector = null;

		// 実行ログを記載
		log.info(Config.getMsg("RRDTool.ExportCSV.Exec") + " : " + Arrays.toString(args));

		// 引数の数の確認
		if (args.length != 6) {
			log.error(Config.getMsg("RRDTool.ExportCSV.ArgsInvalid"));
			System.exit(11);
		}

		// 引数の格納
		collectorID = args[0];
		facilityID = args[1];
		listFilePath = args[2];
		startDateEpochStr = args[3];
		endDateEpochStr = args[4];
		outputFilePath = args[5];

		// エクスポート期間の確認
		try {
			startDateEpoch = Long.decode(startDateEpochStr);
			endDateEpoch = Long.decode(endDateEpochStr);

			if (startDateEpoch <= 0 || endDateEpoch <= 0 || endDateEpoch < startDateEpoch) {
				// 検索日時が正数（0より大きい）でない、もしくは検索終了日時が検索開始日時より前である場合
				log.error(Config.getMsg("RRDTool.ExportCSV.DateFormatInvalid"));
				System.exit(52);
			}

			// ミリ秒へ変換
			startDateEpoch *= 1000;
			endDateEpoch *= 1000;
		} catch (NumberFormatException e) {
			// 検索日時が整数でない場合（Long型として読み込めない場合）
			log.error(Config.getMsg("RRDTool.ExportCSV.DateFormatInvalid"), e);
			System.exit(52);
		}

		// 出力項目定義ファイルの読み込み
		csvCols = readColumnDefFile(listFilePath);

		// マネージャーへの接続
		try {
			m_ctx = LoginManager.getContextManager().getNamingContext(Config.getConfig("Login.USER"),
					Config.getConfig("Login.PASSWORD"), Config.getConfig("Login.URL"));
			collector = (CollectorController) ((CollectorControllerHome) m_ctx.lookup(CollectorControllerHome.JNDI_NAME))
					.create();
		} catch (Exception e) {
			log.error(Config.getMsg("RRDTool.ExportCSV.ConnectManagerFailed"), e);
			System.exit(14);
		}

		try {
			// 収集プロパティの取得を試み、取得できない場合はエラー処理を実施する
			property = collector.getCollectorProperty(collectorID);

			// 指定の収集IDの収集が存在しない場合は、nullが返る
			if (property == null) {
				log.error(Config.getMsg("RRDTool.ExportCSV.CollectorIDNotFound"));
				System.exit(53);
			}

		} catch (RemoteException e) {
			log.error(Config.getMsg("RRDTool.ExportCSV.ConnectManagerFailed"), e);
			System.exit(14);
		}

		// ファシリティIDの妥当性確認
		if (!checkFacilityDefinition(property, facilityID)) {
			log.error(Config.getMsg("RRDTool.ExportCSV.FacilityIDNotFound"));
			System.exit(54);
		}

		// 出力項目の妥当性確認
		invalidColFlg = !checkCollectorDefinition(property, csvCols);

		// 収集性能値の格納
		try {
			collectedDataInfos = (ArrayList<CollectedDataInfo>) collector.getRecordCollectedData(collectorID, facilityID,
					new Date(startDateEpoch), new Date(endDateEpoch));
		} catch (RemoteException e) {
			log.error(Config.getMsg("RRDTool.ExportCSV.ConnectManagerFailed"), e);
			System.exit(14);
		}

		// 不要な性能値の除去
		csvColFlg = examineCollectedDataInfo(collectedDataInfos, csvCols);

		// 出力項目の 収集性能値が1つも存在しない場合
		if (collectedDataInfos.size() == 0) {
			log.error(Config.getMsg("RRDTool.ExportCSV.ColValueNotFound"));
			System.exit(55);
		}

		// 性能値の日時ソート
		Collections.sort(collectedDataInfos, new CollectedDataInfoDateComparator());

		// CSVファイルの出力
		writeCSVFile(outputFilePath, csvCols, collectedDataInfos);
		
		// マネージャー接続の切断
		try {
			LoginManager.getContextManager().logout();
		} catch (NamingException e) {
			log.warn(Config.getMsg("RRDTool.ExportCSV.DisconnectManagerFailed"), e);
		}
		
		if (invalidColFlg) {
			// 不正な出力項目定義がある場合
			System.exit(101);
		} else if (csvColFlg.containsValue(false)) {
			// 性能値のない出力項目がある場合
			System.exit(102);
		} else {
			// 正常終了
			log.info(Config.getMsg("RRDTool.ExportCSV.Exit"));
			System.exit(0);
		}
	}

	/**
	 * 出力項目定義ファイルを読み込んで、出力項目情報を取得する。
	 * 
	 * @param fileName
	 *            出力項目定義ファイルのファイルパス（絶対パス、相対パスともに可）
	 * @return 出力項目情報の配列（出力項目順に格納されている）
	 */
	public static ArrayList<CSVColumn> readColumnDefFile(String fileName) {

		// 返り値
		ArrayList<CSVColumn> ret = new ArrayList<CSVColumn>();

		// ファイル読み込み用変数
		BufferedReader br = null;
		String line = null;
		String colName = null;
		String itemCode = null;
		String deviceName = null;

		// 出力項目定義のフォーマット確認用変数
		StringTokenizer defTokenizer = null;
		String tokenStr = "\t";
		Pattern colNamePattern = Pattern.compile("^[^\t\",]+$");
		boolean invalidDefFormat = false;

		try {
			// 出力項目定義ファイルを開く
			br = new BufferedReader(new FileReader(fileName));

			while ((line = br.readLine()) != null) {

				// カラム数の確認
				defTokenizer = new StringTokenizer(line, tokenStr, false);
				if (defTokenizer.countTokens() == 2) {
					// 出力項目名<tab>項目コードの場合
					colName = defTokenizer.nextToken();
					itemCode = defTokenizer.nextToken();
					deviceName = "";
				} else if (defTokenizer.countTokens() == 3) {
					// 出力項目名<tab>項目コード<tab>デバイス名の場合
					colName = defTokenizer.nextToken();
					itemCode = defTokenizer.nextToken();
					deviceName = defTokenizer.nextToken();
				} else {
					// 不正な書式の行の場合
					colName = null;
					itemCode = null;
					deviceName = null;
					log.error(Config.getMsg("RRDTool.ExportCSV.LineFormatInvalid") + " : " + line);
					invalidDefFormat = true;
				}

				// 出力項目名の書式確認
				if (colName != null && !colNamePattern.matcher(colName).matches()) {
					// 出力項目名に不正な場合
					log.error(Config.getMsg("RRDTool.ExportCSV.ColNameFormatInvalid") + " : " + colName);
					invalidDefFormat = true;
				}

				// 出力カラム定義の格納
				if (!invalidDefFormat) {
					ret.add(new CSVColumn(colName, itemCode, deviceName));
				}

			}

		} catch (Exception e) {
			// ファイルの読み込み処理に失敗した場合
			log.error(Config.getMsg("RRDTool.ExportCSV.ReadFileFailed"), e);
			System.exit(12);
		} finally {
			try {
				// 出力項目定義ファイルを閉じる
				if (br != null) {
					br.close();
				}
			} catch (IOException e) {
				// ファイルの読み込み処理に失敗した場合
				log.error(Config.getMsg("RRDTool.ExportCSV.ReadFileFailed"), e);
				System.exit(12);
			}
			if (invalidDefFormat) {
				// 不正な項目定義が存在した場合
				System.exit(13);
			}
		}

		return ret;
	}

	/**
	 * 性能収集定義に該当する収集対象（収集ID, デバイス名）が存在するかどうかを確認する。
	 * 
	 * @param property
	 *            性能収集定義オブジェクト
	 * @param csvCols
	 *            出力項目情報の配列（出力項目順に格納されている）
	 * @return 不正な出力項目情報がひとつでも存在する場合false, それ以外はtrue
	 */
	public static boolean checkCollectorDefinition(CollectorProperty property, ArrayList<CSVColumn> csvCols) {

		boolean ret = true;
		boolean invalidColumn = false;

		// 出力項目が性能収集定義に含まれるかどうかを確認する
		for (CSVColumn csvCol : csvCols) {
			invalidColumn = true;
			for (CollectorItemInfo collectorItemInfo : (ArrayList<CollectorItemInfo>) property.getItemList()) {
				if (csvCol.equalsCollection(collectorItemInfo.getItemCode(), collectorItemInfo.getDeviceName())) {
					invalidColumn = false;
					break;
				}
			}
			if (invalidColumn) {
				// 性能収集定義に含まれない出力項目の場合
				log.warn(Config.getMsg("RRDTool.ExportCSV.ColDefinitionInvalid") + " : " + csvCol.getItemCode() + ", "
						+ csvCol.getDeviceName());
				ret = false;
			}
		}

		return ret;
	}

	/**
	 * 性能収集定義に該当するファシリティIDが存在するかどうかを確認する。
	 * 
	 * @param property
	 *            性能収集定義オブジェクト
	 * @param facilityId
	 *            出力対象ノードのファシリティID
	 * @return ファシリティIDが存在しない場合false, それ以外はtrue
	 */
	public static boolean checkFacilityDefinition(CollectorProperty property, String facilityId) {
		// 収集対象スコープ配下のスコープツリーを取得
		FacilityTreeItem tree = property.getCollectorData().getFacilityTree();

		// トップのファシリティから順番に子要素を再帰的に調べる
		return checkChildFacility(tree, facilityId);
	}

	/**
	 * 指定ファシリティツリーの配下に該当のファシリティIDの要素が存在するか否かを再帰的に調べる。
	 * 
	 * @param treeItem
	 *            ファシリティツリー情報
	 * @param facilityId
	 *            ファシリティID
	 * @return
	 */
	private static boolean checkChildFacility(FacilityTreeItem treeItem, String facilityId) {
		// 一致するものが見つかった場合は、trueを返す
		if (treeItem.getData().getFacilityId().equals(facilityId)) {
			return true;
		}

		for (int i = 0; i < treeItem.getChildren().length; i++) {
			FacilityTreeItem child = treeItem.getChildren()[i];

			// 子要素を再帰的に捜査し、見つかった場合にはtrueを返す
			// falseの場合は継続して捜査を実行
			if (checkChildFacility(child, facilityId) == true) {
				return true;
			}
		}

		// 一致するものが見つからなかった場合はnullを返す
		return false;
	}

	/**
	 * 出力項目定義に存在しない性能情報を配列から取り除き、各出力項目における性能情報が存在するかどうかを定義した連想配列を返す。
	 * 
	 * @param collectedDataInfos
	 *            収集性能値の格納配列
	 * @param csvCols
	 *            出力項目定義オブジェクト
	 * @return 出力項目オブジェクトをキーとして、収集性能値が1つ以上存在すればtrue, 存在しなければfalseを値とする連想配列
	 */
	public static HashMap<CSVColumn, Boolean> examineCollectedDataInfo(ArrayList<CollectedDataInfo> collectedDataInfos,
			ArrayList<CSVColumn> csvCols) {

		boolean valueExistFlg = false;
		CollectedDataInfo collectedDataInfo = null;
		HashMap<CSVColumn, Boolean> ret = new HashMap<CSVColumn, Boolean>();

		// すべての出力項目の収集性能値存在フラグをfalseとする
		for (CSVColumn csvCol : csvCols) {
			ret.put(csvCol, false);
		}

		// すべての収集性能値オブジェクトが収集項目に含まれるかどうかを確認する
		for (int i = 0; i < collectedDataInfos.size(); i++) {
			collectedDataInfo = collectedDataInfos.get(i);
			valueExistFlg = false;
			for (CSVColumn csvCol : csvCols) {
				if (csvCol.equalsCollection(collectedDataInfo.getItemCode(), collectedDataInfo.getDeviceName())) {
					valueExistFlg = true;
					ret.put(csvCol, true);
				}
			}
			// 出力項目に含まれない収集性能値オブジェクトを取り除く
			if (!valueExistFlg) {
				collectedDataInfos.remove(i);
				i--;
			}
		}

		return ret;
	}

	/**
	 * 収集性能値をCSVファイルに出力する。<br>
	 * <br> - 出力例<br>
	 * "Date","Time","CPU_ALL","SWAP_ALL","PCK_RCV eth0"[LF]<br>
	 * "08/05/30","0:00",0.64,0.00,22.27,11.07[LF]<br>
	 * "08/05/30","0:30",1.00,0.00,26.40,12.93[LF]<br>
	 * <br>
	 * ただし、"Date", "Time"は性能収集の予定日時ではなく、実際に収集した日時である。<br>
	 * また、何らかの原因により収集遅延が発生し、1分間の中で（yyyy-MM-dd HH:mm）に<br>
	 * 同一キー（collector_id, item_code,device_name)に対する複数の性能値が存在する場合は<br>
	 * 最も古い日時の性能値を出力内容とする。（古い日時の性能値のほうが計測時間が長く、信頼性が高いため）
	 * 
	 * @param filePath
	 *            CSVファイルのファイルパス（絶対パス、相対パスともに可）
	 * @param csvCols
	 *            出力項目定義オブジェクト
	 * @param collectedDataInfos
	 *            収集性能値の格納配列（ただし、日時でソートされており、出力項目以外の収集性能値が存在しないこと。また、性能値がnullでないこと。）
	 */
	public static void writeCSVFile(String filePath, ArrayList<CSVColumn> csvCols,
			ArrayList<CollectedDataInfo> collectedDataInfos) {

		// ファイル書き込み用変数
		FileOutputStream fos = null;
		BufferedOutputStream bos = null;
		FileChannel channel = null;
		StringBuffer line = new StringBuffer();

		// 日付書式用変数
		SimpleDateFormat dateformat = new SimpleDateFormat("yy/MM/dd");
		SimpleDateFormat timeformat = new SimpleDateFormat("H:mm");
		SimpleDateFormat hashKeyformat = new SimpleDateFormat("yyyy/MM/dd HH:mm");

		// 収集性能値書式用変数
		DecimalFormat valueformat = new DecimalFormat("#.00");

		// 日時文字列（yyyy-MM-dd hh:mm）をキーとした収集性能値の連想配列
		HashMap<String, ArrayList<CollectedDataInfo>> dataHash = new HashMap<String, ArrayList<CollectedDataInfo>>();
		String hashKey = null;
		ArrayList<String> hashKeys = new ArrayList<String>();

		// 連想配列への収集性能値の格納
		for (CollectedDataInfo collectedDataInfo : collectedDataInfos) {
			hashKey = hashKeyformat.format(collectedDataInfo.getDate());
			if (!dataHash.containsKey(hashKey)) {
				dataHash.put(hashKey, new ArrayList<CollectedDataInfo>());
			}
			dataHash.get(hashKey).add(collectedDataInfo);
			if (!hashKeys.contains(hashKey)) {
				hashKeys.add(hashKey);
			}
		}

		// CSVファイルへの出力
		try {
			// 出力ファイルを開く
			fos = new FileOutputStream(filePath);

			// ファイルの排他制御（排他ロックを取得）
			channel = fos.getChannel();
			if (channel.tryLock() == null) {
				// ロックに失敗した場合
				log.error(Config.getMsg("RRDTool.ExportCSV.LockFileFailed"));
				System.exit(51);
			}
			bos = new BufferedOutputStream(fos);

			// ヘッダー行の出力
			line.delete(0, line.length());
			line.append("\"Date\",\"Time\",");
			for (int i = 0; i < csvCols.size(); i++) {
				line.append("\"" + csvCols.get(i).getColumnName() + "\"");
				if (i < csvCols.size() - 1) {
					line.append(",");
				}
			}
			// 最終行以外では末尾にLFを付与する
			if (hashKeys.size() != 0) {
				line.append("\n");
			}
			bos.write(line.toString().getBytes());

			// 性能値の出力
			for (String datekey : hashKeys) {
				line.delete(0, line.length());
				line.append("\"" + dateformat.format(dataHash.get(datekey).get(0).getDate()) + "\",");
				line.append("\"" + timeformat.format(dataHash.get(datekey).get(0).getDate()) + "\",");
				// 出力項目ごとに出力
				for (int i = 0; i < csvCols.size(); i++) {
					// 該当日時文字列キーに性能値が存在する場合、出力
					for (CollectedDataInfo collectedDataInfo : dataHash.get(datekey)) {
						if (csvCols.get(i).equalsCollection(collectedDataInfo.getItemCode(),
								collectedDataInfo.getDeviceName())) {
							line.append(valueformat.format(collectedDataInfo.getValue()));
							break;
						}
					}
					if (i < csvCols.size() - 1) {
						line.append(",");
					}
				}
				line.append("\n");
				bos.write(line.toString().getBytes());
			}

		} catch (Exception e) {
			// ファイルの書き込み処理に失敗した場合
			log.error(Config.getMsg("RRDTool.ExportCSV.WriteFileFailed"), e);
			System.exit(51);
		} finally {
			try {
				// 出力ファイルに書き出し内容を反映して閉じる
				if (bos != null) {
					bos.flush();
					bos.close();
				}
			} catch (IOException e) {
				// ファイルの書き込み処理に失敗した場合
				log.error(Config.getMsg("RRDTool.ExportCSV.WriteFileFailed"), e);
				System.exit(51);
			}
		}

		return;
	}

}
