/*
 
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.rrdtool.util;

import java.lang.reflect.Method;
import java.rmi.AccessException;
import java.util.HashMap;
import java.util.Hashtable;

import javax.naming.CommunicationException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.naming.NamingContextFactory;
import org.jboss.security.auth.callback.UsernamePasswordHandler;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.accesscontrol.ejb.session.AccessCheck;
import com.clustercontrol.accesscontrol.ejb.session.AccessCheckHome;
import com.clustercontrol.util.Messages;

/**
 * Hinemos Addon for RRDTool<br>
 * ログインマネージャクラス<br>
 * 
 * @since 3.0.0
 */
public class LoginManager {
	public static final String KEY_EJB_URL = "ejbUrl";
	public static final String VALUE_EJB_URL = "jnp://localhost:1099";
	public static final String KEY_EJB_UID = "ejbUid";
	public static final String VALUE_EJB_UID = "hinemos";

	public static final String USER_OBJECT_ID = "com.clustercontrol.accesscontrol.user";
	public static final String URL_OBJECT_ID = "com.clustercontrol.accesscontrol.url";

	public static final String ACTION_SET_ID = "com.clustercontrol.accesscontrol.ActionSet";
	public static final String ACTION_ID_LOGIN = "com.clustercontrol.accesscontrol.etc.action.LoginAction";
	public static final String ACTION_ID_LOGOUT = "com.clustercontrol.accesscontrol.etc.action.LogoutAction";

	private static LoginManager m_instance = null;
	private NamingContext m_namingcontext = null;
	private LoginContext m_loginContext = null;
	private String m_url = null;
	private String m_uid = null;
	private String m_password = null;

	// 実行ログ出力用クラス
	private static Log log = LogFactory.getLog(LoginManager.class);

	/**
	 * このオブジェクトを取得します。 シングルトン
	 * 
	 * @return LoginManager ログインマネージャ
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public static LoginManager getContextManager() {
		if (m_instance == null) {
			m_instance = new LoginManager();
		}
		return m_instance;
	}

	/**
	 * コンストラクタ
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private LoginManager() {
		try {
			Configuration.setConfiguration(new HinemosClientConfig());
		} catch (Exception e) {
		}
	}

	/**
	 * ログイン(接続先URLあり)
	 * 
	 * @param uid
	 * @param password
	 * @throws NamingException
	 * @throws LoginException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized void login(String uid, String password, String url) throws Exception {
		// ログアウト
		logout();

		m_uid = uid;
		m_password = password;

		// NamingContext取得
		m_namingcontext = getContext(url);

		try {
			// ログインチェック
			AccessCheckHome home = (AccessCheckHome) m_namingcontext.lookup(AccessCheckHome.JNDI_NAME);
			AccessCheck accessCheck = home.create();
			accessCheck.checkLogin();

		} catch (Exception e) {
			logout();
			throw e;
		}
	}

	/**
	 * ログアウト
	 * 
	 * @throws NamingException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized void logout() throws NamingException {
		if (m_loginContext instanceof LoginContext) {
			try {
				m_loginContext.logout();
				m_uid = null;

			} catch (LoginException e) {
			}
		}
		m_loginContext = null;
		m_namingcontext = null;
	}

	/**
	 * ログインチェック
	 * 
	 * @return
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public boolean isLogin() {
		if (m_loginContext instanceof LoginContext) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * RepositoryCollectorを取得します。
	 * 
	 * @return RepositoryCollector
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized NamingContext getNamingContext(String uid, String pw, String url) throws NamingException {
		if (!isLogin()) {

			try {
				// login(dialog.getUserid(), dialog.getPassword());
				login(uid, pw, url);

			} catch (CommunicationException e) {
				// 接続失敗メッセージを出力
				log.error(Messages.getString("message.accesscontrol.22"), e);
			} catch (AccessException e) {
				// ログイン失敗メッセージを出力
				log.error(Messages.getString("message.accesscontrol.6"), e);
			} catch (Exception e) {
				// 予期せぬメッセージを出力
				log.error(Messages.getString("message.accesscontrol.23"), e);
				log.error(Messages.getString("message.accesscontrol.6"));
			}

		}
		return m_namingcontext;
	}

	/**
	 * NamingContextを取得します。
	 * 
	 * @return
	 * @throws NamingException
	 * @throws LoginException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	@SuppressWarnings("unchecked")
	private NamingContext getContext() throws NamingException, LoginException {
		// 接続先URL取得
		String url = getUrl();

		return getContext(url);
	}

	/**
	 * NamingContextを取得します。
	 * 
	 * @return
	 * @throws NamingException
	 * @throws LoginException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	@SuppressWarnings("unchecked")
	private NamingContext getContext(String url) throws NamingException, LoginException {
		// 接続先URL取得
		m_url = url;

		Hashtable props = new Hashtable();
		props.put(InitialContext.PROVIDER_URL, m_url);

		UsernamePasswordHandler handler = new UsernamePasswordHandler(m_uid, m_password.toCharArray());
		m_loginContext = new LoginContext("hinemosClient", handler);
		m_loginContext.login();

		NamingContextFactory ncf = new NamingContextFactory();
		NamingContext namingContext = (NamingContext) ncf.getInitialContext(props);

		return namingContext;
	}

	/**
	 * 接続先URLを取得します。
	 * 
	 * @return String 接続先URL
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private String getUrl() {

		// 設定ファイルから接続先URLを取得
		String url = Config.getConfig("Login.URL");

		return url;
	}

	/**
	 * HinemosClient用のコンフィギュレーションクラス
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	class HinemosClientConfig extends Configuration {
		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.security.auth.login.Configuration#refresh()
		 */
		public void refresh() {

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
		 */
		public AppConfigurationEntry[] getAppConfigurationEntry(String name) {
			AppConfigurationEntry[] entry = null;
			try {
				Class[] parameterTypes = {};
				Method m = getClass().getDeclaredMethod(name, parameterTypes);
				Object[] args = {};
				entry = (AppConfigurationEntry[]) m.invoke(this, args);
			} catch (Exception e) {
			}
			return entry;
		}

		/**
		 * Hinemos Client Configuration
		 * 
		 * @return
		 */
		@SuppressWarnings("unchecked")
		AppConfigurationEntry[] hinemosClient() {
			String name = "org.jboss.security.ClientLoginModule";
			HashMap options = new HashMap();
			options.put("multi-threaded", "true");
			options.put("restore-login-identity", "true");
			AppConfigurationEntry ace = new AppConfigurationEntry(name,
					AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
			AppConfigurationEntry[] entry = { ace };
			return entry;
		}
	}
}