/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を作成するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class AddNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( AddNotify.class );
	
	/**
	 * 通知情報を作成します。
	 * <p>
	 * <ol>
	 *  <li>通知情報を作成します。</li>
	 *  <li>通知イベント情報を作成し、通知情報に設定します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュに追加されます。</li>
	 * </ol>
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext#NotifyInfoContext(int, java.lang.String, com.clustercontrol.notify.bean.NotifyInfo)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	@SuppressWarnings("unchecked")
	public boolean add(NotifyInfo info , String user ) throws CreateException, NamingException, JMSException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.NOTIFY);
    	
		SendQueue queue = null;
		try 
		{
			Timestamp now = new Timestamp(new Date().getTime());
			
			// 通知情報を挿入
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().create(
					info.getNotifyId(),
					info.getDescription(),
				    info.getNotifyType(),
					info.getInhibitionFlg(),
					info.getInhibitionFrequency(),
					info.getInhibitionPeriod(),
					now,
					now,
					user,
					user,
					info.getValidFlg()
			);
			
			addDetails(info, notify);
			
			NotifyInfoContext context = new NotifyInfoContext(NotifyInfoContext.TYPE_ADD, info.getNotifyId(), info);
			
			// 通知情報キャッシュを更新
			queue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFYCONTROL);
			queue.put(context);
			queue.terminate();
			
		} catch (CreateException e) {
            AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		} catch (JMSException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { info.getNotifyId() };
            apllog.put("SYS", "001", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		}
		finally{
			if(queue != null){
				queue.terminate();	
			}
		}
		
		return true;
	}
	
	/**
	 * 通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	public boolean addDetails(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException{
		
		Collection ct = null;
		ArrayList localList = null ;
		
		/***/
		ArrayList<NotifyInfoDetail> eventList =info.getNotifyInfoDetail();
		
		if(info.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_STATUS ){
			
			ct = notify.getNotifyStatusInfo();
			
			 localList = new ArrayList<NotifyStatusInfoLocal>();
				
				// 通知イベント情報を挿入
				NotifyStatusInfo status = null;
				for(int index=0; index<eventList.size(); index++){
					status = (NotifyStatusInfo)eventList.get(index);
					if(status != null){
						NotifyStatusInfoLocal local = NotifyStatusInfoUtil.getLocalHome().create(
								info.getNotifyId(),
								status.getPriority(),
								status.getValidFlg(),
								status.getInhibitionFlg(),
								status.getStatusValidPeriod(),
								status.getStatusInvalidFlg(),
								status.getStatusUpdatePriority()
								);
						localList.add(local);
					}
				}

				if(localList != null && ct !=null){
					ct.addAll(localList);
					notify.setNotifyStatusInfo(ct);
				}
		}else if(info.getNotifyType() == com.clustercontrol.bean.NotifyTypeConstant.TYPE_EVENT){
		
			ct = notify.getNotifyEventInfo();
		
			 localList = new ArrayList<NotifyEventInfoLocal>();
			
			// 通知イベント情報を挿入
			NotifyEventInfo event = null;
			for(int index=0; index<eventList.size(); index++){
				event = (NotifyEventInfo)eventList.get(index);
				if(event != null){
					NotifyEventInfoLocal local = NotifyEventInfoUtil.getLocalHome().create(
							info.getNotifyId(),
							event.getPriority(),
							event.getValidFlg(),
							event.getEventNormalState(),
							event.getInhibitionFlg(),
							event.getEventInhibitionState()
							);
					localList.add(local);
				}
			}

			if(localList != null && ct !=null){
				ct.addAll(localList);
				notify.setNotifyEventInfo(ct);
			}
			
		}else if(info.getNotifyType() == com.clustercontrol.bean.NotifyTypeConstant.TYPE_MAIL ){
			ct = notify.getNotifyMailInfo();
			
			 localList = new ArrayList<NotifyMailInfoLocal>();
				
				// 通知イベント情報を挿入
				NotifyMailInfo mail = null;
				for(int index=0; index<eventList.size(); index++){
					 mail  = (NotifyMailInfo)eventList.get(index);
					if( mail != null){
						NotifyMailInfoLocal local = NotifyMailInfoUtil.getLocalHome().create(
								 info.getNotifyId(),
								 mail.getPriority(),
								 mail.getMailTemplateId(),
								 mail.getValidFlg(),
								 mail.getInhibitionFlg(),
								 mail.getMailAddress()
								);
						localList.add(local);
					}
				}

				if(localList != null && ct !=null){
					ct.addAll(localList);
					notify.setNotifyMailInfo(ct);
				}
				
		}else if(info.getNotifyType() == com.clustercontrol.bean.NotifyTypeConstant.TYPE_JOB ){
			
			ct = notify.getNotifyJobInfo();
			
			 localList = new ArrayList<NotifyJobInfoLocal>();
				
				// 通知イベント情報を挿入
				NotifyJobInfo job = null;
				for(int index=0; index<eventList.size(); index++){
					job = (NotifyJobInfo)eventList.get(index);
					if(job != null){
						NotifyJobInfoLocal local = NotifyJobInfoUtil.getLocalHome().create(
								info.getNotifyId(),
								job.getPriority(),
								job.getValidFlg(),
								job.getJobId(),
								job.getInhibitionFlg(),
								job.getJobFailurePriority(),
								job.getJobExecFacilityFlg(),
								job.getJobExecFacility()
								);
						localList.add(local);
					}
				}
				

				if(localList != null && ct !=null){
					ct.addAll(localList);
					notify.setNotifyJobInfo(ct);
				}
				
		}else if(info.getNotifyType() == com.clustercontrol.bean.NotifyTypeConstant.TYPE_LOG_ESCALATE){
			ct = notify.getNotifyLogEscalateInfo();	
			
			
			 localList = new ArrayList<NotifyLogEscalateInfoLocal>();
				
				// 通知イベント情報を挿入
				NotifyLogEscalateInfo log = null;
				for(int index=0; index<eventList.size(); index++){
					log = (NotifyLogEscalateInfo)eventList.get(index);
					if(log != null){
						NotifyLogEscalateInfoLocal local = NotifyLogEscalateInfoUtil.getLocalHome().create(
								info.getNotifyId(),
								log.getPriority(),
								log.getValidFlg(),
								log.getInhibitionFlg(),
								log.getEscalateMessage(),
								log.getSyslogPriority(),
								log.getSyslogFacility(),
								log.getEscalateFacilityFlg(),
								log.getEscalateFacility(),
								log.getEscalatePort()
								);
						localList.add(local);
					}
				}
				

				if(localList != null && ct !=null){
					ct.addAll(localList);
					notify.setNotifyLogEscalateInfo(ct);
				}
		}
		return true;
	}
	
}
