/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.ejb.bmp;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.EntityBean;
import javax.ejb.EntityContext;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;

import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PollerProtocolConstant;
import com.clustercontrol.performance.bean.EventCode;
import com.clustercontrol.performance.bean.QuartzConstant;
import com.clustercontrol.performance.bean.ScopeTree;
import com.clustercontrol.performance.job.CollectJob;
import com.clustercontrol.performance.monitor.factory.RunMonitorPerformance2;
import com.clustercontrol.performance.util.FacilityTreeDB;
import com.clustercontrol.performance.util.JNDIConnectionManager;
import com.clustercontrol.performance.util.Repository;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorRunStatus;
import com.clustercontrol.performance.bean.CollectorType;
import com.clustercontrol.performance.dao.CalculatedDataDAO;
import com.clustercontrol.performance.dao.CollectorItemDAO;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;
import com.clustercontrol.performance.ejb.bmp.CollectorStatusLocal;
import com.clustercontrol.performance.ejb.bmp.CollectorStatusPK;
import com.clustercontrol.performance.ejb.bmp.CollectorStatusUtil;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeRelationLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeRelationUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapUtil;
import com.clustercontrol.performance.monitor.ejb.entity.MonitorPerfInfoLocal;
import com.clustercontrol.performance.monitor.ejb.entity.MonitorPerfInfoUtil;
import com.clustercontrol.poller.CollectorAlreadyExistException;
import com.clustercontrol.poller.FacilityNotFoundException;
import com.clustercontrol.poller.NotInitializedException;
import com.clustercontrol.poller.PollerManager;
import com.clustercontrol.poller.PollingController;
import com.clustercontrol.poller.PollingInterval;
import com.clustercontrol.sharedtable.DataTableNotFoundException;
import com.clustercontrol.sharedtable.SharedTable;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 収集データを保持するEntityBeanクラス
 * （収集設定で指定されたスコープ配下のノード郡の収集された性能値も保持する。）
 *
 * @version 2.0.0
 * @since 2.0.0  
 * 
 * @ejb.bean name="Collector"
 *           display-name="Name for Collector"
 *           description="Description for Collector"
 *           jndi-name="Collector"
 *           type="BMP"
 *           view-type="both"
 * 
 * @jboss.depends name="jboss.jca:service=DataSourceBinding,name=HinemosDS"
 * 
 * @ejb.permission
 *     unchecked="true"
 * 
 * @ejb.data-object name="RecordCollectorData"
 *           
 * @ejb.dao
 *  class="com.clustercontrol.performance.ejb.bmp.CollectorDAO" 
 *  impl-class="com.clustercontrol.performance.dao.CollectorDAOImpl"
 *  
 * @jboss.container-configuration name="Collector Configuration"
 */
public class CollectorBean implements EntityBean {
	// ログ出力用オブジェクト
	protected static Log m_log = LogFactory.getLog(CollectorBean.class);
	
	public final static String SHARED_TABLE_JNDI_NAME = "SharedTable";
	public final static String POLLER_MANAGER_JNDI_NAME = "PollerManager";
	
	// 計算済み性能値を格納するDBへのアクセスオブジェクト
//	private CalculatedDataDAO m_dao;
	
	// 寿命の最大値(この値を越える回数参照がない場合は停止すべきと判断される)
	private final int MAX_LIFECOUNT = 2;  
	private int lifetimeCounter;    // 収集が終了するまでのカウンタ
	// 停止は外から remove()メソッドを呼び出すことで行う。
	// Bean が自分自身で停止することはない。
	
	private int collectorType;      // 収集種別(リアルタイム/実績/閾値監視)
	
	// 永続化する変数
	private String collectorID = null; // 収集ID
	private int intervalSec;              // 収集間隔
	private String facilityID;         // 収集対象スコープ
	private ScopeTree scopeTree;       // 性能値算出用のスコープツリー情報
	private CollectorItemInfo[] items; // 収集項目のリスト（収集項目数分確保）
	
	private String label;
	private long scheduledStartTime;   // 収集開始予定時刻
	private long scheduledStopTime;    // 収集終了予定時刻
	private long startTime;            // 収集開始時刻
	private long stopTime;             // 収集終了時刻
	private String scopePath;
	private int periodMin;
	private int presavePeriod;
	
//	private OIDInfo[] targetOID;       // 収集対象のOIDの情報
	
	protected EntityContext entityCtx;  // Entyty Context
	
	public CollectorBean() {
		super();
	}

	/**
	 * EntityBeanを生成する際にコンテナから呼ばれます。
	 * @param data　収集設定の一意識別子pkオブジェクト
	 * 	 * <!-- begin-user-doc -->
	 * The  ejbCreate method.
	 * <!-- end-user-doc -->
	 *
	 * <!-- begin-xdoclet-definition --> 
	 * @ejb.create-method 
	 * <!-- end-xdoclet-definition --> 
	 * @generated
	 */
	public CollectorPK ejbCreate(RecordCollectorData data, List<CollectorItemInfo> itemList)
	throws CreateException {
		m_log.debug("ejbCreate() start : " + data.getCollectorID());
		
//		m_dao = new CalculatedDataDAO();
		
		this.items = (CollectorItemInfo[])itemList.toArray(new CollectorItemInfo[itemList.size()]);
		setCollectorID(data.getCollectorID());
		setCollectorType(data.getCollectorType());
		setFacilityID(data.getFacilityID());
		setIntervalSec(data.getIntervalSec());
		setLabel(data.getLabel());
		setPeriodMin(data.getPeriodMin());
		setScopePath(data.getScopePath());
		setPresavePeriod(data.getPresavePeriod());
		// ファシリティツリー情報の設定は、ここでは行わずに ejbPostCreate() で実行する
		
		m_log.debug("ejbCreate() e : " + data.getCollectorID());
		
		return new CollectorPK(data.getCollectorID());
	}
	
	/**
	 * EntityBeanを生成直後にコンテナから呼ばれます。
	 * 
	 * 収集項目をDBに書き込みます。
	 * 
	 * @param pk 収集設定の一意識別子pkオブジェクト
	 * @throws  
	 */
	public void ejbPostCreate(RecordCollectorData data, List<CollectorItemInfo> itemList)
	throws javax.ejb.CreateException {
		m_log.debug("ejbPostCreate()  Collector Type : " + collectorType);

		try {
			Repository repository = new Repository();
			
			// リポジトリからツリー情報を取得
			FacilityTreeItem facilityTree = repository.getFacilityTree(facilityID);
			
			// リポジトリからデバイス情報を取得
			List<CollectorDeviceInfoData> allDeviceList = repository.getDeviceList(facilityID);

			// プラットフォームIDはScopeTree内で取得するため、ここではnullとする
			scopeTree = new ScopeTree(facilityTree, allDeviceList, null, intervalSec); 

			// スコープのフルパスを設定
			setScopePath(repository.getFacilityPath(facilityID));

			// 実績収集の場合はファシリティツリー情報をDBに格納する
			if(collectorType == CollectorType.RECORD){
				// ツリー情報をDBに書き込む
				new FacilityTreeDB().insertFacilityTree(collectorID, facilityTree);

				// デバイス情報をDBに書き込む
				Iterator<CollectorDeviceInfoData> itr = allDeviceList.iterator();
				while(itr.hasNext()){
					CollectorDeviceInfoData deviceData = itr.next();
					CollectorDeviceInfoUtil.getLocalHome().create(
							collectorID, 
							deviceData.getFacilityId(), 
							deviceData.getDeviceName(), 
							deviceData.getDisplayName(),
							deviceData.getIndex(), 
							deviceData.getDeviceId(), 
							deviceData.getDeviceType());

				}
			}

			// 収集項目用のDAOを生成
			CollectorItemDAO cidao = new CollectorItemDAO();

			for(int i=0; i<items.length; i++){
				// リアルタイム収集の場合は、設定時に収集IDがセットされないためここでセットする
				items[i].setCollectorId(collectorID);

				// 収集項目をDBに書き込む
				cidao.insertCollectorItem(items[i]);
			}

			// 状態を保持するEntityBeanを生成
			CollectorStatusUtil.getLocalHome().create(new CollectorStatusPK(collectorID));
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
	}
	
	/**
	 * DBから収集項目の情報を取得し設定します。
	 *
	 */
	public void setCollectorItems(String collectoId){
		// 収集項目用のDAOを生成
		CollectorItemDAO cidao = new CollectorItemDAO();
		
		List<CollectorItemInfo> list = cidao.selectCollectorItemByCollectorID(collectoId);
		items = (CollectorItemInfo[])list.toArray(new CollectorItemInfo[list.size()]);
		
//		// この収集で必要となる収集対象のOID情報を取得
//		targetOID = CalculationMethod.getTargetOid(items);
	}
	
	/**
	 * プライマリキーをもとにEntityBeanを検索する際にコンテナから呼ばれます。
	 * @param pk 収集設定の一意識別子pk
	 * @return pk
	 **/
	public CollectorPK ejbFindByPrimaryKey(CollectorPK pk) throws FinderException {
		return pk; 
	}
	
	/**
	 * 収集種別で検索を行います。
	 *
	 * @param type 収集種別
	 * @return CollectorPK 収集設定の一意識別子pkオブジェクトのリスト
	 **/
	public Collection ejbFindByCollectorType(int type) throws FinderException { 
		m_log.debug("ejbFindByStatus() : Type = " + type);
		
		// ここでは何もせず、DAOの実装クラスに処理をまかせる
		return null; 
	}
	/*
	 * 動作中の無期限収集を取得します。
	 */
	public Collection ejbFindByPresave() throws FinderException{
		m_log.debug("ejbFindByPresave() : ");
		
		// ここでは何もせず、DAOの実装クラスに処理をまかせる
		return null; 
		
	}
	
	/**
	 * 収集種別とファシリティIDで検索を行います。
	 * 
	 * @param type 収集種別
	 * @param facilityId ファシリティID
	 * @return CollectorPK 収集設定の一意識別子pkオブジェクトのリスト
	 **/
	public Collection ejbFindByCollectorTypeAndFacilityId(int type, String facilityId) throws FinderException { 
		m_log.debug("ejbFindByCollectorTypeAndFacilityId() : Type = " + type + ", FacilityID = " + facilityId);
		
		// ここでは何もせず、DAOの実装クラスに処理をまかせる
		return null; 
	}
	
	/**
	 * 全ての収集の検索を行います。
	 *
	 * @return CollectorPK 収集設定の一意識別子pkオブジェクトのリスト
	 **/
	public Collection ejbFindAll() throws FinderException { 
		m_log.debug("ejbFindAll() : ");
		
		// ここでは何もせず、DAOの実装クラスに処理をまかせる
		return null; 
	}
	
	public void ejbActivate() throws EJBException, RemoteException {
	}
	
	public void ejbLoad() throws EJBException, RemoteException {
		m_log.debug("ejbLoad() start : " + collectorID);
		
		m_log.debug("ejbLoad() end   : " + collectorID);
	}
	
	public void ejbPassivate() throws EJBException, RemoteException {
	}
	
	public void ejbRemove()
	throws RemoveException,	EJBException, RemoteException {
		m_log.debug("ejbRemove() start : " + collectorID);
		
		// Quartz に設定されているジョブを削除する
		deleteJob();
		
		try {
			// 状態を保持するEntityBeanを削除
			CollectorStatusUtil.getLocalHome().remove(new CollectorStatusPK(collectorID));

			// この収集IDに関連する収集項目を削除
			CollectorItemDAO cidao = new CollectorItemDAO();
			cidao.deleteCollectorItemByCollectorID(collectorID);
			
			// この収集IDに関連するデバイス情報を削除
			Iterator<CollectorDeviceInfoLocal> itr = 
				CollectorDeviceInfoUtil.getLocalHome().findByCollectorId(collectorID).iterator();
			while(itr.hasNext()){
				CollectorDeviceInfoLocal data = itr.next();
				data.remove();
			}
			
			// ファシリティツリー情報を削除
			// スコープの関係をを削除
			Iterator<CollectorScopeRelationLocal> relations = 
				CollectorScopeRelationUtil.getLocalHome().findByCollectorId(collectorID).iterator();
			while(relations.hasNext()){
				CollectorScopeRelationLocal data = relations.next();
				data.remove();
			}
			
			// スコープ情報を削除
			Iterator<CollectorScopeSnapLocal> snaps = 
				CollectorScopeSnapUtil.getLocalHome().findByCollectorId(collectorID).iterator();
			while(snaps.hasNext()){
				CollectorScopeSnapLocal data = snaps.next();
				data.remove();
			}
			
			// 性能値データを削除
			CalculatedDataDAO calDao = new CalculatedDataDAO();
			calDao.delete(collectorID);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
			throw new EJBException(e);
		}
		
		m_log.debug("ejbRemove() end   : " + collectorID);
	}
	
	public void ejbStore() throws EJBException, RemoteException {
		// ここでは何もせず、DAOの実装クラスに処理をまかせる
	}
	
	public void setEntityContext(EntityContext ctx)
	throws EJBException, RemoteException {
		entityCtx = ctx;
	}
	
	public void unsetEntityContext() throws EJBException, RemoteException {
	}
	
	/**
	 * 収集設定を取得します。
	 * @return 収集設定情報
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 */
	public RecordCollectorData getRecordColletorData(){
		m_log.debug("getRecordColletorData() strat : ");
		
		RecordCollectorData data =  new RecordCollectorData();
		
		data.setCollectorID(getCollectorID());
		data.setCollectorType(getCollectorType());
		data.setFacilityID(getFacilityID());
		data.setIntervalSec(getIntervalSec());
		data.setCount(getCount());
		data.setLabel(getLabel());
//		data.setMonitorFlag(isMonitorFlag());
		data.setStatus(getStatus());
		data.setPeriodMin(getPeriodMin());
		data.setScopePath(getScopePath());
		data.setStartDate(getStartDate());
		data.setStopDate(getStopDate());
		data.setFacilityTree(getFacilityTree());
		data.setRealCollectPeriod(getRealCollectPeriod());
		data.setEndStatus(getEndStatus());
		data.setPresavePeriod(getPresavePeriod());
		data.setLastCollectDate(getLastCollectDate());
		m_log.debug("getRecordColletorData() end   : ");
		return data;
	}
	
	/**
	 * 収集を開始します。
	 * @throws NamingException 
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void startCollect() throws NamingException{
		m_log.debug("startCollect() start : " + collectorID + " " + collectorType);
		
		setCollectorType(collectorType);
		
		// SNMPポーリングをスタートさせる
		startPoller();
		
		// 収集終了予定時刻を設定
		if(periodMin == -1){
			this.stopTime = Long.MAX_VALUE;
		} else {
			this.stopTime = System.currentTimeMillis() + periodMin * 60 * 1000l;
		}
		
		// 収集開始時刻を設定
		setStartDate(new Date(System.currentTimeMillis()));
		
		// Quartz にスケジュールを設定
		scheduleJob();
				
		// すぐに getValue() が呼ばれた場合にそなえる
		try {
			scopeTree.fetchMibValue(this.collectorID, this.facilityID);
		} catch (NotInitializedException e) {
			m_log.error(e.getMessage(), e);
		}
		
		m_log.debug("startCollect() end   : " + collectorID + " " + collectorType);
	}
	
	
	/**
	 * 収集を再開します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 * 
	 * @return 収集を再開した場合は true 再開しなかった場合は false を返す
	 * @throws NamingException
	 */
	public boolean restartCollect() throws NamingException {
		m_log.debug("restartCollect() start : " + collectorID + " "
				+ collectorType + " to " + this.stopTime);
		
		boolean isRunning = false;
		
		// ステータスが収集中かどうかを調べる
		if(getStatus() == CollectorRunStatus.RUNNING){
			isRunning = true;
		}
		
		boolean isRestart = false;
		
		// 収集終了時刻を設定
//		if(period == -1){
//			this.stopTime = Long.MAX_VALUE;
//		}
		
//		// 現在時刻よりも収集終了時刻のほうが大きい場合は再度収集を開始する
//		if(System.currentTimeMillis() < stopTime){
			// SNMPポーリングをスタートさせる
			startPoller();
			
			// この収集で必要となる収集対象のOID情報を取得
//			this.targetOID = CalculationMethod.getTargetOid(this.items);
			
			// すぐに getValue() が呼ばれた場合にそなえる
			try {
				scopeTree.fetchMibValue(this.collectorID, this.facilityID);
			} catch (NotInitializedException e) {
				m_log.error(e.getMessage(), e);
			}
			
			// Quartz にスケジュールを設定
			scheduleJob();
			
			isRestart = true;

			if(collectorType == CollectorType.RECORD){
				// アプリケーションログを出力
				// 収集IDをログ情報としてログを出力する
				String[] messages = { "CollectorID : " + collectorID };
				outputLog(EventCode.WarningRestart, messages);
			} else if (collectorType == CollectorType.MONITOR){
				// アプリケーションログを出力
				// 監視IDをログ情報としてログを出力する
//				String[] messages = { "MonitorID : " + label };
//				outputLog(EventCode.WarningRestart, messages);
			}
			
			// 既に収集中であったもので再実行したものはエラー状態を変更しない
			if(!isRunning){
				setErrorStatus(EndStatusConstant.TYPE_WARNING);
			}
//		}
		
		m_log.debug("restartCollect() end   : " + collectorID + " "
				+ collectorType + " to " + this.stopTime);
		
		return isRestart;
	}
	
	/**
	 * ポーラーを起動する
	 * 
	 * @ejb.interface-method
	 *	view-type="local" 
	 * 
	 * 	 * @throws NamingException 
	 */
	public void startPoller() throws NamingException {
		if(scopeTree == null || facilityID == null){
			// エラー処理
			return;
		}

		// SharedTableをルックアップ
		try {
			InitialContext ctx = new InitialContext();
			SharedTable sst = (SharedTable)ctx.lookup(SHARED_TABLE_JNDI_NAME);

			// 対象ファシリティに含まれるノードのリストを取得
			String[] nodeIDList = scopeTree.getNodeIDList(facilityID);
			for(int i=0; i<nodeIDList.length; i++){
				// ポーリング対象のOIDを設定
				HashMap<String, TreeSet<String>> treeSetMap = new HashMap<String, TreeSet<String>>();
				for(int j=0; j<items.length; j++){
					String platformId = Repository.getPlatformId(nodeIDList[i]);
					String subPlatformId = Repository.getSubPlatformId(nodeIDList[i]);
					
					// ItemCodeからItemCodeMstのデータを取得
					CollectorItemCodeMstLocal itemCodeMst = 
						CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(items[j].getItemCode());
					
					CollectorCategoryCollectMstPK pk = new CollectorCategoryCollectMstPK(
							platformId,
							subPlatformId,
							itemCodeMst.getCategoryCode()
					);
					
					CollectorCategoryCollectMstLocal collect = 
						CollectorCategoryCollectMstUtil.getLocalHome().findByPrimaryKey(pk);
					
					// collectMethod, platformId, subPlatformId, itemoCodeがマッチするポーリング情報を取得
					Collection<CollectorPollingMstLocal> pollingMstBeans = 
						CollectorPollingMstUtil.getLocalHome().
						findByCollectMethodAndPlatformIdAndSubPlatformIdAndItemCode(
								collect.getCollectMethod(),
								platformId,
								subPlatformId,
								items[j].getItemCode());
					
					Iterator<CollectorPollingMstLocal> itrPolling = pollingMstBeans.iterator();

					while(itrPolling.hasNext()){
						CollectorPollingMstLocal pollingLocal = itrPolling.next();
						
						TreeSet<String> treeSet = treeSetMap.get(pollingLocal.getCollectMethod());
						
						// TreeSetが存在しない場合は新規作成し、treeSetMapに追加
						if(treeSet == null || treeSet.size() == 0){
							treeSet = new TreeSet<String>();
							treeSetMap.put(pollingLocal.getCollectMethod(), treeSet);
							
							m_log.debug("startPoller create treeSetMap key : " + pollingLocal.getCollectMethod());
						}
						// 値を追加
						treeSet.add(pollingLocal.getPollingTarget());
						
						m_log.debug("startPoller treeSet add EntryKey : " + pollingLocal.getPollingTarget());
					}
				}

				// テーブル生成
				// テーブルが存在しない場合は生成する
				// 時系列情報を必要としないためページサイズは1を指定する
				if(sst.containsTable(HinemosModuleConstant.PERFORMANCE, nodeIDList[i], 2) == false){
					sst.createDataTable(HinemosModuleConstant.PERFORMANCE, nodeIDList[i], 2);
				}

				// ポーラを生成する
				Object obj = ctx.lookup(POLLER_MANAGER_JNDI_NAME);
				PollerManager manager = 
					(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

				// ポーラを取得
				PollingController poller = manager.getPoller(HinemosModuleConstant.PERFORMANCE, nodeIDList[i]);
				// ない場合は生成する
				if(poller == null){
					poller = manager.createPoller(HinemosModuleConstant.PERFORMANCE, nodeIDList[i], false, HinemosModuleConstant.PERFORMANCE, nodeIDList[i]);
				}
				
				// HashMap<String, TreeSet<String>>からHashMap<String, ArrayList<String>>へ詰め替えを行う
				HashMap<String, List<String>> pollingMap = new HashMap<String, List<String>>();
				
				Set<String> keySet = treeSetMap.keySet();
				Iterator<String> itrKeySet = keySet.iterator();
				
				while(itrKeySet.hasNext()){
					String collectMethod = itrKeySet.next();
					
					m_log.debug("collectMethod : " + collectMethod);
					
					pollingMap.put(collectMethod, new ArrayList<String>(treeSetMap.get(collectMethod)));
				}
				
				m_log.debug("startPoller map size : " + pollingMap.size());
				
				poller.startPolling(collectorID, intervalSec, pollingMap);
			}
		} catch (NotInitializedException e) {
			m_log.error(e.getMessage(), e);
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
		} catch (CollectorAlreadyExistException e) {
			m_log.error(e.getMessage(), e);
		} catch (FacilityNotFoundException e) {
			m_log.error(e.getMessage(), e);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		}
	}
	
	/**
	 * 値を取得します。(他のgetValueメソッドから呼ばれる大元となるメソッド)
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 **/
	public CollectedDataInfo getValue(String facilityId, CollectorItemInfo cii){
		// 生存期間カウンタをリセット
		resetLifetimeCounter();
		
		CollectedDataInfo data = scopeTree.getValue(facilityId, cii);
		
		// リアルタイム収集で値が取得できなかった場合は前の値を返す
		if((collectorType == CollectorType.REALTIME) && Double.isNaN(data.getValue())){
			data = scopeTree.getTempValue(facilityId, cii);
		}
		
		return data;
	}
	
//	/**
//	 * 値を取得します。
//	 * 
//	 * @ejb.interface-method
//	 *	view-type="both" 
//	 **/
//	public CollectedDataInfo getValue(String facilityId, CollectorItemInfo itemInfo){
//		for(int i=0; i<items.length; i++){
//			if(items[i].getItemCode() == itemCode){
//				return getValue(
//						facilityId, 
//						items[i]
//				);
//			}
//		}
//		
//		// 指定の収集項目IDに一致する収集項目が見付からなかった場合
//		return null;
//	}
	
	/**
	 * 値を取得します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 **/
	public CollectedDataInfo[] getValueAll(String facilityId){
		CollectedDataInfo[] dataList = new CollectedDataInfo[items.length];
		
		for(int i=0; i<items.length; i++){
			dataList[i] = getValue(
					facilityId, 
					items[i]
			);
			
		}
		
		return dataList;
	}
	
	/**
	 * 指定スコープのサブスコープのファシリティの値を取得します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 **/
	public List<CollectedDataInfo> getSubScopeValues(String facilityId, CollectorItemInfo cii){
		ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();		
		String[] subScopeFids = scopeTree.getSubScopeIDList(facilityId);
		
		for(int i=0; i<subScopeFids.length; i++){
			ret.add(getValue(subScopeFids[i], cii));
		}
		
		return ret;
	}
	
//	/**
//	 * 指定スコープのサブスコープのファシリティの値を取得します。
//	 * 
//	 * @ejb.interface-method
//	 *	view-type="both" 
//	 **/
//	public List<CollectedDataInfo> getSubScopeValues(String facilityId, String itemCode){
//		for(int i=0; i<items.length; i++){
//			if(itemCode.equals(items[i].getItemCode())){
//				return getSubScopeValues(
//						facilityId, 
//						items[i]
//				);
//			}
//		}
//		
//		// 指定の収集項目IDに一致する収集項目が見付からなかった場合
//		return null;
//	}
	
	/**
	 * 収集を終了します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 **/
	public void stopCollect(){
		m_log.debug("stopCollect() start : " + collectorID);
		
		// Quartz に設定されているジョブを削除する
		deleteJob();
		
		// ポーラを停止する
		// SNMP収集値の共有テーブルをルックアップ
		try {
			InitialContext ctx = new InitialContext();
			SharedTable sst = (SharedTable)ctx.lookup(SHARED_TABLE_JNDI_NAME);

			// 対象ファシリティに含まれるノードのリストを取得
			String[] nodeIDList = scopeTree.getNodeIDList(facilityID);
			
			m_log.debug("stopCollect() stop polling : " + nodeIDList);
			
			for(int i=0; i<nodeIDList.length; i++){
				// ポーラを生成する
				Object obj = ctx.lookup(POLLER_MANAGER_JNDI_NAME);
				PollerManager manager = 
					(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

				// ポーラを取得し停止処理を実行
				PollingController poller = manager.getPoller(HinemosModuleConstant.PERFORMANCE, nodeIDList[i]);
				if(poller != null){
					poller.stopPolling(collectorID);
				}
				
				// テーブルから参照登録を削除
				if(sst.containsTable(HinemosModuleConstant.PERFORMANCE, nodeIDList[i], 2) == false){
					sst.unregisterCollector(HinemosModuleConstant.PERFORMANCE, nodeIDList[i], collectorID);
				}
			}
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
		}
		
		// 収集終了時刻を設定
		setStopDate(new Date(System.currentTimeMillis()));
		
		m_log.debug("stopCollect() end   : " + collectorID);
	}	
	
	/**
	 * 収集スケジュールをQuartzに設定する
	 * @throws NamingException 
	 */
	private void scheduleJob() throws NamingException {
		m_log.debug("scheduleJob()  Group : "  + QuartzConstant.GROUP_COLLECT + " : " + collectorID + " : " + intervalSec);
		
		//JobDetail作成
		JobDetail job = new JobDetail(
				collectorID, 
				QuartzConstant.GROUP_COLLECT, 
				CollectJob.class);
		
		//ジョブ完了時に削除されないようにする
		job.setDurability(true);
		
		//JobDetailに呼び出すメソッドの引数を設定
		job.getJobDataMap().put("pk", new CollectorPK(collectorID));
		job.getJobDataMap().put("type", collectorType);
		
		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(collectorID, QuartzConstant.GROUP_COLLECT);
		
        //起動失敗した場合は、次の起動予定時刻をセットするように設定
        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);

        try {
        	String cronString = PollingInterval.parseCronExpression(intervalSec);

        	cronString = "10 */10 * * * ? *";

        	if(intervalSec < 60){  // 秒単位の場合(リアルタイム、実績収集)
				int delayTimeUnder1min = QuartzConstant.getDelayTimeUnder1min();
				cronString = delayTimeUnder1min + "/" + intervalSec + " * * * * ? *";
        	} else {
				int delayTimeOver1min = QuartzConstant.getDelayTimeOver1min();
        		if (60 <= intervalSec && intervalSec < 3600){  // 分単位の場合
					cronString = delayTimeOver1min + " */" + intervalSec/60 + " * * * ? *";
        		} else if (3600 <= intervalSec){  // 時間単位の場合
        			cronString = delayTimeOver1min + " 0 */" + intervalSec/3600 + " * * ? *"; // ポーリングからdelayTime秒ずらして起動
        		} 
        		
        		// 差分をとって計算する場合（最低２回はポーリングを行っている必要がある場合）に合わせて、計算開始時間を１監視間隔分遅らせる
        		startTime += intervalSec*1000;
        		
        		//m_log.info("*BEFORE* startTime = " + startTime);
        		// 定期ポーリングから計算ロジックが走るまでの魔の時間(delayTime)に登録された場合は更にdelayTime分計算開始時間を遅らせる
        		if ((startTime/1000 % intervalSec) <= (delayTimeOver1min)){
        			//m_log.info("*in if 1* startTime = " + startTime);
        			startTime += delayTimeOver1min*1000;
        			//m_log.info("*in if 2* startTime = " + startTime);
        		} 
        	}
			
			cronTrigger.setCronExpression(cronString);
			cronTrigger.setStartTime(new Date(startTime));
			cronTrigger.setEndTime(new Date(stopTime));
		} catch (ParseException e) {
			m_log.error(e.getMessage(), e);
		} catch (IllegalArgumentException e){
			// CronTriggerクラスのsetEndTime()で 
			// end time よりも start time のほうが大きい場合throwされる
			m_log.error(e.getMessage(), e);
			
			// Quartzを起動する必要がないのでこのまま処理を終了する
			return;
		}
		
		//QuartzのSchedulerをルックアップ
		InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
		Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());
		
		try {
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(collectorID, QuartzConstant.GROUP_COLLECT);
			
			// ジョブを登録する
			scheduler.scheduleJob(job, cronTrigger);
		} catch (SchedulerException e) {
			m_log.error(e.getMessage(), e);
		}
	}
	
	/**
	 * 収集起動ジョブを一時停止します。
	 *
	 */
	private void pauseJob() {
		m_log.debug("pauseJob() start : " + collectorID);
		
		//QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());
			
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.pauseJob(collectorID, QuartzConstant.GROUP_COLLECT);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (SchedulerException e) {
			m_log.error(e.getMessage(), e);
		}
		m_log.debug("pauseJob() end   : " + collectorID);
	}
	
	/**
	 * 収集起動ジョブを再開します。
	 *
	 */
	private void resumeJob() {
		m_log.debug("pauseJob() start : " + collectorID);
		
		//QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());
			
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.resumeJob(collectorID, QuartzConstant.GROUP_COLLECT);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (SchedulerException e) {
			m_log.error(e.getMessage(), e);
		}
		m_log.debug("pauseJob() end   : " + collectorID);
	}
	
	/**
	 * 収集起動ジョブを削除します。
	 */
	private void deleteJob() {
		m_log.debug("deleteJob() start : " + collectorID);
		
		//QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());
			
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(collectorID, QuartzConstant.GROUP_COLLECT);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (SchedulerException e) {
			m_log.error(e.getMessage(), e);
		}
		m_log.debug("deleteJob() end   : " + collectorID);
	}
	
	/**
	 * ポーラーのテーブルが保持するMIBの値を取得します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void fetchMibValue(){
		m_log.debug("fetchMibValue() start : " + collectorID + "  " + collectorType);
		
		try{
			if(scopeTree == null){
				m_log.debug("lost facility tree info. " + collectorID);
			} else {
				// MIB値を取得
				try{
					scopeTree.fetchMibValue(this.collectorID, this.facilityID);
				} catch (NotInitializedException e) {
					// ポーラが正しく起動していないためMIBの値の取得に失敗。
					// 再度ポーラを起動する。
					m_log.warn("restart poller : " + collectorID, e);
					startPoller();
					return;
				} 
			}
			
			// 最終収集日時をDBに書き込みます
			setLastCollectDate(new Date(System.currentTimeMillis()));
			
			// 収集回数をDBに書き込みます
			setCollectCount(getCount()+1);
		} catch (Exception e) {
			m_log.error("error in fetchMibValue()", e);
			// 状態を異常に変更
			this.setErrorStatus(EndStatusConstant.TYPE_ABNORMAL);
		} finally {
			// 寿命カウンタを増加させます。
			lifetimeCounter++;
		}
		
		m_log.debug("fetchMibValue() end   : " + collectorID + "  " + collectorType);
	}
	
	/**
	 * 性能値を算出しDBに書き込みます。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void storeCalcValue(){
		m_log.debug("storeCalcValue() start : " + collectorID + "  " + collectorType);
		
		try{
			if(scopeTree == null || items == null){
				m_log.debug("情報を失っています");
			} else {
				try {
					// MIB値を取得（同時に収集時刻を取得）
					// このメソッドが呼ばれた際にDBに書き込む性能値の日時は全て同じものとするため
					long collectTime = scopeTree.fetchMibValue(this.collectorID, this.facilityID);
					Date collectDate = new Date(collectTime);

					// 全てのファシリティのIDを取得
					String[] fids = scopeTree.getAllFacilityIdList();

					for (int i = 0; i < items.length; i++) {
						for (int j = 0; j < fids.length; j++) {
							CollectedDataInfo data = getValue(fids[j], items[i]);

							// このメソッドが呼ばれた際にDBに書き込む性能値の日時は全て同じものとするため
							data.setDate(collectDate);

							// DBへ出力
							CalculatedDataDAO dao = new CalculatedDataDAO();
							dao.insert(collectorID, data);
						}
					}

					// 最終収集時刻を設定
					setLastCollectDate(new Date(scopeTree.getLastCollectTime()));

					// 収集回数をDBに書き込みます
					setCollectCount(getCount() + 1);

				} catch (NotInitializedException e) {
					// ポーラが正しく起動していないためMIBの値の取得に失敗。
					// 再度ポーラを起動する。
					m_log.warn("restart poller : " + collectorID, e);
					startPoller();
					return;
				} 
			}
		} catch (Exception e) {
			m_log.error("error in storeCalcValue()", e);
			// 状態を異常に変更
			this.setErrorStatus(EndStatusConstant.TYPE_ABNORMAL);
		}
		
		m_log.debug("storeCalcValue() end   : " + collectorID + "  " + collectorType);
	}
	
	/**
	 * 性能値を算出し閾値判定します。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void monitorCalcValue(){
		m_log.debug("monitorCalcValue() start : " + collectorID + "  " + collectorType);
		
		if(scopeTree == null || items == null){
			m_log.debug("情報を失っています");
			return;
		}
		
		try{
			// MIB値を取得
			try{
				scopeTree.fetchMibValue(this.collectorID, this.facilityID);
			} catch (NotInitializedException e) {
				// ポーラが正しく起動していないためMIBの値の取得に失敗。
				// 再度ポーラを起動する。
				m_log.warn("restart poller : " + collectorID, e);
				startPoller();
			} 
			
			// 性能管理閾値監視情報を取得
//			MonitorPerfInfoHome home = JNDIConnectionManager.getMonitorPerfInfo();
//			Collection<MonitorPerfInfoLocal> mPerfInfos = home.findAll();
				
			Collection<MonitorPerfInfoLocal> mPerfInfos = MonitorPerfInfoUtil.getLocalHome().findAll();
			
			Iterator<MonitorPerfInfoLocal> itr = mPerfInfos.iterator();
			while(itr.hasNext()){
				MonitorPerfInfoLocal monInfoLocal = itr.next();
				
				if(monInfoLocal.getCollectorId().equals(collectorID)){
					String monitorTypeId = monInfoLocal.getMonitorTypeId();
					String monitorId = monInfoLocal.getMonitorId();
					String itemCode = monInfoLocal.getItemCode();
					String displayName = monInfoLocal.getDisplayName();
					
					// 収集中の項目のうち監視対象項目を特定する
					CollectorItemInfo cii = null;
					
					for(int i=0; i<items.length; i++){
						if(itemCode.equals(items[i].getItemCode()) && displayName.equals(items[i].getDisplayName())){
							cii = items[i];
						}
					}
					
					// 対応する収集項目がない場合
					if(cii == null){
						m_log.error("Not match CollectorItem. (code:" + itemCode + ", display name:"+displayName+")");
						return;
					}
					
					// 対象ファシリティ(ノードもしくはスコープ)の閾値判定を実行する
					CollectedDataInfo data = getValue(this.facilityID, cii);
					
					// 閾値判定を実行
					RunMonitorPerformance2 runMonitor = new RunMonitorPerformance2();
					runMonitor.run(monitorTypeId, monitorId, data);

					 // 監視対象ファシリティがスコープの場合は、スコープに含まれるノードへの処理を行う
					Repository rep = new Repository();
					if(!rep.isNode(this.facilityID)){
						// スコープに含まれるノードの閾値判定を実行
						// 全てのノードのIDを取得
						String[] nodeFids = scopeTree.getNodeIDList(this.facilityID);
						
						for(int j=0; j<nodeFids.length; j++){
							data = getValue(nodeFids[j], cii);
							
							// 閾値判定を実行
							runMonitor = new RunMonitorPerformance2();
							runMonitor.run(monitorTypeId, monitorId, data);
						}
					}
				}
			}
			
			// 最終収集時刻を設定
			setLastCollectDate(new Date(scopeTree.getLastCollectTime()));
			
			// 収集回数をDBに書き込みます
			setCollectCount(getCount()+1);
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
		}
		
		m_log.debug("monitorCalcValue() end   : " + collectorID + "  " + collectorType);
	}
	
	/**
	 * 生存期間カウンタをリセットします。
	 *
	 */
	private void resetLifetimeCounter(){
		lifetimeCounter = 0;
	}
	
	
	/**
	 * 収集IDを取得します。
	 * 
	 * @return 収集ID
	 * 
	 * @ejb.persistence
	 * @ejb.pk-field 
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public String getCollectorID() {
		return collectorID;
	}
	
	/**
	 * 収集IDを設定します。
	 * @param collectorID 収集ID
	 */
	public void setCollectorID(String collectorID) {
		this.collectorID = collectorID;
	}
	
	/**
	 * 収集の種別を取得します。(リアルタイム収集,実績収集,リソース監視)
	 * 
	 * @return 収集種別
	 * 
	 * @ejb.persistence
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public int getCollectorType() {
		return collectorType;
	}
	
	/**
	 * 収集の種別を設定します。(リアルタイム収集,実績収集,リソース監視)
	 * 
	 * @return 収集種別
	 */
	public void setCollectorType(int collectorType) {
		this.collectorType = collectorType;
	}
	
	/**
	 * ファシリティIDを取得します。
	 * @return facilityID ファシリティID
	 * 
	 * @ejb.persistence 
	 */
	public String getFacilityID() {
		return facilityID;
	}
	
	/**
	 * ファシリティIDを設定します。
	 * @param facilityID ファシリティID
	 */
	public void setFacilityID(String facilityID) {
		m_log.debug("setFacilityID() start : " + collectorID);
		
		this.facilityID = facilityID;
		
		m_log.debug("setFacilityID() end   : " + collectorID);
	}
	
	/**
	 * 性能値算出用のファシリティツリー情報を再構築する
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void updateScopeTree() {
		m_log.debug("updateScopeTree() start : " + collectorID);
		
		// ツリー情報を取得（メソッド内で実績、リアルタイムの判定をしている）
		FacilityTreeItem facilityTree = getFacilityTree();
		
		// デバイス情報を取得
		List<CollectorDeviceInfoData> allDeviceList = new ArrayList<CollectorDeviceInfoData>();
		// プラットフォーム情報（サブプラットフォームID含む）を取得
		HashMap<String, ArrayList<String>> platformMap = null;
		if(collectorType == CollectorType.REALTIME || collectorType == CollectorType.MONITOR){
			// リポジトリから取得
			allDeviceList = new Repository().getDeviceList(facilityID);
			platformMap = null;
		} else if(collectorID != null && collectorType == CollectorType.RECORD){
			// DBからファシリティツリー情報を取得
			// デバイス情報をDBから取得する
			try {
				Collection<CollectorDeviceInfoLocal> beans = 
					CollectorDeviceInfoUtil.getLocalHome().findByCollectorId(collectorID);

				// デバイス情報を設定
				Iterator<CollectorDeviceInfoLocal> itr = beans.iterator();
				while(itr.hasNext()){
					CollectorDeviceInfoLocal bean = itr.next();
					allDeviceList.add(
							new CollectorDeviceInfoData(
									bean.getCollectorId(),
									bean.getFacilityId(),
									bean.getDeviceName(),
									bean.getDisplayName(),
									bean.getIndex(),
									bean.getDeviceId(),
									bean.getDeviceType()
							));
				}
			
				// プラットフォームIDをDBから取得する
				platformMap = new HashMap<String, ArrayList<String>>();
				Collection<CollectorScopeSnapLocal> col = CollectorScopeSnapUtil.getLocalHome().findByCollectorId(collectorID);
			
				Iterator<CollectorScopeSnapLocal> itr2 = col.iterator();
				while(itr2.hasNext()){
					CollectorScopeSnapLocal bean = itr2.next();
					
					// bean.getFacilityType() が nodeの場合のみ、HashMapに詰める
					if(bean.getFacilityType().equals(FacilityConstant.TYPE_NODE_STRING)) {
						ArrayList<String> platforms = new ArrayList<String>();
						platforms.add(bean.getPlatformId());
						platforms.add(bean.getSubPlatformId());
						platformMap.put(bean.getFacilityId(), platforms);
					}
				}
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			}
			
		} else {
			// エラー
		}
		
		if(facilityTree != null && allDeviceList != null){
			// ファシリティツリー情報を再構築する
			this.scopeTree = new ScopeTree(facilityTree, allDeviceList, platformMap, intervalSec);
		}
		
		m_log.debug("updateScopeTree() end   : " + collectorID);
	}
	
	
	/**
	 * ファシリティツリーを取得します。
	 * @return facilityTree ファシリティツリー
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 * 
	 * @ejb.persistence 
	 */
	public FacilityTreeItem getFacilityTree() {
		m_log.debug("getFacilityTree() collector type : " + collectorType); 
		
		if(facilityID != null && 
				(collectorType == CollectorType.REALTIME || collectorType == CollectorType.MONITOR)){
			// リポジトリから取得
			// リポジトリの情報が変更されたなどの理由で指定のファシリティIDが存在しない場合 null が返る
			return new Repository().getFacilityTree(facilityID);
		} else if(collectorID != null && collectorType == CollectorType.RECORD){
			// DBからファシリティツリー情報を取得
			return new FacilityTreeDB().getFacilityTree(collectorID);
		} else {
			return null;
		}
	}
	
	/**
	 * 収集間隔を取得します。
	 * @return intervalSec 収集間隔
	 *
	 * @ejb.interface-method
	 *  view-type="both"
	 * 
	 * @ejb.persistence 
	 */
	public int getIntervalSec() {
		return intervalSec;
	}
	
	/**
	 * 収集間隔を設定します。
	 * @param intervalSec 収集間隔
	 */
	public void setIntervalSec(int intervalSec) {
		this.intervalSec = intervalSec;
	}
	
	/**
	 * 収集のラベルを取得します。
	 * @return label ラベル
	 * 
	 * @ejb.persistence 
	 */
	public String getLabel() {
		return label;
	}
	
	/**
	 * ラベルを設定します。
	 * @param label ラベル
	 */
	public void setLabel(String label) {
		this.label = label;
	}
	
	/**
	 * 収集期間を取得します。
	 * @return periodMin 収集期間
	 * 
	 * @ejb.persistence 
	 */
	public int getPeriodMin() {
		return periodMin;
	}
	
	/**
	 * 収集期間を設定します。
	 * @param periodMin 収集期間
	 */
	public void setPeriodMin(int periodMin) {
		this.periodMin = periodMin;
	}
	
	/**
	 * 収集回数を取得します。
	 * 
	 * @ejb.persistence
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 *  
	 * @return 収集回数
	 */
	public long getCount() {
		return getCollectorStatus().getCount();
	}
	
	/**
	 * 最終収集終了時刻を取得します。
	 * 
	 * @ejb.persistence 
	 *  
	 * @ejb.interface-method
	 *  view-type="both"
	 * 
	 * @return 最終収集終了時刻
	 */
	public Date getLastCollectDate() {
		return getCollectorStatus().getLastCollectDate();
	}
	
	
	/**
	 * 収集回数を設定します。
	 * （EntityBeanのsetterメソッドとならないようにあえてsetCount()とは異なる名前としている）
	 * 
	 * @param count 収集回数
	 */
	public void setCollectCount(long count) {
		getCollectorStatus().setCount(count);
	}
	
	/**
	 * 実行中のエラー状態を設定します。
	 * 
	 * @param status 実行中のエラー状態
	 */
	public void setErrorStatus(int status) {
		getCollectorStatus().setErrorStatus(status);
		
		int endStatus = getCollectorStatus().getEndStatus();
		
		// 現在までの状態より深刻な状態の場合は終了状態を変更する
		
		// 現在までの終了状態が"正常"で、状態が"警告"または"異常"に変化した場合
		if(endStatus == EndStatusConstant.TYPE_NORMAL &&
				(status == EndStatusConstant.TYPE_WARNING || 
						status == EndStatusConstant.TYPE_ABNORMAL)){
			getCollectorStatus().setEndStatus(status);
		}
		
		// 現在までの終了状態が"警告"で、状態が"異常"に変化した場合
		if(endStatus == EndStatusConstant.TYPE_WARNING &&
				status == EndStatusConstant.TYPE_ABNORMAL){
			getCollectorStatus().setEndStatus(status);
		}
		
		// 現在までの終了状態が"異常"以外で、状態が"異常"に変化した場合
		if(endStatus != EndStatusConstant.TYPE_ABNORMAL &&
				status == EndStatusConstant.TYPE_ABNORMAL){
			getCollectorStatus().setEndStatus(status);
		}
	}
	
	/**
	 * スコープパスを取得します。
	 * @return scopePath スコープパス
	 * 
	 * @ejb.persistence 
	 */
	public String getScopePath() {
		return scopePath;
	}
	
	/**
	 * スコープパスを設定します。
	 * @param scopePath スコープパス
	 */
	public void setScopePath(String scopePath) {
		this.scopePath = scopePath;
	}
	
	/**
	 * 収集開始日時を取得します。
	 * @return startDate 収集開始時刻(収集を開始していない場合はnullを返す)
	 * @ejb.interface-method
	 *  view-type="both"
	 * @ejb.persistence 
	 */
	public Date getStartDate() {
		if(startTime != 0){
			return new Date(startTime);
		} else {
			return null;
		}
	}
	
	/**
	 * 収集開始日時を設定します。
	 * @param startTime 収集開始日時
	 * 
	 * @ejb.interface-method
	 * view-type="both"
	 */
	public void setStartDate(Date startDate) {
		if(startDate != null){
			this.startTime = startDate.getTime();
		} else {
			this.startTime = 0;
		}
	}
	
//	/**
//	 * 収集開始日時を設定します。
//	 * @param startTime 収集開始日時
//	 * 
//	 *  @ejb.interface-method
//	 *  view-type="both"
//	 *  
//	 */
//	public void setStartTime(long startTime) {
//		this.startTime = startTime;
//	}
	
	/**
	 * ステータスを取得します。
	 * @return status ステータス
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 *  
	 * @ejb.persistence 
	 */
	public int getStatus() {
		//QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());
			
			// 登録されているジョブを取得する
			JobDetail job;
			try {
				job = scheduler.getJobDetail(collectorID, QuartzConstant.GROUP_COLLECT);
			} catch (SchedulerException e) {
				m_log.error(e.getMessage(), e);
				return CollectorRunStatus.ERROR;
			}
			
			// 登録されているトリガーを取得する
			Trigger trigger = null;
			try {
				trigger = scheduler.getTrigger(collectorID, QuartzConstant.GROUP_COLLECT);
			} catch (SchedulerException e) {
				m_log.error(e.getMessage(), e);
				return CollectorRunStatus.ERROR;
			}
			
			if(trigger == null){  // 収集実行ジョブが存在していない場合
				if(getStartDate() == null){
					return CollectorRunStatus.READY;
				} else {
					return CollectorRunStatus.FINISHED;
				}
			} else if(job.isDurable()){
				return CollectorRunStatus.RUNNING;
			} else {
				return CollectorRunStatus.SUSPEND;
			}
		} catch (NamingException e) {
			throw new EJBException(e);
		}
	}
		
	/**
	 * 収集終了時刻を取得します。
	 * @return 収集終了時刻
	 * @ejb.interface-method
	 *  view-type="both"
	 * @ejb.persistence 
	 */
	public Date getStopDate() {
		if(stopTime == 0 || stopTime == Long.MAX_VALUE) {
			return null;  // 無制限の場合
		} else {
			return new Date(stopTime);
		}
	}
	
	/**
	 * 収集終了日時を設定し、DBに永続化します。
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 */
	public void setStopDate(Date stopDate) {
		if(stopDate != null){
			this.stopTime = stopDate.getTime();
		} else {
			this.stopTime = Long.MAX_VALUE;
		}
	}
	
//	/**
//	 * 収集終了日時を設定します。
//	 * @param stopTime 収集終了日時
//	 */
//	public void setStopTime(long stopTime) {
//		this.stopTime = stopTime;
//	}
	
	/**
	 * 実際に収集していた時間を返します。
	 * @return 収集時間(ミリ秒)
	 * 
	 * @ejb.persistence 
	 */
	public long getRealCollectPeriod() {
		Date lastCollect = getCollectorStatus().getLastCollectDate();
		if(lastCollect == null){
			return 0l;
		} else {
			return lastCollect.getTime() - startTime;
		}
	}
	
	private CollectorStatusLocal getCollectorStatus(){
		try {
			return CollectorStatusUtil.getLocalHome().findByPrimaryKey(
					new CollectorStatusPK(collectorID));
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		return null;
	}
	
	/**
	 * 収集終了時の状態を取得します。
	 * @return status 終了状態
	 * 
	 * @ejb.persistence 
	 */
	public int getEndStatus() {
		return getCollectorStatus().getEndStatus();
	}
	
	public void setLastCollectDate(Date date)
	{
		getCollectorStatus().setLastCollectDate(date);
	}
	
	/**
	 * 収集を続けるべきかどうかを判定します。
	 * このコレクターが指定の回数以上参照されていない場合は、falseを返します。
	 * 
	 * @return まだ収集を続ける場合はtrueを、終了すべき状態にある場合は、falseを返します。
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 *  
	 **/
	public boolean checkLifetime(){
		if (MAX_LIFECOUNT >= lifetimeCounter){
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * アプリケーションログにログを出力
	 * 
	 * @param e 例外
	 */
	private void outputLog(String index, String[] messages) {
		AplLogger apllog = new AplLogger("PERF", "perf");
		apllog.put("SYS", index, messages);
	}
	
	// いずれは削除する予定
	/**
	 * 閾値監視フラグを取得します。
	 * @return monitorFlag 閾値監視フラグ
	 * 
	 * @ejb.persistence 
	 */
	public boolean isMonitorFlag() {
		return false;
	}

	/**
	 * 無期限収集用の保存期間情報を取得する。
	 * @return
	 * 	@ejb.interface-method
	 *  view-type="both"
	 *  @ejb.persistence 
	 */
	public int getPresavePeriod() {
		return presavePeriod;
	}
	/**
	 * 無期限収集用の保存期間情報をセットする。 
	 * @param presavePeriod
	 */
	public void setPresavePeriod(int presavePeriod) {
		this.presavePeriod = presavePeriod;
	}
	
	/**
	 * 性能値のマネージメントをおこないます。
	 * 
	 * @ejb.interface-method
	 *	view-type="both" 
	 */
	public void managePresave(Date startDate){

		CalculatedDataDAO dao = new CalculatedDataDAO();
		dao.deleteOverPresave(collectorID, startDate);
		int count = dao.recountFromDB(collectorID);
		if(count > -1){
			this.setCollectCount(count);
		}
	}
	
}
