/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.util.Collection;

import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を削除するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class DeleteNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( DeleteNotify.class );
	
	/**
	 * 通知情報を削除します。
	 * <p>
	 * <ol>
	 *  <li>通知IDより、通知情報を取得します。</li>
	 *  <li>通知情報に設定されている通知イベント情報を削除します。</li>
	 *  <li>通知情報を削除します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュから削除されます。</li>
	 * </ol>
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see #deleteEvents(Collection)
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext#NotifyInfoContext(int, java.lang.String, com.clustercontrol.notify.bean.NotifyInfo)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	public boolean delete(String notifyId) throws FinderException, RemoveException, NamingException, JMSException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.NOTIFY);
    	
		NotifyInfoLocal notify = null;
		SendQueue queue = null;
		try 
		{
			// 通知イベント情報を取得
			notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			
			// 通知詳細情報を削除
            deleteDetails(notify);            
            
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "003", args);
            m_log.debug("delete():" + e.getMessage());
			throw e;
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "003", args);
            m_log.debug("delete():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "003", args);
            m_log.debug("delete():" + e.getMessage());
			throw e;
		}

		try 
		{
            // 関連情報を削除
            deleteRelation(notifyId);
			
            // 抑制情報を削除
            deleteInhibition(notifyId);
			
			// 通知情報を削除
			notify.remove();
						
			NotifyInfoContext context = new NotifyInfoContext(NotifyInfoContext.TYPE_DELETE, notifyId, null);
			
			// 通知情報キャッシュを更新
			queue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFYCONTROL);
			queue.put(context);
			queue.terminate();
			
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "003", args);
            m_log.debug("getNotify():" + e.getMessage());
			throw e;
		} catch (JMSException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "003", args);
            m_log.debug("add():" + e.getMessage());
			throw e;
		}
		finally{
			if(queue != null){
				queue.terminate();	
			}
		}
		return true;
	}
	
	/**
	 * 通知詳細情報を削除します。
	 * @param ct
	 * @param NotofyType
	 * @return
	 * @throws RemoveException
	 */
	public boolean deleteDetails(NotifyInfoLocal notify) throws RemoveException {

		Integer notifyType = notify.getNotifyType();

		
		if( notifyType == com.clustercontrol.bean.NotifyTypeConstant.TYPE_STATUS){
			
			Collection ct = notify.getNotifyStatusInfo();
			return deleteStatus(ct);
		
		}else if( notifyType == com.clustercontrol.bean.NotifyTypeConstant.TYPE_EVENT){
			
			Collection ct = notify.getNotifyEventInfo();
			return deleteEvents(ct);
		
		}else if( notifyType == com.clustercontrol.bean.NotifyTypeConstant.TYPE_MAIL){
		
			Collection ct = notify.getNotifyMailInfo();
			return deleteMails(ct);
		
		}else if( notifyType == com.clustercontrol.bean.NotifyTypeConstant.TYPE_JOB){
		
			Collection ct = notify.getNotifyJobInfo();
			return deleteJobs(ct);
		
		}else if( notifyType == com.clustercontrol.bean.NotifyTypeConstant.TYPE_LOG_ESCALATE){
		
			Collection ct = notify.getNotifyLogEscalateInfo();
			return deleteLogEscalate(ct);
		
		}
		return false;
	}
	
	/**
	 * 通知ステータス情報を削除します。
	 * 
	 * @param ct 削除対象の通知ステータス情報のコレクション
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyStatusInfoBean
	 */
	public boolean deleteStatus(Collection ct) throws RemoveException {

		// 通知ステータス情報を削除
        if(ct != null && ct.size() > 0){
            Object[] eventArray = ct.toArray();
            for(int j = 0; j < eventArray.length; j++){
            	NotifyStatusInfoLocal event = (NotifyStatusInfoLocal)eventArray[j];
                if(event != null){
                	event.remove();
                }
            }
        }
		return true;
	}
	
	/**
	 * 通知イベント情報を削除します。
	 * 
	 * @param ct 削除対象の通知イベント情報のコレクション
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public boolean deleteEvents(Collection ct) throws RemoveException {

		// 通知イベント情報を削除
        if(ct != null && ct.size() > 0){
            Object[] eventArray = ct.toArray();
            for(int j = 0; j < eventArray.length; j++){
            	NotifyEventInfoLocal event = (NotifyEventInfoLocal)eventArray[j];
                if(event != null){
                	event.remove();
                }
            }
        }
		return true;
	}
	
	/**
	 * 通知メール情報を削除します。
	 * 
	 * @param ct 削除対象の通知メール情報のコレクション
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyMailInfoBean
	 */
	public boolean deleteMails(Collection ct) throws RemoveException {

		// 通知イベント情報を削除
        if(ct != null && ct.size() > 0){
            Object[] eventArray = ct.toArray();
            for(int j = 0; j < eventArray.length; j++){
            	NotifyMailInfoLocal event = (NotifyMailInfoLocal)eventArray[j];
                if(event != null){
                	event.remove();
                }
            }
        }
		return true;
	}
	
	/**
	 * 通知ジョブ情報を削除します。
	 * 
	 * @param ct 削除対象の通知ジョブ情報のコレクション
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyJobInfoBean
	 */
	public boolean deleteJobs(Collection ct) throws RemoveException {

		// 通知イベント情報を削除
        if(ct != null && ct.size() > 0){
            Object[] eventArray = ct.toArray();
            for(int j = 0; j < eventArray.length; j++){
            	NotifyJobInfoLocal event = (NotifyJobInfoLocal)eventArray[j];
                if(event != null){
                	event.remove();
                }
            }
        }
		return true;
	}
	
	/**
	 * 通知ログエスカレーション情報を削除します。
	 * 
	 * @param ct 削除対象の通知ログエスカレーション情報のコレクション
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoBean
	 */
	public boolean deleteLogEscalate(Collection ct) throws RemoveException {

		// 通知イベント情報を削除
        if(ct != null && ct.size() > 0){
            Object[] eventArray = ct.toArray();
            for(int j = 0; j < eventArray.length; j++){
            	NotifyLogEscalateInfoLocal event = (NotifyLogEscalateInfoLocal)eventArray[j];
                if(event != null){
                	event.remove();
                }
            }
        }
		return true;
	}
	
	/**
	 * 関連情報を削除します。
	 * 
	 * @param notifyId
	 * @return 削除に成功した場合、</code> true </code>
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotifyRelation
	 */
	
	public boolean deleteRelation(String notifyId) {
		// 通知関連情報を削除
		DeleteNotifyRelation deleteRelation = new DeleteNotifyRelation();
		deleteRelation.deleteBaseOnNotifyId(notifyId);
		
		return true;
	}
	
	
	/**
	 * 抑制情報を削除します。
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、</code> true </code>
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotifyInhibition
	 */
	public boolean deleteInhibition(String notifyId) {

		// 抑制情報を削除
		DeleteNotifyInhibition deleteInhibition = new DeleteNotifyInhibition();
		deleteInhibition.deleteBaseOnNotifyId(notifyId);
        
		return true;
	}
	
	
}
