/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dao.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.dao.FileSystemInfoDAOImpl;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoBean;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoPK;
import com.clustercontrol.repository.util.RepositoryCacheProperties;

/**
 * FileSystemInfo(BMP Entity Bean)のDAO実装クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class CachedFileSystemInfoDAOImpl extends FileSystemInfoDAOImpl {
	protected static Log m_log = LogFactory.getLog( CachedFileSystemInfoDAOImpl.class );
	
	// 初期化されているか否かを示すフラグ
	private Boolean m_isInit = false;
	
	// ファシリティIDからの検索キャッシュ
	private static HashMap<String, Collection> m_facilityIdCache = null;
	
	// 保持しているキャッシュ用HashMapのリスト
	private static List<HashMap> m_cacheMapList  = new ArrayList<HashMap>();
	
	// 初期化
	@ Override
	public void init() {
		synchronized (m_isInit) {
			if (m_isInit) {
				// 既に初期化されているため何もしない
				return;
			} else {
				// キャッシュ対象となるもののHashMapを生成
				if (RepositoryCacheProperties
						.getProperties(RepositoryCacheProperties.FILESYSTEM_FACILITYID_KEY) > 0) {
					m_facilityIdCache = new HashMap<String, Collection>();
					m_cacheMapList.add(m_facilityIdCache);
					m_log.info("FileSystemInfoDAO findByFacilityId : enable");
				}

				m_isInit = true;
			}
		}
	}
	
	/**
	 * 全てのキャッシュをクリアする。
	 */
	private static void clearCacheAll(){
		synchronized (m_cacheMapList) {
			Iterator<HashMap> itr = m_cacheMapList.iterator();
			while(itr.hasNext()){
				HashMap map = itr.next();
				
				synchronized (map) {
					m_log.debug("clearCache() clear : " + map.size());
					map.clear();
				}
			}
		}
	}

//	LDAPアクセス必須
//	public void load(FileSystemInfoPK pk, FileSystemInfoBean ejb) throws EJBException {
//	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#store(com.clustercontrol.repository.ejb.FileSystemInfoBean)
	 */
	@Override
	public void store(FileSystemInfoBean ejb) throws EJBException {
		m_log.debug("store()");
		
		super.store(ejb);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#remove(com.clustercontrol.repository.ejb.FileSystemInfoPK)
	 */
	@Override
	public void remove(FileSystemInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove()");
		
		super.remove(pk);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#create(com.clustercontrol.repository.ejb.FileSystemInfoBean)
	 */
	@Override
	public FileSystemInfoPK create(FileSystemInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create()");
		
		FileSystemInfoPK pk = super.create(ejb);
		
		// キャッシュをクリアする
		// （更新後にクリアしないと他のスレッドによりキャッシュの更新が入る可能性がある）
		clearCacheAll();
		
		return pk;
	}
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#findByFacilityId(java.lang.String)
	 */
	@Override
	public Collection findByFacilityId(String facilityId) throws FinderException {
		// キャッシュを利用しない設定の場合
		if(m_facilityIdCache == null){
			return super.findByFacilityId(facilityId);
		}

		synchronized (m_facilityIdCache) {
			// キャッシュに存在する場合はそれを返す
			if(m_facilityIdCache.containsKey(facilityId)){
				return m_facilityIdCache.get(facilityId);
			}
		}
		
		// キャッシュに存在しない場合
		Collection ret = super.findByFacilityId(facilityId);
		synchronized (m_facilityIdCache) {
			m_log.debug("findByFacilityId(String facilityId) put : " + facilityId
					+ "  entry size : " + ret.size()
					+ "  total entry : " + m_facilityIdCache.size());
			m_facilityIdCache.put(facilityId, ret);
		}
		return ret;
	}
}
