/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.util;

import java.io.UnsupportedEncodingException;
import java.security.Principal;

import javax.naming.InitialContext;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.security.SimpleGroup;
import org.jboss.security.auth.spi.LdapExtLoginModule;

/**
 * Hinemos用LdapLoginModuleクラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class LdapLoginModule extends LdapExtLoginModule
{
	protected static Log log = LogFactory.getLog( LdapLoginModule.class );
	
	/** 再接続回数 */
	private static final int RECONNECT_MAX = 5;
	
	private static final String ROLES_CTX_DN_OPT = "rolesCtxDN";
	private static final String ROLE_ATTRIBUTE_ID_OPT = "roleAttributeID";
	private static final String ROLE_ATTRIBUTE_IS_DN_OPT = "roleAttributeIsDN";
	private static final String ROLE_NAME_ATTRIBUTE_ID_OPT = "roleNameAttributeID";
	
	private static final String BASE_CTX_DN = "baseCtxDN";
	private static final String BASE_FILTER_OPT = "baseFilter";
	private static final String ROLE_FILTER_OPT = "roleFilter";
	private static final String ROLE_RECURSION = "roleRecursion";
	private static final String DEFAULT_ROLE = "defaultRole";
	private static final String SEARCH_TIME_LIMIT_OPT = "searchTimeLimit";
	private static final String SEARCH_SCOPE_OPT = "searchScope";
	
	public LdapLoginModule()
	{
	}
	
	private transient SimpleGroup userRoles = new SimpleGroup("Roles");
	
	/**
	 Validate the inputPassword by creating a ldap InitialContext with the
	 SECURITY_CREDENTIALS set to the password.
	 @param inputPassword the password to validate.
	 @param expectedPassword ignored
	 */
	protected boolean validatePassword(String inputPassword, String expectedPassword)
	{
		boolean isValid = false;
		if (inputPassword != null)
		{
			// See if this is an empty password that should be disallowed
			if (inputPassword.length() == 0)
			{
				// Check for an allowEmptyPasswords option
				boolean allowEmptyPasswords = true;
				String flag = (String) options.get("allowEmptyPasswords");
				if (flag != null)
					allowEmptyPasswords = Boolean.valueOf(flag).booleanValue();
				if (allowEmptyPasswords == false)
				{
					log.trace("Rejecting empty password due to allowEmptyPasswords");
					return false;
				}
			}
			
			try
			{
				// Validate the password by trying to create an initial context
				String username = getUsername();
				isValid = createLdapInitContext(username, inputPassword);
				defaultRole();
				isValid = true;
			}
			catch (Exception e)
			{
				log.debug("Failed to validate password", e);
			}
		}
		return isValid;
	}
	
	/**
	 @todo move to a generic role mapping function at the base login module
	 */
	private void defaultRole()
	{
		try
		{
			String defaultRole = (String) options.get(DEFAULT_ROLE);
			if (defaultRole == null || defaultRole.equals(""))
			{
				return;
			}
			Principal p = super.createIdentity(defaultRole);
			log.trace("Assign user to role " + defaultRole);
			userRoles.addMember(p);
		}
		catch (Exception e)
		{
			log.debug("could not add default role to user", e);
		}
	}
	
	/**
	 Bind to the ldap server for authentication. 
	 
	 @param username
	 @param credential
	 @return true if the bind for authentication succeeded
	 @throws NamingException
	 */
	private boolean createLdapInitContext(String username, Object credential) throws Exception
	{
		baseDN = (String) options.get(BASE_CTX_DN);
		baseFilter = (String) options.get(BASE_FILTER_OPT);
		roleFilter = (String) options.get(ROLE_FILTER_OPT);
		roleAttributeID = (String) options.get(ROLE_ATTRIBUTE_ID_OPT);
		if (roleAttributeID == null)
			roleAttributeID = "role";
		// Is user's role attribute a DN or the role name
		String roleAttributeIsDNOption = (String) options.get(ROLE_ATTRIBUTE_IS_DN_OPT);
		roleAttributeIsDN = Boolean.valueOf(roleAttributeIsDNOption).booleanValue();
		roleNameAttributeID = (String) options.get(ROLE_NAME_ATTRIBUTE_ID_OPT);
		if (roleNameAttributeID == null)
			roleNameAttributeID = "name";
		rolesCtxDN = (String) options.get(ROLES_CTX_DN_OPT);
		String strRecursion = (String) options.get(ROLE_RECURSION);
		try
		{
			recursion = Integer.parseInt(strRecursion);
		}
		catch (Exception e)
		{
			if (trace)
				log.trace("Failed to parse: " + strRecursion + ", disabling recursion");
			// its okay for this to be 0 as this just disables recursion
			recursion = 0;
		}
		String timeLimit = (String) options.get(SEARCH_TIME_LIMIT_OPT);
		if (timeLimit != null)
		{
			try
			{
				searchTimeLimit = Integer.parseInt(timeLimit);
			}
			catch (NumberFormatException e)
			{
				if (trace)
					log.trace("Failed to parse: " + timeLimit + ", using searchTimeLimit=" + searchTimeLimit);
			}
		}
		String scope = (String) options.get(SEARCH_SCOPE_OPT);
		if ("OBJECT_SCOPE".equalsIgnoreCase(scope))
			searchScope = SearchControls.OBJECT_SCOPE;
		else if ("ONELEVEL_SCOPE".equalsIgnoreCase(scope))
			searchScope = SearchControls.ONELEVEL_SCOPE;
		if ("SUBTREE_SCOPE".equalsIgnoreCase(scope))
			searchScope = SearchControls.SUBTREE_SCOPE;

		InitialLdapContext ctx = null;
		
		try{
			// Get the admin context for searching
			ctx = constructInitialLdapContext();
			// Validate the user by binding against the userDN
			String userDN = bindDNAuthentication(ctx, username, credential, baseDN, baseFilter);
		
			// Query for roles matching the role filter
			SearchControls constraints = new SearchControls();
			constraints.setSearchScope(searchScope);
			constraints.setReturningAttributes(new String[0]);
			constraints.setTimeLimit(searchTimeLimit);
			rolesSearch(ctx, constraints, username, userDN, recursion, 0);
		} finally {
			if(ctx != null){
				ctx.close();
			}
		}
		return true;
	}
	
	/**
	 @param ctx - the context to search from
	 @param user - the input username
	 @param credential - the bind credential
	 @param baseDN - base DN to search the ctx from
	 @param filter - the search filter string
	 @return the userDN string for the successful authentication 
	 @throws NamingException
	 */
	protected String bindDNAuthentication(InitialLdapContext ctx,
			String user, Object credential, String baseDN, String filter)
	throws NamingException
	{
		SearchControls constraints = new SearchControls();
		constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
		constraints.setReturningAttributes(new String[0]);
		constraints.setTimeLimit(searchTimeLimit);
		
		NamingEnumeration results = null;
		
		
		Object[] filterArgs = {user};
		results = ctx.search(baseDN, filter, filterArgs, constraints);
		if (results.hasMore() == false)
		{
			throw new NamingException("Search of baseDN(" + baseDN + ") found no matches");
		}
		
		SearchResult sr = (SearchResult) results.next();
		String name = sr.getName();
		String userDN = null;
		if (sr.isRelative() == true)
			userDN = name + "," + baseDN;
		else
			throw new NamingException("Can't follow referal for authentication: " + name);
		
		Attributes attrs = ctx.getAttributes(userDN);
		byte[] passwordByte = (byte[])attrs.get("userPassword").get(0);
		try {
			String password = new String(passwordByte, "UTF-8");
			if(!password.equals(credential)){
				throw new NamingException("Can't follow referal for authentication: " + name);
			}
		} catch (UnsupportedEncodingException e) {
			throw new NamingException("Can't follow referal for authentication: " + name);
		}
		
		results = null;
		
		return userDN;
	}
	
	private InitialLdapContext constructInitialLdapContext() throws NamingException
	{
		InitialLdapContext ctx = null;
		try {
			//プロバイダLDAP用のコンテキストの作成
			InitialContext iniCtx = new InitialContext();
			ctx = getDirContext("external/hinemos/ldap/provider");
		} catch (NamingException e) {
		    log.debug("LDAP Connection Error : " + e.getMessage());

			try {
				//コンシューマLDAP用のコンテキストの作成
				InitialContext iniCtx = new InitialContext();
				ctx = getDirContext("external/hinemos/ldap/consumer");
			} catch (NamingException e1) {
			    log.debug("LDAP Connection Error : " + e1.getMessage());
			}
		}
		
		return ctx;
	}
	
	/**
	 * 指定されたJNDI名のLDAP用コンテキストを取得します。
	 * 
	 * @param jndiName JNDI名
	 * @return LDAP用コンテキスト
	 * @throws NamingException
	 */
	@SuppressWarnings("static-access")
	private InitialLdapContext getDirContext(String jndiName) throws NamingException {
		InitialLdapContext ctx = null;
		
		//LDAP用のコンテキストの作成
		for (int i = 0; i < RECONNECT_MAX; i++) {
//			TimeKeeper timer = new TimeKeeper(15000); //timeout of 15 secs
			TimeKeeper timer = TimeKeeperService.start();
			try {
				log.debug("getDirContext() : LDAP Connect " + jndiName);
				InitialContext iniCtx = new InitialContext();
				ctx = (InitialLdapContext)iniCtx.lookup(jndiName);
				
				if(ctx != null)
					break;
			} 
			catch (NamingException e) {
				log.error("getDirContext() : LDAP Connection Error " + jndiName + " : " + e.getMessage(), e);
			}
			finally {
//				timer.setLdapResponse(true);
				Thread.currentThread().interrupted();
								
				// タイマースレッドを停止
//				timer.interrupt();
				TimeKeeperService.stop(timer);
			}
		}
		
		if(ctx == null)
			throw new NamingException();
		
		return ctx;
	}
	
//	class TimeKeeper extends Thread {
//		long timeout; //msec
//		long timerStart;
//		
//		boolean ldapResponse = false;
//		Thread parent;
//		
//		TimeKeeper(long timout) {
//			parent = Thread.currentThread();
//			this.timeout = timout;
//			this.start();
//		}
//		
//		@SuppressWarnings("static-access")
//		public void run() {
//			timerStart = System.currentTimeMillis();
//			
//			long waitTime = timeout;
//			
//			while (waitTime > 0) {
//				try {
//					this.sleep(waitTime);
//				} catch (InterruptedException ie) {
//					// ignore and continue
//					if (isLdapResponse()){
//						break;
//					}
//				}
//				waitTime = timeout - (System.currentTimeMillis() - timerStart);
//			}
//			synchronized (this) {
//				if (!isLdapResponse()){
//					log.error("LDAP Connection Timeout. [ " + timeout + " ms ]");
//					parent.interrupt();
//				}
//			}
//		}
//		
//		public boolean isLdapResponse() {
//			return ldapResponse;
//		}
//
//		synchronized public void setLdapResponse(boolean ldapResponse) {
//			this.ldapResponse = ldapResponse;
//		}
//	}
}
