/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.bean;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;

/**
 * ある収集設定に含まれるファシリティのデバイス情報を集めたDTOクラス<P>
 * 
 * 収集プロパティ情報をクライアント側で表示する際には、収集時点のデバイス情報
 * が必要となるので、その情報をセットでやり取りするためのクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class DeviceDataSet implements Serializable {
	private static final long serialVersionUID = -9072139745460507914L;
	
	private HashMap<String, HashMap<String, ArrayList<CollectorDeviceInfoData>>> deviceMap;
	
	/**
	 * コンストラクター
	 *
	 */
	public DeviceDataSet(){
		deviceMap = new HashMap<String, HashMap<String, ArrayList<CollectorDeviceInfoData>>>();
	}
	
	/**
	 * デバイス情報を登録します。
	 * 
	 * @param deviceData　個々のデバイス情報
	 */
	public void putDeviceList(CollectorDeviceInfoData deviceData){
		String facilitID = deviceData.getFacilityId();
		String deviceType = deviceData.getDeviceType();
		
		HashMap<String, ArrayList<CollectorDeviceInfoData>> facilityDeviceMap = 
			getFacilityDeviceMap(facilitID);
		// 指定のファシリティIDでの登録が初めての場合
		if(facilityDeviceMap == null){
			// ファシリティIDをキーに登録
			facilityDeviceMap = new HashMap<String, ArrayList<CollectorDeviceInfoData>>();
			deviceMap.put(facilitID, facilityDeviceMap);
		}
		
		ArrayList<CollectorDeviceInfoData> deviceList = 
			facilityDeviceMap.get(deviceType);
		// 指定のデバイス種別での登録が初めての場合
		if(deviceList == null){
			// デバイス種別をキーに登録
			deviceList = new ArrayList<CollectorDeviceInfoData>();
			facilityDeviceMap.put(deviceType, deviceList);
		}
		// デバイス情報を追加
		deviceList.add(deviceData);
	}
	
	/**
	 * ファシリティが保持していた
	 * デバイス情報のリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * 
	 * @param facilityID 取得したいファシリティ
	 * @param deviceType　デバイス種別 "Disk" or "Nic"
	 * @return　list Disk または Nic のデバイス情報のリスト
	 */
	public List<CollectorDeviceInfoData> getDeviceList(String facilityID, String deviceType){
		HashMap<String, ArrayList<CollectorDeviceInfoData>> facilityDeviceMap = getFacilityDeviceMap(facilityID);
		if(facilityDeviceMap != null){
			return getDeviceList(facilityDeviceMap, deviceType);
		} else {
			return null;
		}
	}
	/**
	 * あるファシリティ（ノード）が保持していたデバイスで
	 * のデバイス種別がdiskのもののリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * @param  facilityID 取得したいファイシリティのID
	 * @return　list　Diskに関するデバイス情報のリスト
	 */
	public List<CollectorDeviceInfoData> getDisk(String facilityID){
		return getDeviceList(facilityID, "disk");
	}
	/**
	 * あるファシリティ（ノード）が保持していたデバイスで
	 * のデバイス種別がnicのもののリストを取得します。
	 * 
	 * 存在しない場合はnullを返す。
	 * 
	 * @param  facilityID 取得したいファイシリティのID
	 * @return　list　NICに関するデバイス情報のリスト
	 */
	public List<CollectorDeviceInfoData> getNic(String facilityID){
		return getDeviceList(facilityID, "nic");
	}
	
	/**
	 * このインスタンスが保持している全てのデバイス情報を返します。
	 * 
	 * @return　list　デバイス情報のリスト
	 */
	public List<CollectorDeviceInfoData> getAll(){
		ArrayList<CollectorDeviceInfoData> deviceList = new ArrayList<CollectorDeviceInfoData>();
		
		Set<String> keys = deviceMap.keySet();
		
		Iterator<String> itr = keys.iterator();
		
		while(itr.hasNext()){
			String facilitID = itr.next();
			HashMap<String, ArrayList<CollectorDeviceInfoData>> facilityDeviceMap = getFacilityDeviceMap(facilitID);
			Collection<ArrayList<CollectorDeviceInfoData>> dl = facilityDeviceMap.values();
			Iterator<ArrayList<CollectorDeviceInfoData>> itrDl = dl.iterator();
			
			while(itrDl.hasNext()){
				deviceList.addAll(itrDl.next());
			}
		}
		
		return deviceList;
	}
	
	/**
	 * ファシリティ毎のデバイス情報を保持するハッシュマップを返します。
	 * 
	 * @param facilityID　取得したいファシリティのID
	 * @return　HashMap ファイシリティに対応するデバイス全体のHashMap
	 */
	private HashMap<String, ArrayList<CollectorDeviceInfoData>> getFacilityDeviceMap(String facilityID){
		return deviceMap.get(facilityID);
	}
	
	/**
	 * デバイスマップから指定のデバイスタイプに合致するもののリストを返します。
	 * 
	 * @param facilityDeviceMap
	 * @param deviceType
	 * @return
	 */
	private ArrayList<CollectorDeviceInfoData> getDeviceList(HashMap<String, ArrayList<CollectorDeviceInfoData>> facilityDeviceMap, String deviceType){
		return facilityDeviceMap.get(deviceType);
	}
}
