/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.job;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;

import com.clustercontrol.performance.bean.QuartzConstant;
import com.clustercontrol.performance.util.JNDIConnectionManager;
import com.clustercontrol.performance.bean.CollectorType;
import com.clustercontrol.performance.ejb.bmp.CollectorLocal;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;
import com.clustercontrol.performance.ejb.bmp.CollectorUtil;

/**
 * Quartzから呼び出され収集を実行するクラス
 *
 */
public class CollectJob implements Job{
	protected static Log m_log = LogFactory.getLog( CollectJob.class );

//	private long m_sleepTime1 = 3000;  // 指定の時間ずらして起動する
//	private long m_sleepTime2 = 10000;  // 指定の時間ずらして起動する
	
	/**
	 * Quartzから呼び出され、リアルタイム、実績、リソース監視の各収集方法に応じてEntityBeanのメソッドを呼び出す。
	 */
	public void execute(JobExecutionContext context) throws JobExecutionException {
		// 引数を取得します
		JobDataMap dataMap = context.getJobDetail().getJobDataMap();
		
		CollectorPK pk  = (CollectorPK)dataMap.get("pk");
		int collectorType  = dataMap.getInt("type");
		
		// デバッグログ出力
		m_log.debug("CollectJob execute : " + pk + "  type : " + collectorType);
		
		// 収集用EntityBeanをルックアップ
		try {
			CollectorLocal bean = null;
			bean = CollectorUtil.getLocalHome().findByPrimaryKey(pk);
			
			if(bean.getCollectorID() == null || 
					collectorType == CollectorType.UNKNOWN){
				// コレクタが正常に動作していない
				m_log.warn("正常に初期化されていない CollectorBean にアクセスしました。");
			} else if(collectorType == CollectorType.REALTIME){
				// リアルタイム収集の場合
				
				// 設定回数まで参照されていない場合は収集を停止する。
				if (!bean.checkLifetime()){
					m_log.debug("参照されていない収集を停止  : " + pk.collectorID);
					bean.stopCollect();
					bean.remove();
					return;
				}
				
				// DBへの出力は行わずポーラーからの値取得のみ行う
				bean.fetchMibValue();
			} else if(collectorType == CollectorType.RECORD) {
				// 実績収集の場合
				
				// ポーラーからの値取得後にDBに出力する
				bean.storeCalcValue();
			} else if(collectorType == CollectorType.MONITOR) {
				// 閾値監視の場合
				
				// ポーラーからの値取得後に閾値判定する
				bean.monitorCalcValue();
			} else {
				// 何もしない
			}			
		} catch (NamingException e) {
			m_log.error(e);
			// リアルタイム収集の場合
			if(collectorType == CollectorType.REALTIME){
				// Quartzの収集ジョブを停止
				deleteJob(pk.getCollectorID());
			}
		} catch (FinderException e) {
			m_log.error(pk.collectorID + " is already stopped.");
			m_log.error(e);
			// リアルタイム収集の場合
			if(collectorType == CollectorType.REALTIME){
				// Quartzの収集ジョブを停止
				deleteJob(pk.getCollectorID());
			}
//		} catch (InterruptedException e) {
//			m_log.error(e);
//			// リアルタイム収集の場合
//			if(collectorType == CollectorType.REALTIME){
//				// Quartzの収集ジョブを停止
//				deleteJob(pk.getCollectorID());
//			}
		} catch (EJBException e) {
			m_log.error(e);
		} catch (RemoveException e) {
			m_log.error(e);
			// リアルタイム収集の場合
			if(collectorType == CollectorType.REALTIME){
				// Quartzの収集ジョブを停止
				deleteJob(pk.getCollectorID());
			}
		}
		
		// デバッグログ出力
//		m_log.debug("execute end   : " + pk);
	}
	
	/**
	 * 収集起動ジョブを削除します。
	 */
	private void deleteJob(String collectorId) {
		m_log.debug("deleteJob() start : " + collectorId);
		
        //QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());

			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
            scheduler.deleteJob(collectorId, QuartzConstant.GROUP_COLLECT);
		} catch (NamingException e) {
			throw new EJBException(e);
        } catch (SchedulerException e) {
            m_log.error(e);
        }
        
        m_log.debug("deleteJob() end   : " + collectorId);
	}
}