/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.factory;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.monitor.run.factory.RunMonitorStringValueType;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoLocal;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoPK;
import com.clustercontrol.snmp.ejb.entity.MonitorSnmpInfoUtil;
import com.clustercontrol.snmp.util.RequestSnmp;
import com.clustercontrol.util.Messages;

/**
 * SNMP監視 文字列監視を実行するファクトリークラス<BR>
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class RunMonitorSnmpString extends RunMonitorStringValueType {
	
	protected static Log m_log = LogFactory.getLog( RunMonitorSnmpString.class );
	
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	/** SNMP監視情報 */
	protected static final ArrayList<String> m_attributeList = new ArrayList<String>();
	static{
		m_attributeList.add(FacilityAttributeConstant.IPPROTOCOLNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
		m_attributeList.add(FacilityAttributeConstant.SNMPTIMEOUT);
		m_attributeList.add(FacilityAttributeConstant.SNMPRETRIES);
	}
	
	/** SNMP監視情報 */
	protected MonitorSnmpInfoLocal m_snmp = null;
	
	/** SNMP情報取得 */
	protected RequestSnmp m_request = null;
	
	/** コミュニティ名 */
	protected String m_communityName = null;
	
	/** ポート番号 */
	protected int m_snmpPort = 161;
	
	/** タイムアウト（ms） リポジトリのノード情報から取得 */
	protected int m_pollingTimeout = 1000;

	/** リトライ回数 リポジトリのノード情報から取得 */
	protected int m_pollingRetries = 3;
	
	/** OID */
	protected String m_snmpOid = null;
	
	/** バージョン */
	protected int m_snmpVersion = 1;
	
	/** オリジナルメッセージ */
	protected String m_messageOrg = null;
	
	/** メッセージID */
	protected String m_messageId = "";
	
	/** メッセージ */
	protected String m_message = "";
	
	/**
	 * コンストラクタ
	 * 
	 */
	public RunMonitorSnmpString() throws NamingException, JMSException, CreateException{
		super();
	}
	
	/**
	 * マルチスレッドを実現するCallableTaskに渡すためのインスタンスを作成するメソッド
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#runMonitorInfo()
	 * @see com.clustercontrol.monitor.run.util.CallableTask
	 */
	public RunMonitor createMonitorInstance() throws NamingException, JMSException, CreateException{
		return new RunMonitorSnmpString();
	}
	
	/**
	 * OID値を取得
	 * 
	 * @param facilityId ファシリティID
	 * @return 値取得に成功した場合、true
	 */
	@Override
	public boolean collect(String facilityId) {
		
		m_nodeDate = m_now.getTime();
		m_value = null;
		
		// メッセージを設定
		m_message = "";
		m_messageOrg = Messages.getString("oid") + " : " + m_snmpOid;
		
		try{
			
			// IPアドレスの取得
			String ipNetworkNumber = null;
			try {
				// ノードの属性取得
				HashMap facilityAttrMap = m_repository.getNodeDetail(facilityId, m_attributeList);
				//バージョンを先に読んでIPv4のIPv6どちらを読みだすかせんたくする。

				Integer ipProtNum = (Integer)facilityAttrMap.get(FacilityAttributeConstant.IPPROTOCOLNUMBER);
				//m_log.error((Integer)facilityAttrMap.get(FacilityAttributeConstant.IPPROTOCOLNUMBER));
				if(ipProtNum != null && ipProtNum == 6){
					 ipNetworkNumber = (String)facilityAttrMap.get(FacilityAttributeConstant.IPNETWORKNUMBERV6);
				} else {
				     ipNetworkNumber = (String)facilityAttrMap.get(FacilityAttributeConstant.IPNETWORKNUMBER);
				}
				
				// SNMPポーリングのタイムアウト
				Integer timeout = (Integer)facilityAttrMap.get(FacilityAttributeConstant.SNMPTIMEOUT);
				if(timeout != null){
					m_log.debug("timeout   : " + timeout);
					m_pollingTimeout = timeout;
				
				}

				// SNMPポーリングのリトライ回数
				Integer retries = (Integer)facilityAttrMap.get(FacilityAttributeConstant.SNMPRETRIES);
				m_log.debug("retries   : " + retries);
				if(retries != null){
					m_pollingRetries = retries;
				}
			}
			catch(FinderException e){
				m_log.debug("run():" + e.getMessage());
	            
	            m_message = Messages.getString("message.snmp.6");
	            m_messageOrg = m_messageOrg + " (" + e.getMessage() + ")";
				return false;
			}
			catch(NamingException e){
				m_log.debug("run():" + e.getMessage());
	            
	            m_message = Messages.getString("message.snmp.6");
	            m_messageOrg = m_messageOrg + " (" + e.getMessage() + ")";
				return false;
			}
			
			// SNMP値取得
			RequestSnmp m_request = new RequestSnmp();

			boolean result = m_request.polling(
					InetAddress.getByName(ipNetworkNumber), 
					m_communityName, 
					m_snmpPort, 
					m_snmpOid, 
					m_snmpVersion,
					m_pollingTimeout,
					m_pollingRetries);
			
			if(result){
				
				m_value = m_request.getValue();
				m_nodeDate = m_request.getDate();

				m_messageOrg = m_messageOrg + ", " + Messages.getString("select.value") + " : " + m_value;
			}
			else{
				m_message = m_request.getMessage();
			}
			return result;
		}
		catch (UnknownHostException e) {
			m_log.debug("run():" + e.getMessage());

			m_message = Messages.getString("message.snmp.5");
			m_messageOrg = m_messageOrg + " (" + e.getMessage() + ")";
			return false;
		}
	}
	
	/* (非 Javadoc)
	 * SNMP監視情報を設定
	 * @see com.clustercontrol.monitor.run.factory.OperationNumericValueInfo#setMonitorAdditionInfo()
	 */
	@Override
	public void setCheckInfo() throws FinderException, NamingException{
		
		// SNMP監視情報を取得
		MonitorSnmpInfoPK pk = new MonitorSnmpInfoPK(m_monitorId, m_monitorTypeId);
		m_snmp = MonitorSnmpInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// SNMP監視情報を設定
		m_communityName = m_snmp.getCommunityName().trim();
		m_snmpPort = m_snmp.getSnmpPort().intValue();
		m_snmpOid = m_snmp.getSnmpOid().trim();
		m_snmpVersion = SnmpVersionConstant.stringToType(m_snmp.getSnmpVersion().trim()).intValue();
		
	}

	/* (非 Javadoc)
	 * ノード用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageId()
	 */
	@Override
	public String getMessageId(int id) {
		
		String messageId = super.getMessageId(id);
		if(messageId == null || "".equals(messageId)){
			return MESSAGE_ID_UNKNOWN;
		}
		return messageId;
	}
	
	/* (非 Javadoc)
	 * ノード用メッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessage(int)
	 */
	@Override
	public String getMessage(int id) {
		
		String message = super.getMessage(id);
		if(message == null || "".equals(message)){
			return m_message;
		}
		return message;
	}

	/* (非 Javadoc)
	 * ノード用オリジナルメッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageOrg(int)
	 */
	@Override
	public String getMessageOrg(int id) {
		return m_messageOrg;
	}
	
	/* (非 Javadoc)
	 * スコープ用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#getMessageIdForScope(int)
	 */
	@Override
	public String getMessageIdForScope(int priority) {
		
		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
}
