/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.action;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.widgets.Composite;

import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.util.FacilityTreeItemUtil;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * スコープツリーアイテムを取得するクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class FacilityTree {
	// ログ
	private static Log m_log = LogFactory.getLog( FacilityTree.class );

	// ファシリティツリーのクライアント側ローカルキャッシュを保持
	private static FacilityTreeItem m_facilityTreeCache = null;

	// ファシリティツリーを表示しているビューComposite
	private static final Set<Composite> m_DisplayViewSet = new HashSet<Composite>();

	/**
	 * スコープツリー構造を返します。
	 * 
	 * @return スコープツリー構造
	 */
	public FacilityTreeItem getTreeItem() {
		// m_log.debug("FacilityTree.getTreeItem() start : scopeOnly =  " + scopeOnly);

		FacilityTreeItem item = null;

		if (m_facilityTreeCache == null) {
			// 存在しない場合はキャッシュ再交信再取得
			refleshCache();
		}
		// 要素を取得
		item = FacilityTreeItemUtil.deepCopy(m_facilityTreeCache, null);

		//m_log.debug("FacilityTree.getTreeItem() stop : scopeOnly =  " + scopeOnly);
		return item;
	}

	/**
	 * セパレータを取得します。<BR>
	 * 
	 * スコープの区切り文字（セパレータ）を取得します。
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		String separator = ">";
		return separator;
	}

	/**
	 * 保持しているローカルキャッシュをクリアして画面を再描画する
	 */
	public static void clearCache() {
		//m_log.debug("FacilityTree.clearCache() start");

		// キャッシュ情報を削除する
		//m_log.debug("FacilityTree.clearCache() : clear cache");

		m_facilityTreeCache = null;

		//m_log.debug("FacilityTree.clearCache() stop");
	}

	/**
	 * キャッシュおよび画面のリフレッシュ
	 */
	public static void reflesh() {
		//m_log.debug("FacilityTree.reflesh() start");

		refleshCache();
		refleshComposite();

		//m_log.debug("FacilityTree.reflesh() stop");
	}

	/**
	 * キャッシュのリフレッシュ
	 */
	public static void refleshCache() {
		//m_log.debug("FacilityTree.refleshCache() : start");

		// 先にキャッシュクリア
		// clearCache() ;

		// 再登録
		FacilityTreeItem item = null;
		try {
			item = RepositoryEndpointWrapper.getFacilityTree();

			if (item != null) {
				m_facilityTreeCache = item;
			}

		} catch (Exception e) {
			m_log.warn("refleshCache(), " + e.getMessage(), e);
			return;
		}

		//m_log.debug("FacilityTree.refleshCache() : stop");
	}

	/**
	 * ファシリティツリー画面を再描画する
	 */
	public static void refleshComposite() {
		//m_log.debug("FacilityTree.refleshComposite() : start");

		synchronized (m_DisplayViewSet) {

			Iterator<Composite> iterator = m_DisplayViewSet.iterator();
			while (iterator.hasNext()) {
				FacilityTreeComposite composite = (FacilityTreeComposite) iterator
				.next();
				if (!composite.isDisposed()) {
					//m_log.debug("FacilityTree.refleshComposite() : target is "
					//+ composite.getClass().getCanonicalName());
					composite.update();
				}
			}
		}

		//m_log.debug("refleshComposite() : stop");
	}

	/**
	 * キャッシュ更新時にリフレッシュするCompositeを一覧に追加
	 * 
	 * @param composite
	 */
	public void addComposite(FacilityTreeComposite composite) {
		//m_log.debug("FacilityTree.addComposite() : start");

		synchronized (m_DisplayViewSet) {
			//m_log.debug("FacilityTree.addComposite()  composite is "
			//+ composite.getClass().getCanonicalName());
			m_DisplayViewSet.add(composite);
		}

		//m_log.debug("FacilityTree.addComposite() : stop");
	}

	/**
	 * キャッシュ更新時にリフレッシュするCompositeを一覧から削除
	 * 
	 * @param composite
	 */
	public void delComposite(FacilityTreeComposite composite) {
		//m_log.debug("FacilityTree.delComposite() : start");

		synchronized (m_DisplayViewSet) {
			m_DisplayViewSet.remove(composite);
		}
		//m_log.debug("FacilityTree.delComposite() : stop");
	}
}