/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.rmi.AccessException;
import java.sql.Timestamp;

import javax.naming.CommunicationException;
import javax.naming.NamingException;
import javax.security.auth.login.LoginException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.StatusLineContributionItem;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.WorkbenchWindow;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.accesscontrol.dialog.LoginDialog;
import com.clustercontrol.accesscontrol.etc.action.LogoutAction;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.preference.ClusterControlCorePreferencePage;

/**
 * ログインマネージャクラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class LoginManager {
	private static Log m_log = LogFactory.getLog( LoginManager.class );

	public static final String KEY_EJB_URL = "ejbUrl";
	public static final String KEY_EJB_URL_NUM = "numOfUrlHistory";
	public static final String VALUE_EJB_URL = "http://localhost:8080/HinemosWS/";
	public static final String KEY_EJB_UID = "ejbUid";
	public static final String VALUE_EJB_UID = "hinemos";
	public static final String KEY_INTERVAL = "managerPollingInterval";
	public static final int VALUE_INTERVAL = 60;

	public static final String USER_OBJECT_ID = "com.clustercontrol.accesscontrol.user";
	public static final String URL_OBJECT_ID = "com.clustercontrol.accesscontrol.url";

	public static final String ACTION_SET_ID    = "com.clustercontrol.accesscontrol.ActionSet";
	public static final String ACTION_ID_LOGIN  = "com.clustercontrol.accesscontrol.etc.action.LoginAction";
	public static final String ACTION_ID_LOGOUT = "com.clustercontrol.accesscontrol.etc.action.LogoutAction";

	public static final String KEY_HTTP_CONNECT_TIMEOUT = "httpConnectTimeout";
	public static final int VALUE_HTTP_CONNECT_TIMEOUT = 10;
	public static final String KEY_HTTP_REQUEST_TIMEOUT = "httpRequestTimeout";
	public static final int VALUE_HTTP_REQUEST_TIMEOUT = 60;

	private static LoginManager m_instance = null;
	private String m_url = null;
	private String m_uid = null;
	private String m_password = null;

	private Display m_dislpay = null;
	private IStatusLineManager m_lineManager = null;

	/** ダイアログ表示制御用 */
	private static boolean openInformation = false;

	/**
	 * このオブジェクトを取得します。 シングルトン
	 * 
	 * @return LoginManager ログインマネージャ
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public static LoginManager getContextManager() {
		if (m_instance == null) {
			m_instance = new LoginManager();
		}
		return m_instance;
	}

	/**
	 * コンストラクタ
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private LoginManager() {
	}

	/**
	 * ログイン(接続先URLあり)
	 * 
	 * @param uid
	 * @param password
	 * @throws NamingException
	 * @throws LoginException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized void login(String uid, String password, String url) throws Exception {
		//ログアウト
		logout();

		m_uid = uid;
		m_password = password;
		m_url = url;


		IPreferenceStore store = ClusterControlPlugin.getDefault().getPreferenceStore();
		int httpConnectTimeout = store.getInt(KEY_HTTP_CONNECT_TIMEOUT);
		int httpRequestTimeout = store.getInt(KEY_HTTP_REQUEST_TIMEOUT);
		m_log.info("connect.timeout=" + httpConnectTimeout + ", " +
				"request.timeout=" + httpRequestTimeout);

		EndpointManager.init(uid, password, url, httpConnectTimeout, httpRequestTimeout);
		try {
			//ログインチェック
			AccessEndpointWrapper.checkLogin();

			// ステータスバーへユーザIDを登録
			IWorkbench workbench = PlatformUI.getWorkbench();
			WorkbenchWindow workbenchWindow = (WorkbenchWindow)workbench.getActiveWorkbenchWindow();
			IActionBars bars = workbenchWindow.getActionBars();
			m_lineManager = bars.getStatusLineManager();
			m_dislpay = workbench.getDisplay();

			StatusLineContributionItem statusLine = new StatusLineContributionItem(USER_OBJECT_ID,100);
			statusLine.setText(Messages.getString("hinemos.user") + " : " + m_uid +
					"     " +
					Messages.getString("connection.url") + " : " + m_url);

			m_lineManager.add(statusLine);
			m_lineManager.update(true);

			//UIDと接続先をプレファレンスに書き戻す。
			//セッションに接続先URLを格納
			store.setValue(ClusterControlCorePreferencePage.EJB_URL,url);
			store.setValue(ClusterControlCorePreferencePage.EJB_UID,uid);

			//今回の接続先URLがURL履歴に存在するかチェックする。
			int numOfUrlHistory = store.getInt(LoginManager.KEY_EJB_URL_NUM);
			boolean urlExist = false;
			for(int i=0; i<numOfUrlHistory; i++){
				String histUrl = store.getString(LoginManager.KEY_EJB_URL + "_" + i);
				if(url.equals(histUrl)){
					//TODO 存在する場合、対象のURLを履歴の先頭に移動させ、他のURLを一つずつ後方にずらす
					urlExist = true;
					break;
				}
			}
			//存在しない場合、URL履歴の末尾に追加
			if(!urlExist && !url.equals("")){
				store.setValue(LoginManager.KEY_EJB_URL + "_" + numOfUrlHistory ,url);
				numOfUrlHistory++;
				store.setValue(LoginManager.KEY_EJB_URL_NUM, numOfUrlHistory);
			}

			// クライアント側で保持するローカル情報の削除処理
			LocalSession.getLocalSession().clearSessionData();
			// マネージャの死活監視開始
			LocalSession.getLocalSession().startChecktask(getInterval());

		} catch (Exception e) {
			logout();
			throw e;
		}
	}


	/**
	 * ログアウト
	 * 
	 * @throws NamingException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized void logout() throws NamingException {
		EndpointManager.logout();
		m_uid = null;
		m_url = null;

		// ステータスバーからユーザIDを削除
		if(m_lineManager != null){
			IContributionItem item = m_lineManager.find(USER_OBJECT_ID);
			m_lineManager.remove(item);
			m_lineManager.update(true);
		}

		LocalSession.getLocalSession().clearSessionData();
		LocalSession.getLocalSession().stopChecktask();
	}

	/**
	 * ログインチェック
	 * 
	 * @return
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public boolean isLogin() {
		if (m_uid != null) {
			return true;
		}

		return false;
	}

	public synchronized void getNamingContext() throws NamingException {
		if(!isLogin()){

			Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

			//ログインダイアログ表示
			LoginDialog dialog = new LoginDialog(shell);

			//ダイアログ表示
			if (dialog.open() == IDialogConstants.OK_ID) {
				try {

					//	login(dialog.getUserid(), dialog.getPassword());
					login(dialog.getUserid(), dialog.getPassword(),dialog.getUrl());

					// ログイン成功ダイアログを生成
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.accesscontrol.5"));

					m_log.info("Login Success : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
					m_log.info("Hinemos Client started");
				} catch (CommunicationException e) {
					// 接続失敗ダイアログを生成
					Status status = new Status(
							IStatus.ERROR,
							ClusterControlPlugin.getPluginId(),
							IStatus.OK,
							Messages.getString("message.accesscontrol.22"),
							e);

					ErrorDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.accesscontrol.21"),
							status);
					m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				} catch (AccessException e) {
					// ログイン失敗ダイアログを生成
					MessageDialog.openWarning(
							null,
							Messages.getString("failed"),
							Messages.getString("message.accesscontrol.6"));
					m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				} catch (Exception e) {
					// 予期せぬエラーダイアログを生成
					Status status = new Status(
							IStatus.ERROR,
							ClusterControlPlugin.getPluginId(),
							IStatus.OK,
							Messages.getString("message.accesscontrol.23"),
							e);

					ErrorDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.accesscontrol.6"),
							status
					);
					m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				}
			}
		}
		return;
	}

	/**
	 * 接続先URLを取得します。
	 * 
	 * @return String 接続先URL
	 * @version 4.0.0
	 * @since 2.0.0
	 */
	public static String getUrl() {

		//リソースストアから接続先URLを取得
		IPreferenceStore store = ClusterControlPlugin.getDefault()
		.getPreferenceStore();
		String url = store.getString(KEY_EJB_URL);
		if (url.compareTo("") == 0) {
			url = VALUE_EJB_URL;
		}

		return url;
	}

	/**
	 * 
	 * @return
	 */
	public static int getInterval() {

		//リソースストアから接続先の死活間隔を取得
		IPreferenceStore store = ClusterControlPlugin.getDefault()
		.getPreferenceStore();
		int interval = store.getInt(KEY_INTERVAL);

		//m_log.debug("LoginManager.getInterval() interval = " + interval);
		return interval;
	}

	/**
	 * ユーザIDを取得します。
	 * 
	 * @return String ユーザID
	 * @version 4.0.0
	 */
	public String getUserId() {
		return m_uid;
	}

	/**
	 * パスワードを取得します。
	 * 
	 * @return String パスワード
	 * @version 4.0.0
	 */
	public String getPassword() {
		return m_password;
	}

	/**
	 * LocalSession.SessionCheckTaskからのダイアログ&ログアウト処理を受け付けるスレッド生成用メソッド
	 * @param r
	 * @return
	 */
	private boolean checkAsyncExec(Runnable r){

		if(m_dislpay == null){
			//m_log.debug("LoginManager.checkAsyncExec() m_dislpay is null");
			return false;
		}

		if(!m_dislpay.isDisposed()){
			//m_log.debug("LoginManager.checkAsyncExec() is true");
			m_dislpay.asyncExec(r);
			return true;
		}
		else{
			//m_log.debug("LoginManager.checkAsyncExec() is false");
			return false;
		}
	}


	/**
	 * マネージャ切断をダイアログ表示してログアウト処理を行う
	 */
	public void disconnectManager(){
		checkAsyncExec(new Runnable(){@Override
			public void run() {
			// ログアウトダイアログの表示
			//m_log.debug("LoginManager.disconnectManager() start openInformation = " + openInformation);
			if(!openInformation){
				//m_log.debug("LoginManager.disconnectManager() OpenDialog");

				openInformation = true;

				// ダイアログ表示
				String[] args = {new Timestamp(System.currentTimeMillis()).toString()};
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.force.disconnect",args));

				// ログアウト処理
				(new LogoutAction()).runLogout();

				openInformation = false;
			}else{
				//m_log.debug("LoginManager.disconnectManager() Not OpenDialog");
			}
			//m_log.debug("LoginManager.disconnectManager() stop openInformation = " + openInformation);
		}}
		);

	}

	/**
	 * 定期的な監視を再実行
	 * @param interval 間隔(分)
	 */
	public static void restartChecktask(int interval){
		//m_log.debug("LoginManager.restartChecktask() interval = " + interval);
		LocalSession.getLocalSession().restartChecktask(interval);
	}
}