/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.composite;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;

import com.clustercontrol.collectiverun.bean.ItemConstant;
import com.clustercontrol.collectiverun.composite.action.CopyTreeItemAction;
import com.clustercontrol.collectiverun.composite.action.CreateTreeItemSelectionListener;
import com.clustercontrol.collectiverun.composite.action.DeleteTreeItemSelectionListener;
import com.clustercontrol.collectiverun.composite.action.UpDownTreeItemSelectionListener;
import com.clustercontrol.collectiverun.composite.action.UpdateTreeItemSelectionListener;
import com.clustercontrol.collectiverun.util.CollectiveRunEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.collectiverun.InvalidRole_Exception;
import com.clustercontrol.ws.collectiverun.ItemInfo;
import com.clustercontrol.ws.collectiverun.TreeItem;

/**
 * 一括制御階層ツリーコンポジット
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class TreeComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( TreeComposite.class );

	private TreeViewer m_viewer = null;

	private TreeItem selectItem = null;

	private TreeItem createItem = null;

	private TreeItem updateItem = null;

	public TreeComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	private void initialize() {
		GridLayout layout = new GridLayout(5, true);
		this.setLayout(layout);
		layout.marginHeight = 5;
		layout.marginWidth = 5;

		GridData gridData = null;

		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 5;
		tree.setLayoutData(gridData);

		// 参照ボタン
		Button buttonCreate = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		buttonCreate.setLayoutData(gridData);
		buttonCreate.setText(Messages.getString("create"));
		buttonCreate.addSelectionListener(new CreateTreeItemSelectionListener(this));

		Button buttonDelete = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		buttonDelete.setLayoutData(gridData);
		buttonDelete.setText(Messages.getString("delete"));
		buttonDelete.addSelectionListener(new DeleteTreeItemSelectionListener(this));

		Button buttonUp = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		buttonUp.setLayoutData(gridData);
		buttonUp.setText(Messages.getString("up"));
		buttonUp.addSelectionListener(new UpDownTreeItemSelectionListener(this, true));

		Button buttonDown = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		buttonDown.setLayoutData(gridData);
		buttonDown.setText(Messages.getString("down"));
		buttonDown.addSelectionListener(new UpDownTreeItemSelectionListener(this, false));

		Button buttonUpdate = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		buttonUpdate.setLayoutData(gridData);
		buttonUpdate.setText(Messages.getString("update"));
		buttonUpdate.addSelectionListener(new UpdateTreeItemSelectionListener(this));

		m_viewer = new TreeViewer(tree);
		m_viewer.setContentProvider(new TreeContentProvider());
		m_viewer.setLabelProvider(new TreeLabelProvider());
		m_viewer.setSorter(new TreeViewerSorter());

		//ポップアップメニュー作成
		createContextMenu();

		update();
	}

	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	public TreeViewer getTreeViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	public Tree getTree() {
		return m_viewer.getTree();
	}

	/**
	 * 更新処理
	 * 
	 */
	@Override
	public void update() {
		TreeItem tree = null;
		selectItem = null;
		createItem = null;
		updateItem = null;

		//一括制御情報取得
		try {
			tree = CollectiveRunEndpointWrapper.getTree();

		} catch (InvalidRole_Exception e) {
			// 権限なし
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));

		} catch (Exception e) {
			// 上記以外の例外
			m_log.warn("update(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());

		}

		m_viewer.setInput(tree);

		//ジョブユニットのレベルまで展開
		m_viewer.expandToLevel(2);
	}

	/**
	 * ポップアップメニュー作成
	 * 
	 */
	protected void createContextMenu() {
		MenuManager menuManager = new MenuManager();
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
			 */
			@Override
			public void menuAboutToShow(IMenuManager manager) {

				//メニュー項目設定
				fillContextMenu(manager);
			}
		});
		Menu treeMenu = menuManager.createContextMenu(getTree());
		getTree().setMenu(treeMenu);
	}

	/**
	 * メニュー項目追加
	 * 
	 * @param manager
	 */
	protected void fillContextMenu(IMenuManager manager) {
		//ポップアップメニューにアクションを追加
		CopyTreeItemAction copyAction = new CopyTreeItemAction(this);
		manager.add(copyAction);

		//コピーアクションの使用可/不可を設定
		TreeItem select = getSelectItem();
		if (select instanceof TreeItem) {
			//コピーアクションの使用可/不可を設定
			if(getInfo(select).getType() == ItemConstant.TYPE_COMPOSITE ||
					getInfo(select).getType() == ItemConstant.TYPE_COMMAND_COMPOSITE ||
					getInfo(select).getType() == ItemConstant.TYPE_PARAMETER_COMPOSITE ||
					createItem != null ||
					updateItem != null){
				copyAction.setEnabled(false);
			}
			else{
				copyAction.setEnabled(true);
			}
		}
	}

	/**
	 * 現在選択されているツリーアイテムを返します。
	 * 
	 * @return ツリーアイテム
	 */
	public TreeItem getSelectItem() {
		return selectItem;
	}

	/**
	 * 現在選択されているツリーアイテムを設定
	 * 
	 * @param item
	 */
	public void setSelectItem(TreeItem selectItem) {
		this.selectItem = selectItem;
	}

	/**
	 * 現在作成されているツリーアイテムを返します。
	 * 
	 * @return
	 */
	public TreeItem getCreateItem() {
		return createItem;
	}

	/**
	 * 現在作成されているツリーアイテムを設定
	 * 
	 * @param createItem
	 */
	public void setCreateItem(TreeItem createItem) {
		this.createItem = createItem;
	}

	/**
	 * 現在更新されているツリーアイテムを返します。
	 * 
	 * @return
	 */
	public TreeItem getUpdateItem() {
		return updateItem;
	}

	/**
	 * 現在更新されているツリーアイテムを設定
	 * 
	 * @param updateItem
	 */
	public void setUpdateItem(TreeItem updateItem) {
		this.updateItem = updateItem;
	}

	private ItemInfo getInfo(TreeItem item){
		if(item.getCommandInfo() != null){
			return item.getCommandInfo();
		}
		else if(item.getCommandParameterInfo() != null){
			return item.getCommandParameterInfo();
		}
		else if(item.getParameterInfo() != null){
			return item.getParameterInfo();
		}
		else if(item.getParameterSelectInfo() != null){
			return item.getParameterSelectInfo();
		}
		else if(item.getTypeInfo() != null){
			return item.getTypeInfo();
		}
		else{
			return null;
		}
	}
}