/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.view.action;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.monitor.bean.ConfirmConstant;
import com.clustercontrol.monitor.composite.EventListComposite;
import com.clustercontrol.monitor.util.ConvertListUtil;
import com.clustercontrol.monitor.util.MonitorEndpointWrapper;
import com.clustercontrol.monitor.view.EventView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.EventDataInfo;
import com.clustercontrol.ws.monitor.HinemosUnknown_Exception;
import com.clustercontrol.ws.monitor.InvalidRole_Exception;
import com.clustercontrol.ws.monitor.MonitorNotFound_Exception;

/**
 * 監視[イベント]ビューの確認アクションによるイベントの確認の更新処理を行うアクライアント側アクションクラス<BR>
 * 
 * @version 2.0.0
 * @since 1.0.0
 */
public class EventConfirmAction implements IViewActionDelegate {

	// ログ
	private static Log m_log = LogFactory.getLog( EventConfirmAction.class );

	/** アクションID */
	public static final String ID = "com.clustercontrol.monitor.view.action.EventConfirmAction";

	/** ビュー */
	private IViewPart viewPart;


	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart viewPart) {
		this.viewPart = viewPart;
	}

	/**
	 * 監視[イベント]ビューの選択されたアイテムを確認に更新し、ビューを更新します。
	 * <p>
	 * <ol>
	 * <li>監視[イベント]ビューで、選択されているアイテムを取得します。</li>
	 * <li>取得したイベント情報の確認を確認済みに更新します。 </li>
	 * <li>監視[イベント]ビューを更新します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.monitor.view.EventView
	 * @see com.clustercontrol.monitor.view.EventView#update()
	 */
	@Override
	public void run(IAction action) {
		// 選択アイテムを取得します。
		EventView view = (EventView) this.viewPart.getAdapter(EventView.class);

		EventListComposite composite = (EventListComposite) view
		.getListComposite();
		StructuredSelection selection = (StructuredSelection) composite
		.getTableViewer().getSelection();

		List list = selection.toList();

		// 選択しているイベント情報を「確認済」にして、再描画します。
		if (list != null) {
			ArrayList<EventDataInfo> eventInfoList = ConvertListUtil.listToEventLogDataList(list);

			if (eventInfoList != null && eventInfoList.size()>0) {
				try {
					MonitorEndpointWrapper.modifyConfirm(eventInfoList, ConfirmConstant.TYPE_CONFIRMED);
					view.update();
				} catch (InvalidRole_Exception e) {
					// アクセス権なしの場合、エラーダイアログを表示する
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} catch (MonitorNotFound_Exception e) {
					MessageDialog.openError(null, Messages.getString("message"),
							Messages.getString("mesasge.monitor.60") + ", " + e.getMessage());
				} catch (HinemosUnknown_Exception e) {
					MessageDialog.openError(null, Messages.getString("message"),
							Messages.getString("mesasge.monitor.60") + ", " + e.getMessage());
				} catch (Exception e) {
					m_log.warn("run(), " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}

			}
		}
	}

	/**
	 * 選択を変更した際に呼ばれます。
	 * 
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction,
	 *      org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
	}
}