/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.bean.EventConfirmConstant;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.notify.NotifyInfo;
import com.clustercontrol.ws.notify.NotifyEventInfo;

/**
 * 通知（イベント）作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 3.0.0
 */
public class NotifyEventCreateDialog extends NotifyBasicCreateDialog {

	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 		= 1;

	/** カラム数（通知）。 */
	private static final int WIDTH_NOTIFY	 		= 1;

	/** カラム数（状態）。 */
	private static final int WIDTH_STATE		= 3;

	/** カラム数（抑制）。 */
	private static final int WIDTH_INHIBITION_FLG 	= 2;

	/** カラム数（空欄）。 */
	private static final int WIDTH_BLANK 			= 5;

	// ----- instance フィールド ----- //

	/** 通知タイプ
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final int TYPE_EVENT = 1;

	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;

	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;

	/** 通知（通知） チェックボックス。 */
	private Button m_checkEventNormalInfo = null;
	/** 通知（警告） チェックボックス。 */
	private Button m_checkEventNormalWarning = null;
	/** 通知（異常） チェックボックス。 */
	private Button m_checkEventNormalCritical = null;
	/** 通知（不明） チェックボックス。 */
	private Button m_checkEventNormalUnknown = null;

	/** 通知状態（通知） ボタン。 */
	private Combo m_comboEventNormalInfo = null;
	/** 通知状態（警告） ボタン。 */
	private Combo m_comboEventNormalWarning = null;
	/** 通知状態（異常） ボタン。 */
	private Combo m_comboEventNormalCritical = null;
	/** 通知状態（不明） ボタン。 */
	private Combo m_comboEventNormalUnknown = null;

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyEventCreateDialog(Shell parent) {
		super(parent);
	}

	public NotifyEventCreateDialog(Shell parent, int notifyIdType) {
		super(parent, notifyIdType);
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyEventCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
	@Override
	protected void customizeDialog(Composite parent) {

		super.customizeDialog(parent);

		// 通知IDが指定されている場合、その情報を初期表示する。
		NotifyInfo info = null;
		if(this.notifyId != null){
			info = new GetNotify().getNotify(this.notifyId);
		}else{
			info = new NotifyInfo();
		}
		this.setInputData(info);

	}

	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
	@Override
	protected void customizeSettingDialog(Composite parent) {
		final Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.notify.event.create.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * イベント
		 */
		// イベントグループ
		Group groupEvent = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupEvent.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupEvent.setLayoutData(gridData);
		groupEvent.setText(Messages.getString("notifies.event"));

		/*
		 * 重要度 ごとの設定
		 */
		// ラベル（重要度）
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority"));

		// ラベル（通知）
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_NOTIFY;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.attribute"));

		// ラベル（状態）
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("status"));

		// 空欄
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG + WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　イベント　重要度：通知
		label = this.getLabelPriority(groupEvent, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
		this.m_checkEventNormalInfo = this.getCheckEventNormal(groupEvent);
		this.m_comboEventNormalInfo = this.getComboEventNormal(groupEvent);

		// 空欄
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG + WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　イベント　重要度：警告
		label = this.getLabelPriority(groupEvent, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
		this.m_checkEventNormalWarning = this.getCheckEventNormal(groupEvent);
		this.m_comboEventNormalWarning = this.getComboEventNormal(groupEvent);

		// 空欄
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG + WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　イベント　重要度：危険
		label = this.getLabelPriority(groupEvent, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
		this.m_checkEventNormalCritical = this.getCheckEventNormal(groupEvent);
		this.m_comboEventNormalCritical = this.getComboEventNormal(groupEvent);

		// 空欄
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG + WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　イベント　重要度：不明
		label = this.getLabelPriority(groupEvent, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
		this.m_checkEventNormalUnknown = this.getCheckEventNormal(groupEvent);
		this.m_comboEventNormalUnknown = this.getComboEventNormal(groupEvent);

		// 空欄
		label = new Label(groupEvent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG + WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

	}

	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
	@Override
	public NotifyInfo getInputData() {
		return this.inputData;
	}

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
	@Override
	protected void setInputData(NotifyInfo notify) {

		this.inputData = notify;

		// 通知基本情報
		super.m_notifyBasic.setInputData(notify);

		// 初回通知情報
		super.m_notifyInitial.setInputData(notify);

		// 抑制情報
		super.m_notifyInhibition.setInputData(notify);

		// イベント情報
		List<NotifyEventInfo> list = notify.getNotifyEventInfo();
		if (list != null && list.size() > 0) {
			for(int index=0; index<list.size(); index++){
				NotifyEventInfo event = list.get(index);
				if(event != null){
					// 通知の場合
					if(PriorityConstant.TYPE_INFO ==  event.getPriority()){
						this.setInputDataForEvent(event, this.m_checkEventNormalInfo, 	this.m_comboEventNormalInfo);
					}
					// 警告の場合
					else if(PriorityConstant.TYPE_WARNING ==  event.getPriority()){
						this.setInputDataForEvent(event, this.m_checkEventNormalWarning, 	this.m_comboEventNormalWarning);
					}
					// 危険の場合
					else if(PriorityConstant.TYPE_CRITICAL ==  event.getPriority()){
						this.setInputDataForEvent(event, this.m_checkEventNormalCritical, this.m_comboEventNormalCritical);
					}
					// 不明の場合
					else if(PriorityConstant.TYPE_UNKNOWN ==  event.getPriority()){
						this.setInputDataForEvent(event, this.m_checkEventNormalUnknown, 	this.m_comboEventNormalUnknown);
					}
				}
			}
		}

		// 有効／無効
		if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
			this.m_confirmValid.setSelection(false);
		}
		else {
			this.m_confirmValid.setSelection(true);
		}

		// 入力制御
		if(this.referenceFlg){
			this.setEnabled(false);
		}
		else{
			this.setEnabled(true);
		}

	}

	/**
	 * 引数で指定された通知イベント情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知イベント情報
	 * @param checkEventNormal 通知チェックボックス
	 * @param comboEventNormal 通知状態コンボボックス
	 * @param checkEventInhibition 抑制チェックボックス
	 * @param comboEventInhibition 抑制状態コンボボックス
	 */
	private void setInputDataForEvent(NotifyEventInfo info,
			Button checkEventNormal,
			Combo comboEventNormal
	) {
		// イベント通知
		if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
			checkEventNormal.setSelection(true);
		}
		// イベント画面出力時の通知状態
		if (info.getEventNormalState() != null) {
			comboEventNormal.setText(EventConfirmConstant.typeToString(info.getEventNormalState().intValue()));
		}
	}

	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForEvent(ArrayList, int, Button, Combo, Button, Combo)
	 */
	@Override
	protected NotifyInfo createInputData() {
		NotifyInfo info = new NotifyInfo();

		// 通知タイプの設定
		info.setNotifyType(TYPE_EVENT);

		// 通知基本情報
		validateResult = super.m_notifyBasic.createInputData(info, this.notifyId);
		if(validateResult != null){
			return null;
		}

		// 初回通知情報
		validateResult = super.m_notifyInitial.createInputData(info);
		if(validateResult != null){
			return null;
		}

		// 抑制情報
		validateResult = super.m_notifyInhibition.createInputData(info);
		if(validateResult != null){
			return null;
		}

		// イベント情報
		ArrayList<NotifyEventInfo> list = new ArrayList<NotifyEventInfo>();
		list = this.createInputDataForEvent(list, PriorityConstant.TYPE_INFO,		this.m_checkEventNormalInfo,	this.m_comboEventNormalInfo, null, null);
		list = this.createInputDataForEvent(list, PriorityConstant.TYPE_WARNING,	this.m_checkEventNormalWarning,	this.m_comboEventNormalWarning, null, null);
		list = this.createInputDataForEvent(list, PriorityConstant.TYPE_CRITICAL,	this.m_checkEventNormalCritical,this.m_comboEventNormalCritical, null, null);
		list = this.createInputDataForEvent(list, PriorityConstant.TYPE_UNKNOWN,	this.m_checkEventNormalUnknown,	this.m_comboEventNormalUnknown, null, null);

		if(list == null){
			return null;
		}

		List<NotifyEventInfo> notifyEventInfo = info.getNotifyEventInfo();
		notifyEventInfo.clear();
		notifyEventInfo.addAll(list);

		// 有効/無効
		if (this.m_confirmValid.getSelection()) {
			info.setValidFlg(ValidConstant.TYPE_VALID);
		} else {
			info.setValidFlg(ValidConstant.TYPE_INVALID);
		}

		return info;
	}

	/**
	 * 入力値を設定した通知イベント情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知イベント情報
	 * 
	 */
	protected ArrayList<NotifyEventInfo> createInputDataForEvent(ArrayList<NotifyEventInfo> list,
			int priority,
			Button checkEventNormal,
			Combo comboEventNormal,
			Button checkEventInhibition,
			Combo comboEventInhibition
	) {

		if(list == null)
			return null;

		NotifyEventInfo event = new NotifyEventInfo();
		event.setEventNormalState(EventConfirmConstant.TYPE_UNCONFIRMED);

		// 重要度
		event.setPriority(priority);

		// イベント通知
		if (checkEventNormal.getSelection()) {
			event.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
		} else {
			event.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
		}

		// イベント通知時の状態
		if (comboEventNormal.getText() != null
				&& !"".equals((comboEventNormal.getText()).trim())) {
			event.setEventNormalState(new Integer(EventConfirmConstant.stringToType(comboEventNormal.getText())));
		}

		list.add(event);
		return list;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		NotifyInfo info = this.getInputData();
		if(info != null){
			if(this.notifyId == null){
				// 作成の場合
				result = new AddNotify().add(info);
			}
			else{
				// 変更の場合
				result = new ModifyNotify().modify(info);
			}
		}

		return result;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
	@Override
	protected void setValidateResult(String id, String message) {

		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}

	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {

		if(!this.referenceFlg){
			super.createButtonsForButtonBar(parent);
		}
		else{
			// 閉じるボタン
			this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
		}
	}
	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
	@Override
	protected void setEnabled(boolean enable) {
		super.m_notifyBasic.setEnabled(enable);
		super.m_notifyInhibition.setEnabled(enable);
	}

	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
	protected Label getLabelPriority(Composite parent,
			String text,
			Color background
	) {

		// ラベル（重要度）
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(text + " : ");
		label.setBackground(background);

		return label;
	}

	/**
	 * イベント通知の通知のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getCheckEventNormal(Composite parent) {

		// チェックボックス（通知する）
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_NOTIFY;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		return button;
	}

	/**
	 * イベント通知の通知状態コンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたコンボボックス
	 */
	protected Combo getComboEventNormal(Composite parent) {

		// コンボボックス（通知状態）
		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(EventConfirmConstant.STRING_UNCONFIRMED);
		combo.add(EventConfirmConstant.STRING_CONFIRMED);
		combo.setText(EventConfirmConstant.STRING_UNCONFIRMED);

		return combo;
	}

	/**
	 * イベント通知の抑制のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getCheckEventInhibition(Composite parent) {

		// チェックボックス（抑制する）
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		return button;
	}

	/**
	 * イベント通知の抑制状態コンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたコンボボックス
	 */
	protected Combo getComboEventInhibition(Composite parent) {

		// コンボボックス（抑制時の状態）
		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_STATE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(EventConfirmConstant.STRING_CONFIRMED);
		combo.add(EventConfirmConstant.STRING_DESTRUCTION);
		combo.setText(EventConfirmConstant.STRING_CONFIRMED);

		return combo;
	}

}