/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.bean;

import java.util.ArrayList;

import com.clustercontrol.fault.CalendarInvalid;

/**
 * カレンダ情報を格納するDTOクラス
 * 
 * @since 4.0
 * @author Tomoya Takahata
 */
public class CalendarDTO {

	private String calendarId;
	private String calendarName;
	private String description;
	private Long beginDate;
	private Long endDate;

	private Long registerDate;
	private String registerUser;
	private Long updateDate;
	private String updateUser;

	private ArrayList<CalendarWeekdayDTO> weekdays;
	private ArrayList<CalendarIrregularDTO> irregulars;

	/**
	 * コンストラクタ(no-argument for JAXB)
	 */
	public CalendarDTO() {

	}

	/**
	 * コンストラクタ
	 * @param calendarId カレンダID
	 * @param calendarName カレンダ名
	 * @param description カレンダの説明
	 * @param beginDate カレンダの開始日時
	 * @param endDate カレンダの終了日時
	 * @param registerDate カレンダの登録日時
	 * @param registerUser カレンダの登録ユーザ名
	 * @param updateDate カレンダの更新日時
	 * @param updateUser カレンダの更新ユーザ名
	 * @param weekdays カレンダの実行規則
	 * @param irregulars カレンダの実行例外
	 * @throws CalendarInvalid
	 */
	public CalendarDTO(String calendarId, String calendarName, String description, Long beginDate, Long endDate,
			Long registerDate, String registerUser, Long updateDate, String updateUser,
			ArrayList<CalendarWeekdayDTO> weekdays, ArrayList<CalendarIrregularDTO> irregulars) throws CalendarInvalid {
		this.calendarId = calendarId;
		this.calendarName = calendarName;
		this.description = description;
		this.beginDate = beginDate;
		this.endDate = endDate;
		this.registerDate = registerDate;
		this.registerUser = registerUser;
		this.updateDate = updateDate;
		this.updateUser = updateUser;
		this.weekdays = weekdays;
		this.irregulars = irregulars;

		validate();
	}

	/**
	 * カレンダIDを格納する。<br/>
	 * @param calendarId カレンダID
	 */
	public void setCalendarId(String calendarId) {
		this.calendarId = calendarId;
	}

	/**
	 * カレンダIDを返す。<br/>
	 * @return カレンダID
	 */
	public String getCalendarId() {
		return calendarId;
	}

	/**
	 * カレンダ名を格納する。<br/>
	 * @param calendarName カレンダ名
	 */
	public void setCalendarName(String calendarName) {
		this.calendarName = calendarName;
	}

	/**
	 * カレンダ名を返す。<br/>
	 * @return
	 */
	public String getCalendarName() {
		return calendarName;
	}

	/**
	 * カレンダの説明を格納する。<br/>
	 * @param description カレンダの説明
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * カレンダの説明を返す。<br/>
	 * @return カレンダの説明
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * カレンダの開始日時を格納する。<br/>
	 * @param beginDate カレンダの開始日時
	 */
	public void setBeginDate(Long beginDate) {
		this.beginDate = beginDate;
	}

	/**
	 * カレンダの開始日時を返す。<br/>
	 * @return カレンダの開始日時
	 */
	public Long getBeginDate() {
		return beginDate;
	}

	/**
	 * カレンダの終了日時を格納する。<br/>
	 * @param endDate カレンダの終了日時
	 */
	public void setEndDate(Long endDate) {
		this.endDate = endDate;
	}

	/**
	 * カレンダの終了日時を返す。<br/>
	 * @return カレンダの終了日時
	 */
	public Long getEndDate() {
		return endDate;
	}

	/**
	 * カレンダの登録日時を格納する。<br/>
	 * @param registerDate カレンダの登録日時
	 */
	public void setRegisterDate(Long registerDate) {
		this.registerDate = registerDate;
	}

	/**
	 * カレンダの登録日時を返す。<br/>
	 * @return カレンダの登録日時
	 */
	public Long getRegisterDate() {
		return registerDate;
	}

	/**
	 * カレンダの登録ユーザ名を格納する。<br/>
	 * @param registerUser 登録ユーザ名
	 */
	public void setRegisterUser(String registerUser) {
		this.registerUser = registerUser;
	}

	/**
	 * カレンダの登録ユーザ名を返す。<br/>
	 * @return 登録ユーザ名
	 */
	public String getRegisterUser() {
		return registerUser;
	}

	/**
	 * カレンダの更新ユーザ名を格納する。<br/>
	 * @param updateDate 更新ユーザ名
	 */
	public void setUpdateDate(Long updateDate) {
		this.updateDate = updateDate;
	}

	/**
	 * カレンダの更新日時を格納する。<br/>
	 * @return カレンダの更新日時
	 */
	public Long getUpdateDate() {
		return updateDate;
	}

	/**
	 * カレンダの更新ユーザ名を格納する。<br/>
	 * @param updateUser 更新ユーザ名
	 */
	public void setUpdateUser(String updateUser) {
		this.updateUser = updateUser;
	}

	/**
	 * カレンダの更新ユーザ名を返す。<br/>
	 * @return 更新ユーザ名
	 */
	public String getUpdateUser() {
		return updateUser;
	}

	/**
	 * 実行規則を格納する。<br/>
	 * @param weekdays 実行規則の配列
	 */
	public void setWeekdays(ArrayList<CalendarWeekdayDTO> weekdays) {
		this.weekdays = weekdays;
	}

	/**
	 * 実行規則を返す。<br/>
	 * @return 実行規則の配列
	 */
	public ArrayList<CalendarWeekdayDTO> getWeekdays() {
		return weekdays;
	}

	/**
	 * 実行例外を格納する。<br/>
	 * @param irregulars 実行例外の配列
	 */
	public void setIrregulars(ArrayList<CalendarIrregularDTO> irregulars) {
		this.irregulars = irregulars;
	}

	/**
	 * 実行例外を返す。<br/>
	 * @return 実行例外の配列
	 */
	public ArrayList<CalendarIrregularDTO> getIrregulars() {
		return irregulars;
	}

	/**
	 * メンバ変数の妥当性を確認する。<br/>
	 * @throws CalendarInvalid 妥当でない値が存在する場合
	 */
	public void validate() throws CalendarInvalid {
		// null check
		if (calendarId == null) {
			throw new CalendarInvalid("calendar id must be defined(not null). (" + this + ")");
		}
		if (calendarName == null) {
			throw new CalendarInvalid("calendar name must be defined(not null). (" + this + ")");
		}
		if (beginDate == null) {
			throw new CalendarInvalid("begin date must be defined(not null). (" + this + ")");
		}
		if (endDate == null) {
			throw new CalendarInvalid("end date must be defined(not null). (" + this + ")");
		}
		if (weekdays == null || irregulars == null) {
			throw new CalendarInvalid("array of weekdays or irregulars must be array. 0 length list is allowed. (" + this + ")");
		}

		// variable consistency
		if (beginDate > endDate) {
			throw new CalendarInvalid("end date must be after begin date. (" + this + ")");
		}
		for (CalendarIrregularDTO dto : irregulars) {
			if (dto.getEndDate() < beginDate) {
				throw new CalendarInvalid("special term must be inner of calendar term. (" + this + ")");
			}
			if (dto.getBeginDate() > endDate) {
				throw new CalendarInvalid("special term must be innner of calendar term. (" + this + ")");
			}
		}
	}

	@Override
	public String toString() {
		String weekdayStr = null;
		String irregularStr = null;

		for (CalendarWeekdayDTO dto : weekdays) {
			weekdayStr += (weekdayStr != null ? ", " : "") + dto.toString();
		}

		for (CalendarIrregularDTO dto : irregulars) {
			irregularStr += (irregularStr != null ? ", " : "") + dto.toString();
		}

		return "CalendarDTO (calendarId = " + calendarId + ", calendarName = " + calendarName
		+ ", beginDate = " + beginDate + ", endDate = " + endDate + ", " + weekdayStr + ", " + irregularStr + ")";
	}

}
