/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarDuplicate;
import com.clustercontrol.fault.CalendarInvalid;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.calendar.bean.CalendarAddInfo;
import com.clustercontrol.calendar.bean.CalendarDTO;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.ScheduleInfo;
import com.clustercontrol.calendar.bean.WeekdayInfo;
import com.clustercontrol.calendar.factory.ModifyCalendar;
import com.clustercontrol.calendar.factory.SelectCalendar;
import com.clustercontrol.calendar.util.CalendarManagerUtil;
import com.clustercontrol.calendar.util.CalendarValidator;

/**
 *
 * <!-- begin-user-doc --> カレンダー情報の制御を行うsession bean <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="CalendarController"
 *    jndi-name="CalendarController"
 *    type="Stateless"
 *    transaction-type="Container"
 *    view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CalendarInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CalendarWeekdayInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CalendarAddInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition -->
 * @generated
 */
public abstract class CalendarControllerBean implements javax.ejb.SessionBean {
	private static Log m_log = LogFactory.getLog( CalendarControllerBean.class );

	@SuppressWarnings("unused")
	private SessionContext m_context;

	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * カレンダ一覧を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダ情報のリスト
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<CalendarInfo> getCalendarList() throws CalendarNotFound, HinemosUnknown {

		ArrayList<CalendarInfo> list = null;
		try {
			SelectCalendar select = new SelectCalendar();
			list = select.getCalendarList();
		} catch (CalendarNotFound e){
			throw e;
		} catch (Exception e){
			m_log.error("getCalendarList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * カレンダ曜日別情報一覧を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダ曜日別情報のリスト
	 * @throws FinderException
	 * @throws HinemosUnknown
	 */
	public ArrayList<WeekdayInfo> getWeekdayList(String id) throws CalendarNotFound, HinemosUnknown {

		ArrayList<WeekdayInfo> list = null;
		try {
			SelectCalendar select = new SelectCalendar();
			list = select.getWeekdayList(id);
		} catch (CalendarNotFound e) {
			throw e;
		} catch (NamingException e) {
			m_log.warn("getWeekdayList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * カレンダ追加情報一覧を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダ追加情報のリスト
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<CalendarAddInfo> getAddList(String id) throws CalendarNotFound, HinemosUnknown {

		ArrayList<CalendarAddInfo> list = null;
		try {
			SelectCalendar select = new SelectCalendar();
			list = select.getAddList(id);
		} catch (CalendarNotFound e) {
			throw e;
		} catch (NamingException e) {
			m_log.warn("getAddList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 各カレンダの実行予定を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダ実行予定のリスト
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<ScheduleInfo> getSchedule(Long base) throws CalendarNotFound, HinemosUnknown {

		ArrayList<ScheduleInfo> list = null;
		try {
			SelectCalendar select = new SelectCalendar();
			list = select.getSchedule(base);
		} catch (NamingException e) {
			m_log.warn("getSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CalendarNotFound e) {
			throw e;
		}

		return list;
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param id
	 * @return カレンダ情報
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public CalendarInfo getCalendar(String id) throws CalendarNotFound, HinemosUnknown {
		CalendarInfo info = null;

		try{
			SelectCalendar select = new SelectCalendar();
			info = select.getCalendar(id);
		} catch (CalendarNotFound e) {
			throw e;
		} catch (NamingException e) {
			m_log.warn("getCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return info;
	}


	/**
	 * カレンダ（基本）情報を登録します。<BR>
	 * 
	 * 引数のDTOの内容をマネージャに登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param info　登録するカレンダー（基本）情報
	 * @return
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 */
	public void addCalendar(CalendarInfo info) throws CalendarDuplicate, HinemosUnknown,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateCalendarInfo(info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.addCalendar(info, pri.getName());

			commit = true;

		} catch (CalendarDuplicate e) {
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		} catch(Exception e){
			m_log.warn("addCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ（基本）情報を変更します。<BR>
	 * 
	 * 引数のプロパティの内容で更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public void modifyCalendar(CalendarInfo info) throws CalendarNotFound, HinemosUnknown,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateCalendarInfo(info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.modifyCalendar(info, pri.getName());

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("modifyCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("modifyCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ(基本）情報を 削除します。<BR>
	 * 
	 * 引数のIDに対応するカレンダー（基本）情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id カレンダーID
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public void deleteCalendar(String id) throws CalendarNotFound, HinemosUnknown  {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			ModifyCalendar modify = new ModifyCalendar();
			modify.deleteCalendar(id);

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("deleteCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteCalendar() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ曜日別情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @return カレンダ曜日別情報
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public WeekdayInfo getWeekday(String id, Integer weekday, Long from, Long to) throws CalendarNotFound, HinemosUnknown {
		WeekdayInfo info = null;

		try {
			SelectCalendar select = new SelectCalendar();
			info = select.getWeekday(id, weekday, from, to);
		} catch (CalendarNotFound e) {
			m_log.warn("getWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.warn("getWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return info;
	}

	/**
	 * カレンダ曜日別情報を登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id
	 * @param info
	 * @return
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 */
	public void addWeekday(String id, WeekdayInfo info) throws CalendarDuplicate, HinemosUnknown, CalendarNotFound,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateWeekdayInfo(id, info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.addWeekday(id, info, pri.getName());

			commit = true;
		} catch (CalendarDuplicate e) {
			m_log.warn("addWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CalendarNotFound e) {
			m_log.warn("addWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("addWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("addWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ曜日別情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 */
	public void modifyWeekday(String id, Integer weekday, Long from, Long to, WeekdayInfo info) throws CalendarNotFound, CalendarDuplicate, HinemosUnknown,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateWeekdayInfo(id, info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.modifyWeekday(id, weekday, from, to, info, pri.getName());

			commit = true;
		} catch (CalendarDuplicate e) {
			m_log.warn("modifyWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CalendarNotFound e) {
			m_log.warn("modifyWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("modifyWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ曜日別情報を削除します。<BR>
	 * 
	 * idで指定されたカレンダー（実行規則）情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id カレンダーID
	 * @param weekday 曜日
	 * @param from 時間帯
	 * @param to 時間帯
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public void deleteWeekday(String id, Integer weekday, Long from, Long to) throws CalendarNotFound, HinemosUnknown  {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			ModifyCalendar modify = new ModifyCalendar();
			modify.deleteWeekday(id, weekday, from, to);

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("deleteWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteWeekday() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ追加情報用プロパティを取得します。<BR>
	 * 
	 * 引数のカレンダーID、日時区間に対応するカレンダー（実行例外）情報を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @return カレンダ追加情報
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public CalendarAddInfo getCalendarAdd(String id, Long from, Long to) throws CalendarNotFound, HinemosUnknown {
		CalendarAddInfo info = null;

		try {
			//カレンダ曜日別情報を取得
			SelectCalendar select = new SelectCalendar();
			info = select.getCalendarAdd(id, from, to);
		} catch (CalendarNotFound e) {
			m_log.warn("getCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (NamingException e) {
			m_log.warn("getCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}


		return info;
	}

	/**
	 * カレンダ追加情報を登録します。<BR>
	 * 
	 * idとプロパティの情報を元にカレンダー（実行例外）情報をマネージャに登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 */
	public void addCalendarAdd(String id, CalendarAddInfo info) throws CalendarNotFound, CalendarDuplicate, HinemosUnknown,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateCalendarAddInfo(id, info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.addCalendarAdd(id, info, pri.getName());

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("addCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CalendarDuplicate e) {
			m_log.warn("addCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("addCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("addCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}


	/**
	 * カレンダ追加情報を変更します。<BR>
	 * 
	 * idで指定されたカレンダー（例外実行）情報を変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 */
	public void modifyCalendarAdd(String id, Long from, Long to, CalendarAddInfo info) throws CalendarNotFound, CalendarDuplicate, HinemosUnknown,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			CalendarValidator.validateCalendarAddInfo(id, info);

			Principal pri = m_context.getCallerPrincipal();

			ModifyCalendar modify = new ModifyCalendar();
			modify.modifyCalendarAdd(id, from, to, info, pri.getName());

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("modifyCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CalendarDuplicate e) {
			m_log.warn("modifyCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("modifyCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ追加情報を削除します。<BR>
	 * 
	 * idで指定されたカレンダー（実行例外）情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param id カレンダーID
	 * @param from 時間帯
	 * @param to 時間帯
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public void deleteCalendarAdd(String id, Long from, Long to) throws CalendarNotFound, HinemosUnknown  {
		TransactionManager tm = null;
		boolean commit = false;

		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			ModifyCalendar modify = new ModifyCalendar();
			modify.deleteCalendarAdd(id, from, to);

			commit = true;
		} catch (CalendarNotFound e) {
			m_log.warn("deleteCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteCalendarAdd() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// read-committedのため、commit後に外部コンポーネントに通知する
						CalendarManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダが実行可能かチェックします。<BR>
	 * 
	 * 引数のカレンダーidが引数のタイムスタンプにおいて実行可能であるかチェックします。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return カレンダが実行可能か
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public Boolean isRun(String id, Long checkTimestamp) throws CalendarNotFound, HinemosUnknown {
		try {
			SelectCalendar select = new SelectCalendar();
			return select.isRun(id, checkTimestamp);
		} catch (CalendarNotFound e) {
			m_log.warn("isRun() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.warn("isRun() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * カレンダIDのリストを取得します。<BR>
	 * 
	 * 引数のArrayListにはカレンダーidが含まれています。
	 * <p>
	 *	list.add(cal.getCalendar_id());
	 * </p>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダIDのリスト
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getCalendarIdList() throws CalendarNotFound, HinemosUnknown {
		ArrayList<String> list = null;
		try {
			SelectCalendar select = new SelectCalendar();
			list = select.getCalendarIdList();
		} catch (CalendarNotFound e) {
			m_log.warn("getCalendarIdList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.warn("getCalendarIdList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * 指定したカレンダIDに該当する設定情報をDTOインスタンスとして返す
	 * @param calendarId カレンダID
	 * @return カレンダDTOインスタンス
	 * @throws CalendarInvalid 指定されたカレンダIDの格納情報が不適切な場合(不具合の可能性あり)
	 * @throws HinemosUnknown 内部で予期せぬエラーが発生した場合
	 * @throws CalendarNotFound 指定されたカレンダIDに該当する情報が格納されていない場合
	 * 
	 * @ejb.interface-method
	 * 
	 */
	public CalendarDTO getCalendarDTO(String calendarId) throws CalendarInvalid, HinemosUnknown, CalendarNotFound{
		return new SelectCalendar().getCalendarDTO(calendarId);
	}
}

