/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.commons.scheduler;

import java.text.ParseException;
import java.util.Date;
import java.util.Map;

import javax.ejb.EJBLocalHome;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.quartz.Trigger;
import org.quartz.impl.StdScheduler;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.commons.quartz.job.EJBLocalInvokerJob;
import com.clustercontrol.jobmanagement.bean.QuartzConstant;

/**
 * Quartへのスケジュール操作を実装したクラス
 */
public class QuartzScheduler implements TriggerScheduler{
	private static Log m_log = LogFactory.getLog(QuartzScheduler.class);

	private static final String RESET_ON_RESTART_KEY = "resetOnRestartFlag";

	private Scheduler _scheduler;

	public QuartzScheduler(String quartzJndiName) throws NamingException{
		//QuartzのSchedulerをルックアップ
		InitialContext context = new InitialContext();

		Object obj = context.lookup(quartzJndiName);
		_scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);
	}

	public QuartzScheduler(Scheduler scheduler){
		_scheduler = scheduler;
	}

	/**
	 * SimpleTrigger でQuartzジョブをスケジューリングします。
	 */
	@Override
	public void scheduleEjbLocalInvokerJobWithSimpleTrigger(
			String jobName ,
			String jobGroupName,
			EJBLocalHome ejbLocalHome,
			String methodName,
			Object[] jdArgs,
			Class<Object>[] jdArgsType,
			Map<String, Object> jobDataMap,
			boolean resetOnRestartFlag,
			boolean validFlag,
			long startTime,
			int intervalSecond)
	throws TriggerSchedulerException{
		m_log.debug("jobName=" + jobName + ", jobGroupName=" + jobGroupName + ", ejbLocalHome=" + ejbLocalHome);

		//JobDetail作成
		JobDetail job = new JobDetail(
				jobName,
				jobGroupName,
				EJBLocalInvokerJob.class);

		//ジョブ完了時に削除されないようにする。
		job.setDurability(true);

		//ジョブ実行失敗時に再実行するようにする。
		//JBoss再起動時に複数同時に再実行される可能性があるためこのオプションは設定しないよう変更する
		//	    job.setRequestsRecovery(true);

		//JobDetailに呼び出すクラスとメソッドを設定
		job.getJobDataMap().put(EJBLocalInvokerJob.EJB_LOCAL_HOME_INSTANCE, ejbLocalHome);
		job.getJobDataMap().put(EJBLocalInvokerJob.EJB_METHOD_KEY, methodName);

		//JobDetailに呼び出すメソッドの引数を設定
		if(jdArgs == null || jdArgsType == null){
			jdArgs = new Object[0];
			job.getJobDataMap().put(EJBLocalInvokerJob.EJB_ARGS_KEY, jdArgs);
		} else {
			job.getJobDataMap().put(EJBLocalInvokerJob.EJB_ARGS_KEY, jdArgs);
			job.getJobDataMap().put(EJBLocalInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
		}

		//その他のデータマップがある場合は追加
		if(jobDataMap != null){
			for(String key : jobDataMap.keySet()){
				job.getJobDataMap().put(key, jobDataMap.get(key));
			}
		}

		// Hinemosマネージャ再起動時にトリガを再設定する必要があるか否かを設定する
		job.getJobDataMap().put(RESET_ON_RESTART_KEY, resetOnRestartFlag);

		Trigger trigger = new SimpleTrigger(
				jobName,
				jobGroupName,
				new Date(startTime),
				new Date(Long.MAX_VALUE),
				SimpleTrigger.REPEAT_INDEFINITELY,
				(long)intervalSecond * 1000);

		// Quartzのバグにより、java.lang.StackOverflowError が発生するため、
		// trigger.getJobDataMap() で取得できるMapに値を設定してはいけいない。

		try {
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			_scheduler.deleteJob(jobName, jobGroupName);
		} catch (SchedulerException e) {
			m_log.warn("scheduleEjbInvokerJob() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}

		try {
			// ジョブを登録する
			_scheduler.scheduleJob(job, trigger);

			// validFlag が falseの場合は、ポーズする。
			if(validFlag == false){
				pauseJob(jobName, jobGroupName);
			}
		} catch (SchedulerException e) {
			m_log.warn("scheduleEjbInvokerJob() scheduleJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * CronTrigger でQuartzジョブをスケジューリングします。
	 */
	@Override
	public void scheduleEjbLocalInvokerJobWithCronTrigger(
			String jobName ,
			String jobGroupName,
			EJBLocalHome ejbLocalHome,
			String methodName,
			Object[] jdArgs,
			Class<Object>[] jdArgsType,
			Map<String, Object> jobDataMap,
			boolean resetOnRestartFlag,
			boolean validFlag,
			String cronExpression)
	throws TriggerSchedulerException{
		//JobDetail作成
		JobDetail job = new JobDetail(
				jobName,
				jobGroupName,
				EJBLocalInvokerJob.class);

		//ジョブ完了時に削除されないようにする。
		job.setDurability(true);

		//ジョブ実行失敗時に再実行するようにする。
		//JBoss再起動時に複数同時に再実行される可能性があるためこのオプションは設定しないよう変更する
		//	    job.setRequestsRecovery(true);

		//JobDetailに呼び出すクラスとメソッドを設定
		job.getJobDataMap().put(EJBLocalInvokerJob.EJB_LOCAL_HOME_INSTANCE, ejbLocalHome);
		job.getJobDataMap().put(EJBInvokerJob.EJB_METHOD_KEY, methodName);

		//JobDetailに呼び出すメソッドの引数を設定
		if(jdArgs == null || jdArgsType == null){
			jdArgs = new Object[0];
			job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
		} else {
			job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
			job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
		}

		//その他のデータマップがある場合は追加
		if(jobDataMap != null){
			for(String key : jobDataMap.keySet()){
				job.getJobDataMap().put(key, jobDataMap.get(key));
			}
		}

		// Hinemosマネージャ再起動時にトリガを再設定する必要があるか否かを設定する
		job.getJobDataMap().put(RESET_ON_RESTART_KEY, resetOnRestartFlag);

		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(jobName, jobGroupName);

		//起動失敗した場合は、次の起動予定時刻をセットするように設定
		cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);

		//スケジュールを設定
		try {
			cronTrigger.setCronExpression(cronExpression);

			// pauseFlag が trueの場合、起動させないように実行開始時刻を少し遅らせる。
			cronTrigger.setStartTime(new Date(System.currentTimeMillis() + 1000l));
		} catch (ParseException e) {
			m_log.warn("scheduleEjbInvokerJob() setCronExpression : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}

		// Quartzのバグにより、java.lang.StackOverflowError が発生するため、
		// trigger.getJobDataMap() で取得できるMapに値を設定してはいけいない。

		try {
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			_scheduler.deleteJob(jobName, jobGroupName);
		} catch (SchedulerException e) {
			m_log.warn("scheduleEjbInvokerJob() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}

		try {
			// ジョブを登録する
			_scheduler.scheduleJob(job, cronTrigger);

			// validFlag が falseの場合は、ポーズする。
			if(validFlag == false){
				pauseJob(jobName, jobGroupName);
			}
		} catch (SchedulerException e) {
			m_log.warn("scheduleEjbInvokerJob() scheduleJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * Quartzジョブを一時停止します。
	 */
	@Override
	public void pauseJob(String jobName ,String jobGroupName) throws TriggerSchedulerException{
		try {
			_scheduler.pauseJob(jobName, jobGroupName);
		} catch (SchedulerException e) {
			m_log.warn("deleteJob() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * CronTriggerで実行契機が登録されているQuartzジョブを変更します。
	 */
	@Override
	public void modifyEjbInvokerJobWithCronTrigger(String jobName ,String jobGroupName, String cronExpression, boolean validFlag) throws TriggerSchedulerException{
		try {
			CronTrigger trigger = (CronTrigger)_scheduler.getTrigger(jobName, jobGroupName);
			trigger.setCronExpression(cronExpression);

			_scheduler.scheduleJob(trigger);

			// validFlag が falseの場合は、ポーズする。
			if(validFlag == false){
				pauseJob(jobName, jobGroupName);
			}
		} catch (SchedulerException e) {
			m_log.warn("modifyEjbInvokerJob() modify : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		} catch (ParseException e) {
			m_log.warn("modifyEjbInvokerJob() modify : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * SimpleTriggerで実行契機が登録されているQuartzジョブを変更します。
	 */
	@Override
	public void modifyEjbInvokerJobWithSimpleTrigger(String jobName, String jobGroupName, int intervalSecond, boolean validFlag) throws TriggerSchedulerException{
		try {
			SimpleTrigger trigger = (SimpleTrigger)_scheduler.getTrigger(jobName, jobGroupName);
			trigger.setRepeatInterval((long)intervalSecond * 1000);

			_scheduler.scheduleJob(trigger);

			// validFlag が falseの場合は、ポーズする。
			if(validFlag == false){
				pauseJob(jobName, jobGroupName);
			}
		} catch (SchedulerException e) {
			m_log.warn("modifyEjbInvokerJob() modify : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * Quartzジョブを削除します。
	 */
	@Override
	public void deleteJob(String jobName ,String jobGroupName) throws TriggerSchedulerException{
		try {
			_scheduler.deleteJob(jobName, jobGroupName);
		} catch (SchedulerException e) {
			m_log.warn("deleteJob() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw new TriggerSchedulerException(e.getMessage(), e, jobName, jobGroupName);
		}
	}

	/**
	 * 再起動時にトリガの再設定を行うフラグが設定されているトリガのスタート時刻を更新します。
	 */
	public void resetTrigger() throws TriggerSchedulerException {
		//現在時刻を取得
		Date now = new Date();
		m_log.debug("set next_fire_time : " + now.getTime());

		String[] triggerGroupNames;
		try {
			triggerGroupNames = _scheduler.getTriggerGroupNames();
		} catch (SchedulerException e) {
			// ログを出力し処理を終了する
			m_log.error(e.getMessage(), e);
			throw new TriggerSchedulerException(e.getMessage(), e, "unknown", "unknown");
		}

		//triggerGroup, triggerNameからトリガを特定
		for (String triggerGroupName : triggerGroupNames) {
			String[] triggerNames;
			try {
				triggerNames = _scheduler.getTriggerNames(triggerGroupName);
			} catch (SchedulerException e) {
				// ログを出力し、次のトリガグループを処理する
				m_log.error(e.getMessage(), e);
				continue;
			}

			for (String triggerName : triggerNames) {

				m_log.debug("triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName);

				try{
					Trigger trigger = _scheduler.getTrigger(triggerName, triggerGroupName);

					//TriggerStateの状態確認 for debug
					if (m_log.isDebugEnabled() && trigger != null){
						switch (_scheduler.getTriggerState(triggerName, triggerGroupName)) {
						case Trigger.STATE_BLOCKED:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_BLOCKED");
							break;
						case Trigger.STATE_COMPLETE:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_COMPLETE");
							break;
						case Trigger.STATE_ERROR:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_ERROR");
							break;
						case Trigger.STATE_NONE:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_NONE");
							break;
						case Trigger.STATE_NORMAL:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_NORMAL");
							break;
						case Trigger.STATE_PAUSED:
							m_log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_PAUSED");
							break;
						default:
							break;
						}
					}

					if(!QuartzConstant.GROUP_NAME.equals(triggerGroupName)){
						// トリガがnullで無い、かつ「有効(STATE_NORMAL)」の場合
						if (trigger != null
								&& _scheduler.getTriggerState(triggerName, triggerGroupName) == Trigger.STATE_NORMAL) {
							m_log.debug("resetTrigger() triggerName : " + triggerName + " is not PAUSED");

							// リスタートすべきかどうかのフラグ（restartFlg）は、本来トリガに関連付けて保存すべきであるが、
							// trigger.getJobDataMap() で取得できるMapに値を設定すると、
							// Quartzのバグにより、java.lang.StackOverflowError が発生するため、設定できない。
							// トリガ名とジョブ名は1対1に対応するため、JobDetailのJobDataMapを利用し、リスタートすべきかどうかのフラグを取得する。
							JobDetail job = _scheduler.getJobDetail(triggerName, triggerGroupName);
							Boolean restartFlg = (Boolean)job.getJobDataMap().get(RESET_ON_RESTART_KEY);
							m_log.debug("resetTrigger() jobName=" + job.getName() + ", jobGroupName=" + job.getGroup() + ", resetOnRestartFlag=" + restartFlg);

							// 再起動時にトリガの再設定をしない設定となっている場合は、この処理をスキップする。
							// 例えば、ジョブ実行のためのスケジュール登録の場合は、JBoss再起動前の
							// 許容時間（org.quartz.jobStore.misfireThresholdで指定した時間 （デフォルト1時間））以内の
							// ジョブに関しては実行対象とする必要があるため、ここでStartTimeのリセットは行わない。
							// ここで、ジョブのStartTimeもリセットしてしまうと、HA構成の場合に、
							// フェイルオーバ中に実行するようにスケジューリングされていたジョブがフェイルオーバ後に実行されなくなる。
							if(restartFlg != null && restartFlg.booleanValue() == true){
								m_log.debug("resetTrigger() setStartTime : " + now);
								trigger.setStartTime(now);
								_scheduler.rescheduleJob(triggerName, triggerGroupName, trigger);
							}
						}
					}
				} catch (SchedulerException e) {
					// トリガの再設定に失敗した
					// ログを出力し、次のトリガを処理する
					m_log.error("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " : " + e.getMessage(), e);
				}
			}
		}
	}
}