/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.factory;

import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.JmsPersistenceConfig;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoLocal;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.util.Messages;

/**
 * 監視管理に通知するクラスです。
 *
 * @version 3.0.0
 * @since 2.0.0
 */
public class Notice {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( Notice.class );

	/**
	 * セッションID、ジョブユニットID、ジョブIDからジョブ通知情報を取得し、<BR>
	 * ジョブ通知情報と終了状態を基に、ログ出力情報作成し、監視管理に通知します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param type 終了状態
	 * @throws NamingException
	 * @throws JobInfoNotFound
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.monitor.message.LogOutputNotifyInfo
	 */
	public void notify(String sessionId, String jobunitId, String jobId, Integer type) throws NamingException, JobInfoNotFound {
		m_log.debug("notify() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId + ", type=" + type);

		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e1) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}

		JobInfoLocal job = sessionJob.getJobInfo();

		//セッションID、ジョブID、通知タイプ(終了状態+開始)から、通知情報を取得
		JobNoticeInfoLocal notice = null;
		try {
			notice = JobNoticeInfoUtil.getLocalHome().findByPrimaryKey(
					new JobNoticeInfoPK(sessionId, jobunitId, jobId, type));
		} catch (FinderException e1) {
			m_log.warn("JobNoticeInfoUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}

		// 通知先の指定において以下の場合は通知する
		// 1.重要度がnullでなく、かつPriorityConstant.TYPE_NONE（空欄）ではない場合
		// 2.通知IDがnullでなく、かつ0文字以上である場合
		if(notice.getPriority() != null && notice.getPriority() != PriorityConstant.TYPE_NONE &&
				notice.getNotifyGroupId() != null && notice.getNotifyGroupId().length() > 0){
			//通知する

			//通知情報作成
			OutputBasicInfo info = new OutputBasicInfo();
			//プラグインID
			info.setPluginId("JOB");
			//アプリケーション
			info.setApplication(Messages.getString("job.management"));
			//監視項目ID
			info.setMonitorId(sessionId);
			//メッセージID、メッセージ、オリジナルメッセージ
			if(type.intValue() == EndStatusConstant.TYPE_BEGINNING){
				info.setMessageId("001");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				info.setMessage(Messages.getString("message.job.38", args1));
			}
			else if(type.intValue() == EndStatusConstant.TYPE_NORMAL){
				info.setMessageId("002");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				info.setMessage(Messages.getString("message.job.39", args1));
			}
			else if(type.intValue() == EndStatusConstant.TYPE_WARNING){
				info.setMessageId("003");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				info.setMessage(Messages.getString("message.job.40", args1));
			}
			else if(type.intValue() == EndStatusConstant.TYPE_ABNORMAL){
				info.setMessageId("004");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				info.setMessage(Messages.getString("message.job.41", args1));
			}
			if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
				JobCommandInfoLocal command = job.getJobCommandInfo();
				//ファシリティID
				info.setFacilityId(command.getFacility_id());
				//スコープ
				info.setScopeText(sessionJob.getScope_text());
			}
			else{
				//ファシリティID
				info.setFacilityId("");
				//スコープ
				info.setScopeText("");
			}
			//重要度
			info.setPriority(notice.getPriority().intValue());
			//発生日時
			info.setGenerationDate(new Date().getTime());

			try{
				//メッセージ送信
				if (m_log.isDebugEnabled()) {
					m_log.debug("sending message. "
							+ JmsPersistenceConfig.getPersistenceString(JmsPersistenceConfig.QUEUE_JOB_MODE)
							+ " : priority=" + info.getPriority()
							+ " pluginId=" + info.getPluginId()
							+ " monitorId=" + info.getMonitorId()
							+ " facilityId=" + info.getFacilityId() + ")");
				}

				// 通知処理
				NotifyControllerUtil.getLocalHome().create().notify(info, notice.getNotifyGroupId());
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			} catch (CreateException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	}

	/**
	 * 遅延通知
	 * 
	 * @param sessionId
	 * @param jobunitId
	 * @param jobId
	 * @param startDelay
	 * @throws FinderException
	 * @throws NamingException
	 */
	/**
	 * セッションID、ジョブユニットID、ジョブIDからジョブ通知情報を取得し、<BR>
	 * ジョブ通知情報と開始遅延フラグを基に、ログ出力情報作成し、監視管理に通知します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param startDelay 開始遅延フラグ（true：開始遅延、false：終了遅延）
	 * @throws NamingException
	 * @throws JobInfoNotFound
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.monitor.message.LogOutputNotifyInfo
	 */
	public void delayNotify(String sessionId, String jobunitId, String jobId, boolean startDelay) throws NamingException, JobInfoNotFound {
		m_log.debug("delayNotify() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId + ", startDelay=" + startDelay);

		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e1) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}

		JobInfoLocal job = sessionJob.getJobInfo();
		JobStartInfoLocal start = job.getJobStartInfo();

		//セッションID、ジョブID、通知タイプ(終了状態+開始)から、通知情報を取得
		JobNoticeInfoLocal notice = null;
		try {
			notice = JobNoticeInfoUtil.getLocalHome().findByPrimaryKey(
					new JobNoticeInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_NORMAL)));
		} catch (FinderException e1) {
			m_log.warn("JobNoticeInfoUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			je.setNoticeType(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
			throw je;
		}

		// 通知先の指定において以下の場合は通知する
		// 1.通知IDがnullでなく、かつ0文字以上である場合
		if(notice.getNotifyGroupId() != null && notice.getNotifyGroupId().length() > 0){
			//通知する

			//通知情報作成
			OutputBasicInfo info = new OutputBasicInfo();
			//プラグインID
			info.setPluginId("JOB");
			//アプリケーション
			info.setApplication(Messages.getString("job.management"));
			//監視項目ID
			info.setMonitorId(sessionId);

			//メッセージID、メッセージ、オリジナルメッセージ
			StringBuilder message = new StringBuilder();
			if(startDelay){
				info.setMessageId("101");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				message.append(Messages.getString("message.job.55", args1));

				//操作
				if(start.getStart_delay_operation().intValue() == YesNoConstant.TYPE_YES){
					int type = start.getStart_delay_operation_type().intValue();

					String[] args2 = {OperationConstant.typeToString(type)};
					message.append(" " + Messages.getString("message.job.57", args2));

					if(type == OperationConstant.TYPE_STOP_SKIP){
						info.setMessageId("102");
						String[] args3 = {start.getStart_delay_operation_end_value().toString()};
						message.append(" " + Messages.getString("message.job.58", args3));
					}
					else if(type == OperationConstant.TYPE_STOP_WAIT){
						info.setMessageId("103");
					}
				}
				info.setMessage(message.toString());
			}
			else{
				info.setMessageId("201");
				String jobType = JobConstant.typeToString(job.getJob_type().intValue());
				String jobName = job.getJob_name();
				String[] args1 = {jobType,jobName,jobId,sessionId};
				message.append(Messages.getString("message.job.56", args1));

				//操作
				if(start.getEnd_delay_operation().intValue() == YesNoConstant.TYPE_YES){
					int type = start.getEnd_delay_operation_type().intValue();

					String[] args2 = {OperationConstant.typeToString(type)};
					message.append(" " + Messages.getString("message.job.57", args2));

					if(type == OperationConstant.TYPE_STOP_AT_ONCE){
						info.setMessageId("202");
					}
					else if(type == OperationConstant.TYPE_STOP_SUSPEND){
						info.setMessageId("203");
					}
					else if(type == OperationConstant.TYPE_STOP_SET_END_VALUE){
						info.setMessageId("204");
					}
				}
				info.setMessage(message.toString());
			}

			if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
				JobCommandInfoLocal command = job.getJobCommandInfo();
				//ファシリティID
				info.setFacilityId(command.getFacility_id());
				//スコープ
				info.setScopeText(sessionJob.getScope_text());
			}
			else{
				//ファシリティID
				info.setFacilityId("");
				//スコープ
				info.setScopeText("");
			}
			//重要度
			if(startDelay)
				info.setPriority(start.getStart_delay_notify_priority().intValue());
			else
				info.setPriority(start.getEnd_delay_notify_priority().intValue());
			//発生日時
			info.setGenerationDate(new Date().getTime());

			try {
				// 通知処理
				NotifyControllerUtil.getLocalHome().create().notify(info, notice.getNotifyGroupId());
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			} catch (CreateException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	}
}
