/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.util;

import java.text.DateFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.notify.bean.OutputBasicInfo;

/**
 * ジョブ変数ユーティリティクラス<BR>
 *
 * @version 3.0.0
 * @since 2.1.0
 */
public class ParameterUtil {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( ParameterUtil.class );


	public static final int TYPE_JOB = 1;
	public static final int TYPE_MONITOR = 2;


	/**
	 * ログ出力情報からパラメータIDに対応する値を取得します。
	 * ジョブセッション作成時に、ジョブ変数に対応する値を取得するために使用します。
	 * 
	 * @param paramId パラメータID
	 * @param info ログ出力情報
	 * @return 値
	 */
	public static String getParameterValue(String paramId, OutputBasicInfo info) {
		String value = null;

		if(info == null)
			return value;

		if(paramId.equals(SystemParameterConstant.FACILITY_ID)){
			//ファシリティID
			value = info.getFacilityId();
		}
		else if(paramId.equals(SystemParameterConstant.PLUGIN_ID)){
			//プラグインID
			value = info.getPluginId();
		}
		else if(paramId.equals(SystemParameterConstant.MONITOR_ID)){
			//監視項目ID
			value = info.getMonitorId();
		}
		else if(paramId.equals(SystemParameterConstant.MESSAGE_ID)){
			//メッセージID
			value = info.getMessageId();
		}
		else if(paramId.equals(SystemParameterConstant.APPLICATION)){
			//アプリケーション
			value = info.getApplication();
		}
		else if(paramId.equals(SystemParameterConstant.PRIORITY)){
			//重要度
			value = String.valueOf(info.getPriority());
		}
		else if(paramId.equals(SystemParameterConstant.MESSAGE)){
			//メッセージ
			value = info.getMessage();
		}
		else if(paramId.equals(SystemParameterConstant.ORG_MESSAGE)){
			//オリジナルメッセージ
			value = info.getMessageOrg();
		}

		return value;
	}

	/**
	 * ジョブセッションからジョブ変数情報の一覧を取得します。
	 * 
	 * @param sessionId ジョブセッションID
	 * @return 連想配列（ジョブ変数名, 値）
	 * @throws JobInfoNotFound
	 * @throws NamingException
	 */
	private static HashMap<String, String> getSessionParameters(String sessionId) throws JobInfoNotFound, NamingException {
		/** Local Variables */
		HashMap<String, String> parameters = new HashMap<String, String>();
		Collection collection = null;
		Iterator itr = null;
		JobParamInfoLocal param = null;

		/** Main */
		if (m_log.isDebugEnabled()) m_log.debug("getting parameters of job session... (session_id = " + sessionId + ")" );
		try {
			collection = JobParamInfoUtil.getLocalHome().findBySessionId(sessionId);

			if(collection != null && collection.size() > 0){
				itr = collection.iterator();
				while(itr.hasNext()){
					param = (JobParamInfoLocal)itr.next();
					if (! parameters.containsKey(param.getParam_id())) {
						if (m_log.isDebugEnabled()) m_log.debug("added parameter. (session_id = " + sessionId + ", param_id = " + param.getParam_id() + ", value = " + param.getValue() + ")");
						parameters.put(param.getParam_id(), param.getValue());
					} else {
						if (m_log.isDebugEnabled()) m_log.debug("duplicated, skipped paramter. (session_id = " + sessionId + ", param_id = " + param.getParam_id() + ", value = " + param.getValue() + ")");
					}
				}
			}
		} catch (FinderException e) {
			if (m_log.isWarnEnabled()) m_log.debug("failed to get parameters of job session... (session_id = " + sessionId + ")", e);

			m_log.warn("JobParamInfoUtil.getLocalHome().findBySessionId() : " + e.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
			je.setSessionId(sessionId);
			throw je;


		} catch (NamingException e) {
			if (m_log.isWarnEnabled()) m_log.debug("failed to get parameters of job session... (session_id = " + sessionId + ")", e);
			throw e;
		}

		if (m_log.isDebugEnabled()) m_log.debug("successful in get parameters of job session... (session_id = " + sessionId + ")" );
		return parameters;
	}

	/**
	 * 
	 * ジョブセッション情報からパラメータIDに対応する値を取得します。
	 * ジョブセッション作成時に、ジョブ変数に対応する値を取得するために使用します。
	 * 
	 * @param sessionId 情報を取得するジョブセッションを特定するためのセッションID
	 * @param paramId 取得する情報を識別するID
	 * @return
	 * @throws NamingException
	 * @throws JobInfoNotFound
	 */
	public static String getJobParameterValue(String sessionId, String paramId) throws NamingException, JobInfoNotFound{
		String ret = null;

		if(paramId.equals(SystemParameterConstant.START_DATE)){

			try {
				//セッション開始日時
				JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
				ret = DateFormat.getDateTimeInstance().format(session.getSchedule_date());
			} catch (FinderException e) {
				m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
				JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
				je.setSessionId(sessionId);
				throw je;
			}

		} else if (paramId.equals(SystemParameterConstant.SESSION_ID)){
			ret = sessionId;

		} else if (paramId.equals(SystemParameterConstant.TRIGGER_TYPE)){
			try {
				JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
				ret = JobTriggerTypeConstant.typeToString(session.getTrigger_type());
			} catch (FinderException e) {
				m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
				JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
				je.setSessionId(sessionId);
				throw je;
			}

		} else if (paramId.equals(SystemParameterConstant.TRIGGER_INFO)){
			try {
				JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
				ret = session.getTrigger_info();
			} catch (FinderException e) {
				m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
				JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
				je.setSessionId(sessionId);
				throw je;
			}

		}

		return ret;
	}


	/**
	 * ジョブセッションからパラメータIDに対応する値を取得します。
	 * ジョブセッション作成後（ジョブ変数を含む）に、変数に格納されている値を取り出すために使用します。
	 * 
	 * @param sessionId セッションID
	 * @param paramId パラメータID
	 * @return 値
	 * @throws JobInfoNotFound
	 * @throws NamingException
	 */
	public static String getSessionParameterValue(String sessionId, String paramId) throws JobInfoNotFound, NamingException {
		String value = null;

		Collection collection = null;
		try {
			collection = JobParamInfoUtil.getLocalHome().findBySessionIdAndParamId(sessionId, paramId);
		} catch (FinderException e) {
			m_log.warn("JobParamInfoUtil.getLocalHome().findBySessionIdAndParamId() : " + e.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setParamId(paramId);
			throw je;
		} catch (NamingException e) {
			throw e;
		}

		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobParamInfoLocal param = (JobParamInfoLocal)itr.next();
				value = param.getValue();
				break;
			}
		}

		return value;
	}

	/**
	 * 引数で指定された文字列からパラメータIDを取得し、<BR>
	 * セッションからパラメータIDに対応する値を取得します。<BR>
	 * 引数で指定された文字列のパラメータID部分を取得した値で置き換えます。
	 * 
	 * @param sessionId セッションID
	 * @param source 置き換え対象文字列
	 * @return 置き換え後の文字列
	 * @throws NamingException
	 * @throws JobInfoNotFound
	 * @throws Exception
	 */
	public static String replaceSessionParameterValue(String sessionId, String source) throws JobInfoNotFound, NamingException {
		// Local Variables
		HashMap<String, String> parameters = new HashMap<String, String>();
		String commandOrig = source;
		String commandConv = source;

		// Main
		if (m_log.isDebugEnabled()) m_log.debug("generating command string... (session_id = " + sessionId + ", command_orig = " + commandOrig + ")");

		if (commandOrig == null) {
			if (m_log.isWarnEnabled()) m_log.warn("registed command is invalid. (session_id = " + sessionId + ", command_orig = " + commandOrig + ")");
		}

		parameters = getSessionParameters(sessionId);

		for (String parameter : parameters.keySet()) {
			if (parameters.get(parameter) != null) {
				if (m_log.isDebugEnabled()) m_log.debug("replace parameter. (session_id = " + sessionId + ", command_orig = " + commandOrig + ", parameter = " + parameter + ", value = " + parameters.get(parameter) + ")");
				commandConv = commandConv.replace(SystemParameterConstant.getParamText(parameter), parameters.get(parameter));
			} else {
				if (m_log.isDebugEnabled()) m_log.debug("skip replace parameter (undef). (session_id = " + sessionId + ", command_orig = " + commandOrig + ", parameter = " + parameter + ")");
			}
		}

		if (m_log.isDebugEnabled()) m_log.debug("successful in generating command string... (session_id = " + sessionId + ", command_orig = " + commandOrig + ", command_conv = " + commandConv + ")");
		return commandConv;
	}


	/**
	 * 
	 * 
	 * 
	 * @param paramId ジョブ変数の種類を示す文字列
	 * @return
	 */
	public static int checkFunctionType(String paramId) {
		int ret = 0;

		// ジョブ関連
		if (paramId.equals(SystemParameterConstant.START_DATE)
				|| paramId.equals(SystemParameterConstant.SESSION_ID)
				|| paramId.equals(SystemParameterConstant.TRIGGER_TYPE)
				|| paramId.equals(SystemParameterConstant.TRIGGER_INFO )){

			ret = TYPE_JOB;

			// 監視関連
		} else if (paramId.equals(SystemParameterConstant.FACILITY_ID)
				|| 	paramId.equals(SystemParameterConstant.PLUGIN_ID)
				|| paramId.equals(SystemParameterConstant.MONITOR_ID)
				|| paramId.equals(SystemParameterConstant.MESSAGE_ID)
				|| paramId.equals(SystemParameterConstant.APPLICATION)
				|| paramId.equals(SystemParameterConstant.PRIORITY)
				|| paramId.equals(SystemParameterConstant.MESSAGE)
				|| paramId.equals(SystemParameterConstant.ORG_MESSAGE)){

			ret = TYPE_MONITOR;
		}

		return ret;
	}

}
