/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.factory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.monitor.bean.ConfirmConstant;
import com.clustercontrol.monitor.bean.EventDataInfo;
import com.clustercontrol.monitor.bean.EventFilterInfo;
import com.clustercontrol.monitor.ejb.session.MonitorControllerBean;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogPK;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.performance.util.ExportCollectedDataFile;
import com.clustercontrol.repository.bean.FacilityTargetConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;
import com.clustercontrol.util.Messages;

/**
 * イベント情報を検索するクラス<BR>
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectEvent {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SelectEvent.class );

	/**
	 * 表示イベント数（デフォルト値）。<BR>
	 * 監視[イベント]ビューに表示するイベント表示数を格納します。
	 */
	private final static int MAX_DISPLAY_NUMBER = 500;

	/**
	 * ダウンロード可能な最大イベント数
	 * CSVおよびPDFで出力される最大のイベント数を格納します。
	 */
	private static final String MAX_REPORT_EVENT_COUNT_KEY = "monitor.common.report.event.count";
	private static volatile Integer MAX_REPORT_EVENT_COUNT = null;
	private static final Integer MAX_REPORT_EVENT_COUNT_DEFAULT = 2000;
	private static String SEPARATOR = ",";
	private static boolean UTF8_BOM = true; // UTF-8のBOMありなし。
	private String DATE_FORMAT;

	/**
	 * コンストラクタ
	 * 各種設定値を格納します
	 */
	public SelectEvent() {
		synchronized (this) {
			if (MAX_REPORT_EVENT_COUNT == null) {
				String maxReportEventCountStr = null;
				Properties properties = new Properties();

				try {
					String etcdir = System.getProperty("hinemos.manager.etc.dir");
					String propertyFile = etcdir + File.separator + "hinemos.properties";

					// プロパティファイルからキーと値のリストを読み込みます
					properties.load(new FileInputStream(propertyFile));

					// ダウンロード可能な最大イベント数を取得
					if (properties.containsKey("monitor.common.report.event.count")) {
						maxReportEventCountStr = properties.getProperty(MAX_REPORT_EVENT_COUNT_KEY);
						MAX_REPORT_EVENT_COUNT = Integer.parseInt(maxReportEventCountStr);
					} else {
						MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
					}
				} catch (Exception e) {
					m_log.warn("a configuration in hinemos.properties is invalid(" + MAX_REPORT_EVENT_COUNT_KEY + " = " + maxReportEventCountStr + ").");
					MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
				} finally {
					if (MAX_REPORT_EVENT_COUNT < -1) {
						m_log.warn("a configuration in hinemos.properties is invalid(" + MAX_REPORT_EVENT_COUNT_KEY + " = " + maxReportEventCountStr + ").");
						MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
					} else if (MAX_REPORT_EVENT_COUNT == -1) {
						MAX_REPORT_EVENT_COUNT = null;
					}
					m_log.info("initialized max event count for report to " + (MAX_REPORT_EVENT_COUNT == null ? "unlimit" : MAX_REPORT_EVENT_COUNT.toString()) + ".");
				}
			}
			String bom = HinemosProperties.getProperty("monitor.common.report.event.bom", "true");
			if (bom != null && "false".equals(bom)) {
				UTF8_BOM = false;
			}
			SEPARATOR = HinemosProperties.getProperty("monitor.common.report.event.separator", ",");
			DATE_FORMAT = HinemosProperties.getProperty("monitor.common.report.event.format",  "yyyy/MM/dd HH:mm:ss");
		}
	}
	/**
	 * イベント情報を取得します。<BR>
	 * 
	 * @param monitorId
	 * @param pluginId
	 * @param facilityId
	 * @param outputDate
	 * @return イベント情報
	 * @throws MonitorNotFound
	 * @throws NamingException
	 * 
	 */
	public static EventDataInfo getEventInfo(
			String monitorId,
			String monitorDetailId,
			String pluginId,
			String facilityId,
			Long outputDate) throws MonitorNotFound, NamingException{

		EventDataInfo info = null;

		// イベントログ情報を取得
		EventLogPK pk = new EventLogPK(
				monitorId,
				monitorDetailId,
				pluginId,
				facilityId,
				new Timestamp(outputDate));
		EventLogLocal event;
		try {
			event = EventLogUtil.getLocalHome().findByPrimaryKey(pk);
			if(event != null){
				info = new EventDataInfo();
				info.setPriority(event.getPriority());
				if(event.getOutputDate() != null){
					info.setOutputDate(event.getOutputDate().getTime());
				}
				if(event.getGenerationDate() != null){
					info.setGenerationDate(event.getGenerationDate().getTime());
				}
				info.setPluginId(event.getPluginId());
				info.setMonitorId(event.getMonitorId());
				info.setMonitorDetailId(event.getMonitorDetailId());
				info.setFacilityId(event.getFacilityId());
				info.setScopeText(event.getScopeText());
				info.setApplication(event.getApplication());
				info.setMessageId(event.getMessageId());
				info.setMessage(event.getMessage());
				info.setMessageOrg(event.getMessageOrg());
				info.setConfirmed(event.getConfirmFlg());
				if(event.getConfirmDate() != null){
					info.setConfirmDate(event.getConfirmDate().getTime());
				}
				info.setConfirmUser(event.getConfirmUser());
				info.setDuplicationCount(event.getDuplicationCount());
				info.setComment(event.getComment());
				if(event.getCommentDate() != null) {
					info.setCommentDate(event.getCommentDate().getTime());
				}
				info.setCommentUser(event.getCommentUser());
			}
		} catch (FinderException e) {
			m_log.debug("getEventInfo():" + e.getMessage());
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.debug("getEventInfo():" + e.getMessage());
			throw e;
		}
		return info;
	}

	/**
	 * 引数で指定された条件に一致するイベント一覧情報を返します。<BR>
	 * 表示イベント数を越えた場合は、表示イベント数分のイベント情報一覧を返します。
	 * 各イベント情報は、EventLogDataインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws CreateException
	 * @throws NamingException
	 * @throws SQLException
	 * @throws MonitorNotFound
	 * 
	 * @see com.clustercontrol.monitor.bean.EventDataInfo
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeCountEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 */
	public ViewListInfo getEventList(String facilityId, EventFilterInfo filter, int messages)
	throws CreateException, NamingException, SQLException, MonitorNotFound {

		ViewListInfo ret = null;

		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;
		Integer confirmFlg = new Integer(ConfirmConstant.TYPE_UNCONFIRMED);
		String confirmUser = null;
		String comment = null;
		String commentUser = null;

		String[] facilityIds = null;

		Collection<EventLogLocal> ct = null;

		Integer limit = new Integer(0);

		try {
			if(filter != null){
				//重要度取得
				if (!"".equals(filter.getPriority()) && filter.getPriority() != null) {
					priority = filter.getPriority();
				}

				//更新日時（自）取得
				if (filter.getOutputDateFrom() instanceof Long) {
					outputFromDate = new Timestamp(filter.getOutputDateFrom());
					outputFromDate.setNanos(0);
				}

				//更新日時（至）取得
				if (filter.getOutputDateTo() instanceof Long) {
					outputToDate = new Timestamp(filter.getOutputDateTo());
					outputToDate.setNanos(999999999);
				}

				//出力日時（自）取得
				if (filter.getGenerationDateFrom() instanceof Long) {
					generationFromDate = new Timestamp(filter.getGenerationDateFrom());
					generationFromDate.setNanos(0);
				}

				//出力日時（至）取得
				if (filter.getGenerationDateTo() instanceof Long) {
					generationToDate = new Timestamp(filter.getGenerationDateTo());
					generationToDate.setNanos(999999999);
				}

				//対象ファシリティ種別取得
				if (!"".equals(filter.getFacilityType())) {
					facilityType = filter.getFacilityType();
				}

				//アプリケーション取得
				if (!"".equals(filter.getApplication())) {
					application = filter.getApplication();
				}

				//メッセージ取得
				if (!"".equals(filter.getMessage())) {
					message = filter.getMessage();
				}

				// 確認有無取得
				confirmFlg = filter.getConfirmFlgType();
				if (confirmFlg != null && confirmFlg == ConfirmConstant.TYPE_ALL){
					confirmFlg = null;
				}

				// 確認ユーザ取得
				if (!"".equals(filter.getConfirmedUser())) {
					confirmUser = filter.getConfirmedUser();
				}

				//コメント取得
				if (!"".equals(filter.getComment())){
					comment = filter.getComment();
				}

				//コメントユーザ取得
				if (!"".equals(filter.getCommentUser())){
					commentUser = filter.getCommentUser();
				}
			}

			// 対象ファシリティのファシリティIDを取得
			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			if (facilityId != null && !"".equals(facilityId)) {
				int level = RepositoryControllerBean.ALL;
				if (FacilityTargetConstant.STRING_BENEATH.equals(facilityType)) {
					level = RepositoryControllerBean.ONE_LEVEL;
				}

				ArrayList<String> facilityIdList = RepositoryControllerUtil
				.getLocalHome().create().getFacilityIdList(facilityId, level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else {
					// ノードの場合
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				}
			}

			if(messages <= 0){
				messages = MAX_DISPLAY_NUMBER;
			}
			limit = new Integer(messages + 1);

			// イベントログ情報一覧を取得
			ct = EventLogUtil.getLocalHome().findEvent(facilityIds,
					priority,
					outputFromDate,
					outputToDate,
					generationFromDate,
					generationToDate,
					application,
					message,
					confirmFlg,
					confirmUser,
					comment,
					commentUser,
					false,
					limit);

			// 2次元配列に変換
			ret = this.collectionToEventList(ct, messages);

		} catch (CreateException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw e;
		}

		return ret;
	}

	/**
	 * 重要度が最高で受信日時が最新のイベント情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ配下のファシリティを、指定されたファシリティのターゲットで取得します。</li>
	 * <li>取得したファシリティに属する重要度が最高 かつ 受信日時が最新の未確認のイベント情報を取得し返します。</li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @param orderFlg ソートの有無
	 * @return イベントのローカルコンポーネントインターフェース
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ALL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ONE_LEVEL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindHighPriorityEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 */
	protected EventLogLocal getHighPriorityEvent(String facilityId, int level, boolean orderFlg)
	throws CreateException, FinderException, NamingException {

		EventLogLocal event = null;
		try {
			String[] facilityIds = null;
			if (level == MonitorControllerBean.ONLY) {
				if (facilityId != null && !"".equals(facilityId)) {
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				} else {
					return null;
				}
			} else {
				// 直下 または 配下すべてのファシリティIDを取得
				ArrayList<String> facilityIdList = RepositoryControllerUtil
				.getLocalHome().create().getFacilityIdList(facilityId,
						level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					if(facilityId != null){
						facilityIdList.add(facilityId);
					}
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				} else {
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}

			// 重要度のリストを取得する
			int[] priorityList = PriorityConstant.PRIORITY_LIST;
			for(int i=0; i<priorityList.length; i++){
				// イベントログ情報一覧から重要度が危険のもので、最近出力されたイベントを取得する。
				Collection<EventLogLocal> ct = EventLogUtil.getLocalHome().findHighPriorityEvent(facilityIds,
						priorityList[i],
						null,
						null,
						null,
						null,
						null,
						null,
						new Integer(ConfirmConstant.TYPE_UNCONFIRMED),
						null,
						orderFlg);
				// 重要度の高いもの順にループされるため、取得できた場合は、それを返す。
				Iterator itr = ct.iterator();
				// イテレータで参照するが、0件か１件しかない。
				if (itr.hasNext()) {
					event = (EventLogLocal) itr.next();
					return event;
				}
			}

		} catch (CreateException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		}
		return event;
	}

	public void deleteEventFile(String filename) {
		File file = new File(ExportCollectedDataFile.exportDirectory + "/" + filename);
		file.delete();
	}

	/**
	 * 引数で指定された条件に一致する帳票出力用のイベント情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたプロパティに格納された検索条件を、プロパティユーティリティ（{@link com.clustercontrol.util.PropertyUtil}）を使用して取得します。</li>
	 * <li>引数で指定されたファシリティ配下のファシリティと検索条件を基に、イベント情報を取得します。</li>
	 * <li>１イベント情報を帳票出力用イベント情報（{@link com.clustercontrol.monitor.bean.ReportEventInfo}）にセットします。</li>
	 * <li>この帳票出力用イベント情報を、イベント情報一覧を保持するリスト（{@link ArrayList}）にセットし返します。<BR>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @throws CreateException
	 * @throws NamingException
	 * @throws IndexOutOfBoundsException
	 * @throws MonitorNotFound
	 * @throws IOException
	 * 
	 * @since 2.1.0
	 * 
	 * @see com.clustercontrol.util.PropertyUtil#getPropertyValue(com.clustercontrol.bean.Property, java.lang.String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.bean.ReportEventInfo
	 */
	public DataHandler getEventFile(String facilityId, EventFilterInfo filter, String filename, String username)
	throws CreateException, NamingException, IndexOutOfBoundsException, MonitorNotFound, IOException {

		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;
		Integer confirmFlg = null;
		String confirmUser = null;
		String comment = null;
		String commentUser = null;

		String filepath = ExportCollectedDataFile.exportDirectory + "/" + filename;
		File file = new File(filepath);
		if (UTF8_BOM) {
			FileOutputStream fos = new FileOutputStream(file);
			fos.write( 0xef );
			fos.write( 0xbb );
			fos.write( 0xbf );
			fos.close();
		}
		FileWriter filewriter = new FileWriter(file, true);

		try {
			//重要度取得
			if (!"".equals(filter.getPriority()) && filter.getPriority() != null) {
				priority = filter.getPriority();
			}

			//更新日時（自）取得
			if (filter.getOutputDateFrom() instanceof Long) {
				outputFromDate = new Timestamp(filter.getOutputDateFrom());
				outputFromDate.setNanos(0);
			}

			//更新日時（至）取得
			if (filter.getOutputDateTo() instanceof Long) {
				outputToDate = new Timestamp(filter.getOutputDateTo());
				outputToDate.setNanos(999999999);
			}

			//出力日時（自）取得
			if (filter.getGenerationDateFrom() instanceof Long) {
				generationFromDate = new Timestamp(filter.getGenerationDateFrom());
				generationFromDate.setNanos(0);
			}

			//出力日時（至）取得
			if (filter.getGenerationDateTo() instanceof Long) {
				generationToDate = new Timestamp(filter.getGenerationDateTo());
				generationToDate.setNanos(999999999);
			}

			//対象ファシリティ種別取得
			if (!"".equals(filter.getFacilityType())) {
				facilityType = filter.getFacilityType();
			}

			//アプリケーション取得
			if (!"".equals(filter.getApplication())) {
				application = filter.getApplication();
			}

			//メッセージ取得
			if (!"".equals(filter.getMessage())) {
				message = filter.getMessage();
			}

			// 確認有無取得
			int confirmFlgType = filter.getConfirmFlgType();
			if (confirmFlgType != -1) {
				confirmFlg = new Integer(confirmFlgType);
			}

			// 確認ユーザ
			if (!"".equals(filter.getConfirmedUser())) {
				confirmUser = filter.getConfirmedUser();
			}

			// コメント
			if (!"".equals(filter.getComment())){
				comment = filter.getComment();
			}

			// コメントユーザ
			if (!"".equals(filter.getCommentUser())){
				commentUser = filter.getCommentUser();
			}

			// ヘッダを追記
			/*
			イベント情報,,,,,,,,,,,,,
			出力日時,2012/02/08 17:20:10,,,,,,,,,,,,
			出力ユーザ,Hinemos Administrator,,,,,,,,,,,,
			重要度,受信日時,出力日時,ファシリティID,アプリケーション,確認,確認ユーザ,メッセージ,,,,,,
			,,,,,未,,,,,,,,
			 */
			SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT);
			filewriter.write(Messages.getString("report.title.monitor.event") + "\n");
			filewriter.write(Messages.getString("report.output.date") + SEPARATOR +
					sdf.format(new Date()) + "\n");
			filewriter.write(Messages.getString("report.output.user") + SEPARATOR + username + "\n");
			filewriter.write(
					Messages.getString("def.result") + SEPARATOR +
					Messages.getString("receive.time") + SEPARATOR +
					Messages.getString("report.output.date") + SEPARATOR +
					Messages.getString("facility.id") + SEPARATOR +
					Messages.getString("application") + SEPARATOR +
					Messages.getString("confirmed") + SEPARATOR +
					Messages.getString("confirm.user") + SEPARATOR +
					Messages.getString("message") + SEPARATOR +
					Messages.getString("comment") + SEPARATOR +
					Messages.getString("comment.user") +
			"\n");
			filewriter.write(
					(priority == null ? "" : PriorityConstant.typeToString(priority)) + SEPARATOR +
					(outputFromDate == null ? "" : sdf.format(new Date(outputFromDate.getTime()))) + " - " +
					(outputToDate == null ? "" : sdf.format(new Date(outputToDate.getTime()))) + SEPARATOR +
					(generationFromDate == null ? "" : sdf.format(new Date(generationFromDate.getTime()))) + " - " +
					(generationToDate == null ? "" : sdf.format(new Date(generationToDate.getTime()))) + SEPARATOR +
					(facilityType == null ? "" : facilityType) + SEPARATOR +
					(application == null ? "" : application) + SEPARATOR +
					(confirmFlg == null ? "" : ConfirmConstant.typeToString(confirmFlg)) + SEPARATOR +
					(confirmUser == null ? "" : confirmUser) + SEPARATOR +
					(message == null ? "" : message) + SEPARATOR +
					(comment == null ? "" : comment) + SEPARATOR +
					(commentUser == null ? "" : commentUser) + "\n");
			filewriter.write(
					Messages.getString("number") + SEPARATOR +
					Messages.getString("def.result") + SEPARATOR +
					Messages.getString("receive.time") + SEPARATOR +
					Messages.getString("report.output.date") + SEPARATOR +
					Messages.getString("facility.id") + SEPARATOR +
					Messages.getString("scope") + SEPARATOR +
					Messages.getString("monitor.id") + SEPARATOR +
					Messages.getString("monitor.detail.id") + SEPARATOR +
					Messages.getString("message.id") + SEPARATOR +
					Messages.getString("plugin.id") + SEPARATOR +
					Messages.getString("application") + SEPARATOR +
					Messages.getString("confirmed") + SEPARATOR +
					Messages.getString("confirm.user") + SEPARATOR +
					Messages.getString("comment") + SEPARATOR +
					Messages.getString("comment.date") + SEPARATOR +
					Messages.getString("comment.user") + SEPARATOR +
					Messages.getString("message") + SEPARATOR +
					Messages.getString("message.org") + "\n");

			// 対象ファシリティのファシリティIDを取得
			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			String[] facilityIds = null;
			if (facilityId != null && !"".equals(facilityId)) {

				int level = RepositoryControllerBean.ALL;
				if (FacilityTargetConstant.STRING_BENEATH.equals(facilityType)) {
					level = RepositoryControllerBean.ONE_LEVEL;
				}

				ArrayList<String> facilityIdList = RepositoryControllerUtil
				.getLocalHome().create().getFacilityIdList(facilityId, level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else {
					// ノードの場合
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				}
			}

			// イベントログ情報一覧を取得
			Collection<EventLogLocal> ct = EventLogUtil.getLocalHome().findEvent(
					facilityIds,
					priority,
					outputFromDate,
					outputToDate,
					generationFromDate,
					generationToDate,
					application,
					message,
					confirmFlg,
					confirmUser,
					comment,
					commentUser,
					true,
					MAX_REPORT_EVENT_COUNT);

			// 帳票出力用に変換
			collectionToFile(ct, filewriter);

		} catch (CreateException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg, comment, commentUser);
			throw e;
		} finally {
			filewriter.close();
		}

		// リストをファイルに書き出し。
		FileDataSource source = new FileDataSource(file);
		DataHandler handler = new DataHandler(source);

		return handler;
	}

	/**
	 * DBより取得したイベント情報をイベント一覧情報に格納します。
	 * <p>
	 * <ol>
	 * <li>１イベント情報をEventLogDataのインスタンスとし、イベント情報一覧を保持するリスト（{@link ArrayList}）に格納します。<BR>
	 * <li>イベント情報一覧を、引数で指定されたビュー一覧情報（{@link com.clustercontrol.bean.ViewListInfo}）にセットします。</li>
	 * </ol>
	 * <p>
	 * また、イベント数を重要度毎にカウントし、
	 * 表示イベント数よりもイベント数が少ない場合は、重要度別イベント数を引数で指定されたビュー一覧情報にセットします。
	 * 
	 * @param ct イベント情報取得結果
	 * @param eventList ビュー一覧情報
	 * @param messages イベント最大表示件数
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 */
	private ViewListInfo collectionToEventList(Collection<EventLogLocal> ct, int messages) {
		int critical = 0;
		int warning = 0;
		int info = 0;
		int unknown = 0;

		ViewListInfo viewListInfo = new ViewListInfo();
		ArrayList<EventDataInfo> list = new ArrayList<EventDataInfo>();

		for (EventLogLocal event : ct) {

			EventDataInfo eventInfo = new EventDataInfo();
			eventInfo.setPriority(event.getPriority());
			if (event.getOutputDate() != null) {
				eventInfo.setOutputDate(event.getOutputDate().getTime());
			}
			if (event.getGenerationDate() != null) {
				eventInfo.setGenerationDate(event.getGenerationDate().getTime());
			}
			eventInfo.setPluginId(event.getPluginId());
			eventInfo.setMonitorId(event.getMonitorId());
			eventInfo.setMonitorDetailId(event.getMonitorDetailId());
			eventInfo.setFacilityId(event.getFacilityId());
			eventInfo.setScopeText(event.getScopeText());
			eventInfo.setApplication(event.getApplication());
			eventInfo.setMessageId(event.getMessageId());
			eventInfo.setMessage(event.getMessage());
			eventInfo.setConfirmed(event.getConfirmFlg());
			eventInfo.setConfirmUser(event.getConfirmUser());
			eventInfo.setComment(event.getComment());
			if (event.getCommentDate() != null ) {
				eventInfo.setCommentDate(event.getCommentDate().getTime());
			}
			eventInfo.setCommentUser(event.getCommentUser());

			list.add(eventInfo);

			//最大表示件数以下の場合
			if(event.getPriority().intValue() == PriorityConstant.TYPE_CRITICAL)
				critical++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_WARNING)
				warning++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_INFO)
				info++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_UNKNOWN)
				unknown++;

			//取得したイベントを最大表示件数まで格納したら終了
			if(list.size() >= messages)
				break;
		}

		//イベント数を設定
		viewListInfo.setCritical(critical);
		viewListInfo.setWarning(warning);
		viewListInfo.setInfo(info);
		viewListInfo.setUnKnown(unknown);
		viewListInfo.setTotal(ct.size());

		viewListInfo.setEventList(list);

		return viewListInfo;
	}

	/**
	 * DBより取得したイベント情報を帳票出力用イベント情報一覧に格納します。
	 * 
	 * @param ct イベント情報取得結果
	 * @return 帳票出力用イベント情報一覧
	 * 
	 * @version 2.1.0
	 * @throws IOException
	 * @since 2.1.0
	 */
	private void collectionToFile(Collection<EventLogLocal> ct, FileWriter filewriter) throws IOException {

		int n = 0;
		SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT);
		String outputDate;
		String generationDate;
		String commentDate;
		for (EventLogLocal event : ct) {
			n ++;
			if (event.getOutputDate() != null) {
				outputDate = sdf.format(new Date(event.getOutputDate().getTime()));
			} else {
				outputDate = "";
			}
			if (event.getGenerationDate() != null) {
				generationDate = sdf.format(new Date(event.getGenerationDate().getTime()));
			} else {
				generationDate = "";
			}
			if (event.getCommentDate() != null) {
				commentDate = sdf.format(new Date(event.getCommentDate().getTime()));
			} else {
				commentDate = "";
			}
			filewriter.write(
					n + SEPARATOR +
					PriorityConstant.typeToString(event.getPriority()) + SEPARATOR +
					outputDate + SEPARATOR +
					generationDate + SEPARATOR +
					event.getFacilityId() + SEPARATOR +
					event.getScopeText() + SEPARATOR +
					event.getMonitorId() + SEPARATOR +
					event.getMonitorDetailId() + SEPARATOR +
					event.getMessageId() + SEPARATOR +
					event.getPluginId() + SEPARATOR +
					event.getApplication() + SEPARATOR +
					ConfirmConstant.typeToString(event.getConfirmFlg()) + SEPARATOR +
					event.getConfirmDate() + SEPARATOR +
					event.getConfirmUser() + SEPARATOR +
					event.getComment() + SEPARATOR +
					commentDate + SEPARATOR +
					event.getCommentUser() + SEPARATOR +
					event.getMessage() + SEPARATOR +
					event.getMessageOrg() +
			"\n");
		}
	}

	/**
	 * アプリケーションログにログを出力
	 * 
	 * @param e 例外
	 */
	private void outputLog(
			Exception e,
			String method,
			String facilityId,
			Integer priority,
			Timestamp outputFromDate,
			Timestamp outputToDate,
			Timestamp generationFromDate,
			Timestamp generationToDate,
			String facilityType,
			String application,
			String message,
			Integer confirmFlg,
			String comment,
			String commentUser){

		int priorityInt = -1;
		String outputFromDateString = null;
		String outputToDateString = null;
		String generationFromDateString = null;
		String generationToDateString = null;
		int confirmFlgInt = -1;

		if(priority != null)
			priorityInt = priority.intValue();
		if(outputFromDate != null)
			outputFromDateString = outputFromDate.toString();
		if(outputToDate != null)
			outputToDateString = outputToDate.toString();
		if(generationFromDate != null)
			generationFromDateString = generationFromDate.toString();
		if(generationToDate != null)
			generationToDateString = generationToDate.toString();
		if(confirmFlg != null)
			confirmFlgInt = confirmFlg.intValue();

		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message,ConfirmConstant.typeToString(confirmFlgInt)};
		apllog.put("SYS", "003", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
