package com.clustercontrol.monitor.run.util;

import java.util.ArrayList;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.custom.bean.CustomCheckInfo;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.logfile.bean.LogfileCheckInfo;
import com.clustercontrol.custom.CustomConstant;
import com.clustercontrol.monitor.bean.MonitorTypeMstConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.performance.monitor.bean.PerfCheckInfo;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.ping.bean.PingCheckInfo;
import com.clustercontrol.port.bean.PortCheckInfo;
import com.clustercontrol.port.ejb.entity.MonitorProtocolMasterUtil;
import com.clustercontrol.process.bean.ProcessCheckInfo;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.snmp.bean.SnmpCheckInfo;
import com.clustercontrol.snmptrap.bean.MonitorTrapConstant;
import com.clustercontrol.snmptrap.bean.TrapCheckInfo;
import com.clustercontrol.sql.bean.SqlCheckInfo;
import com.clustercontrol.util.Messages;
import com.clustercontrol.winservice.bean.WinServiceCheckInfo;

public class MonitorValidator {

	private static Log m_log = LogFactory.getLog( MonitorValidator.class );

	/**
	 * 監視設定(MonitorInfo)の妥当性チェック
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public static void validateMonitorInfo(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{

		// 監視共通
		validateMonitorCommonSettings(monitorInfo);

		// 監視種別のチェック
		int monitorType = monitorInfo.getMonitorType();
		if(monitorType == MonitorTypeConstant.TYPE_TRUTH){
			validateMonitorTruthSettings(monitorInfo);
		}else if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			validateMonitorNumericSettings(monitorInfo);
		}else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			validateMonitorStringSettings(monitorInfo);
		}else if(monitorType == MonitorTypeConstant.TYPE_TRAP){
			validateMonitorTrapSettings(monitorInfo);
		}else{
			throw new InvalidSetting("Invalid Monitor Type. monitorType = " + monitorInfo.getMonitorType());
		}

		// 監視種別ID(プラグインID)のチェック
		String monitorTypeId = monitorInfo.getMonitorTypeId();
		if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
			validateHinemosAgent(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
			if(monitorType == MonitorTypeConstant.TYPE_NUMERIC) {
				validateHttpNumeric(monitorInfo);
			} else if (monitorType == MonitorTypeConstant.TYPE_STRING){
				validateHttp(monitorInfo);
			}else{
				throw new InvalidSetting("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
			validatePerformance(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
			validatePing(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
			validatePort(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
			validateProcess(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
			if(monitorType == MonitorTypeConstant.TYPE_NUMERIC) {
				validateSnmpNumeric(monitorInfo);
			} else if (monitorType == MonitorTypeConstant.TYPE_STRING){
				validateSnmp(monitorInfo);
			}else{
				throw new InvalidSetting("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
			if(monitorType == MonitorTypeConstant.TYPE_NUMERIC) {
				validateSqlNumeric(monitorInfo);
			} else if (monitorType == MonitorTypeConstant.TYPE_STRING){
				validateSql(monitorInfo);
			}else{
				throw new InvalidSetting("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
			validateSystemlog(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
			validateLogfile(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
			validateCustom(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
			validateSnmptrap(monitorInfo);
		}else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
			validateWinService(monitorInfo);
		}else {
			throw new InvalidSetting("Invalid monitorTypeId. monitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

	}


	/**
	 * 監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateMonitorCommonSettings(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateMonitorCommonSettings() monitorId = " + monitorInfo.getMonitorId());

		//
		// 共通項目を対象とする。ただし、監視間隔については各監視機能で実装する(トラップ系の監視があるため)
		// 数値監視、文字列監視、真偽値監視は各継承したクラスで実装する
		//

		// monitorId
		if (monitorInfo.getMonitorId() == null || monitorInfo.getMonitorId().length() == 0) {
			throw new InvalidSetting(Messages.getString("message.monitor.1"));
		}
		CommonValidator.validateId(Messages.getString("monitor.id"), monitorInfo.getMonitorId(), 64);

		// monitorTypeId
		// monitorType
		boolean flag = true;
		for (ArrayList<Object> a : MonitorTypeMstConstant.getListAll()) {
			if (a.get(0).equals(monitorInfo.getMonitorTypeId()) &&
					a.get(1).equals(monitorInfo.getMonitorType())) {
				flag = false;
				break;
			}
		}
		if (flag) {
			throw new InvalidSetting("Invalid MonitorType. monitorTyeId(pluginId) = " + monitorInfo.getMonitorTypeId() + ", monitorType = " + monitorInfo.getMonitorType());
		}


		// description : not implemented
		CommonValidator.validateString(Messages.getString("description"),
				monitorInfo.getDescription(), false, 1, 256);

		// facilityId
		if(monitorInfo.getFacilityId() == null || "".equals(monitorInfo.getFacilityId())){
			throw new InvalidSetting(Messages.getString("message.hinemos.3"));
		}else{
			try{
				FacilityUtil.getLocalHome().findByPrimaryKey(monitorInfo.getFacilityId());
			} catch (NamingException e) {
				throw new HinemosUnknown("add monitor unknown error. FacilityId = " + monitorInfo.getFacilityId(), e);
			} catch (FinderException e) {
				throw new InvalidSetting("FacilityId is not exist in repository. FacilityId = " + monitorInfo.getFacilityId());
			}
		}

		// runInterval : not implemented
		if(monitorInfo.getRunInterval() != RunIntervalConstant.TYPE_MIN_01
				&& monitorInfo.getRunInterval() != RunIntervalConstant.TYPE_MIN_05
				&& monitorInfo.getRunInterval() != RunIntervalConstant.TYPE_MIN_10
				&& monitorInfo.getRunInterval() != RunIntervalConstant.TYPE_MIN_30
				&& monitorInfo.getRunInterval() != RunIntervalConstant.TYPE_MIN_60){

			// if polling type monitoring
			if(!HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorInfo.getMonitorTypeId()) &&
					!HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorInfo.getMonitorTypeId()) &&
					!HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorInfo.getMonitorTypeId())){
				throw new InvalidSetting("RunInterval is not 1 min / 5 min / 10 min / 30 min / 60 min.");
			}
		}

		// delayTime : not implemented
		// triggerType : not implemented

		// calendarId
		CommonValidator.validateCalenderId(monitorInfo.getCalendarId(), false);

		// failurePriority : not implemented

		// application
		if(monitorInfo.getMonitorFlg() == YesNoConstant.TYPE_YES){
			CommonValidator.validateString(Messages.getString("application"),
					monitorInfo.getApplication(), true, 1, 64);
		}

		// notifyGroupId : not implemented

		// notifyId
		if(monitorInfo.getNotifyId() != null){
			for(NotifyRelationInfo notifyInfo : monitorInfo.getNotifyId()){
				CommonValidator.validateNotifyId(notifyInfo.getNotifyId(), true);
			}
		}

		// monitorFlg : not implemented
		// collectorFlg
		if(monitorInfo.getCollectorFlg() == YesNoConstant.TYPE_YES){
			if(monitorInfo.getMonitorType() != MonitorTypeConstant.TYPE_NUMERIC){
				throw new InvalidSetting("CollectorFlg is true. but this monitorType is not numeric.");
			}
		}

	}

	/**
	 * 文字列用監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateMonitorStringSettings(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateMonitorStringSettings() monitorId = " + monitorInfo.getMonitorId());

		ArrayList<MonitorStringValueInfo> stringValueInfoList = monitorInfo.getStringValueInfo();
		if(stringValueInfoList == null || stringValueInfoList.size() == 0){
			if(monitorInfo.getMonitorFlg() == YesNoConstant.TYPE_YES){
				throw new InvalidSetting(Messages.getString("message.monitor.27"));
			}
		}else{
			for(MonitorStringValueInfo info : stringValueInfoList){

				// monitorId : not implemented

				// orderNo : not implemented

				// description : not implemented
				String description = info.getDescription();
				if (description != null) {
					CommonValidator.validateString(Messages.getString("description"),
							description, true, 0, 256);
				}

				// processType : not implemented

				// pattern
				if(info.getPattern() == null ){
					throw new InvalidSetting("Pattern is not defined. monitorId = " + monitorInfo.getMonitorId() + ", orderNo = " + info.getOrderNo());
				} else if ("".equals(info.getPattern()) && info.getProcessType() == ProcessConstant.TYPE_YES){
					throw new InvalidSetting("Pattern is empty string. monitorId = " + monitorInfo.getMonitorId() + ", orderNo = " + info.getOrderNo());
				} else {
					CommonValidator.validateString(Messages.getString("pattern.matching.expression"),
							info.getPattern(), true, 1, 1024);
				}
				try{
					Pattern.compile(info.getPattern());
				}
				catch(PatternSyntaxException e){
					throw new InvalidSetting("Pattern is not regular expression. monitorId = " + monitorInfo.getMonitorId() + ", orderNo = " + info.getOrderNo(), e);
				}
				// priority : not implemented

				// message
				if(info.getProcessType() == ProcessConstant.TYPE_YES){
					CommonValidator.validateString(Messages.getString("message"), info.getMessage(), true, 1, 256);
				}

				// caseSensitivityFlg : not implemented

				// validFlg : not implemented

			}
		}
	}

	/**
	 * 数値用監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateMonitorNumericSettings(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		// itemName
		if(monitorInfo.getCollectorFlg() == YesNoConstant.TYPE_YES){
			CommonValidator.validateString(Messages.getString("collection.display.name"), monitorInfo.getItemName(), true, 1, 64);

			// measure
			CommonValidator.validateString(Messages.getString("collection.unit"), monitorInfo.getMeasure(), true, 1, 64);
		}
	}

	private static void validateNumeric(MonitorInfo monitorInfo, int timeout)
	throws InvalidSetting {

		if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
			return;
		}

		Double infoLower = (monitorInfo.getNumericValueInfo().get(0)).getThresholdLowerLimit();
		Double infoUpper = (monitorInfo.getNumericValueInfo().get(0)).getThresholdUpperLimit();
		Double warnLower = (monitorInfo.getNumericValueInfo().get(1)).getThresholdLowerLimit();
		Double warnUpper = (monitorInfo.getNumericValueInfo().get(1)).getThresholdUpperLimit();
		int runInterval = monitorInfo.getRunInterval();

		if (infoLower == null || infoUpper == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.7"));
		}
		if (warnLower == null || warnUpper == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.8"));
		}
		CommonValidator.validateDouble(Messages.getString("info"), infoLower, DataRangeConstant.DOUBLE_LOW, DataRangeConstant.DOUBLE_HIGH);
		CommonValidator.validateDouble(Messages.getString("info"), infoUpper, DataRangeConstant.DOUBLE_LOW, DataRangeConstant.DOUBLE_HIGH);
		CommonValidator.validateDouble(Messages.getString("warn"), warnLower, DataRangeConstant.DOUBLE_LOW, DataRangeConstant.DOUBLE_HIGH);
		CommonValidator.validateDouble(Messages.getString("warn"), warnUpper, DataRangeConstant.DOUBLE_LOW, DataRangeConstant.DOUBLE_HIGH);

		// ping監視のみ通常のinfo/warnの閾値ではない
		if(!HinemosModuleConstant.MONITOR_PING.equals(monitorInfo.getMonitorTypeId())){
			if (infoLower > infoUpper) {
				throw new InvalidSetting(Messages.getString("message.monitor.7"));
			}
			if (warnLower > warnUpper) {
				throw new InvalidSetting(Messages.getString("message.monitor.8"));
			}
		}

		if (timeout < 0) {
			return;
		}
		// 間隔よりタイムアウトが大きい場合
		if (runInterval*1000 < timeout) {
			throw new InvalidSetting(Messages.getString("message.monitor.43"));
		}

		if(HinemosModuleConstant.MONITOR_PING.equals(monitorInfo.getMonitorTypeId())){
			if (timeout < infoLower) {
				throw new InvalidSetting(Messages.getString("message.monitor.50"));
			}
			if (timeout < warnLower) {
				throw new InvalidSetting(Messages.getString("message.monitor.51"));
			}
		} else {
			// タイムアウトより通知の上限が大きい場合
			if (timeout < infoUpper) {
				throw new InvalidSetting(Messages.getString("message.monitor.50"));
			}
			// タイムアウトより警告の上限が大きい場合
			if (timeout < warnUpper) {
				throw new InvalidSetting(Messages.getString("message.monitor.51"));
			}
		}
	}

	/**
	 * 真偽値用監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateMonitorTruthSettings(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateMonitorTruthSettings() monitorId = " + monitorInfo.getMonitorId());
		m_log.debug("validateMonitorTruthSettings() is not implemented. ");
	}



	/**
	 * TRAp用監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateMonitorTrapSettings(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateMonitorTrapSettings() monitorId = " + monitorInfo.getMonitorId());
		m_log.debug("validateMonitorTrapSettings() is not implemented. ");
	}

	/**
	 * Hinemosエージェント監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateHinemosAgent(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateHinemosAgent() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo : not impletemted( check info is not exists)
		m_log.debug("validateHinemosAgent() is not needed. ");

		// monitorType
		if(!HinemosModuleConstant.MONITOR_AGENT.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Agent Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}
	}

	/**
	 * HTTP監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateHttpNumeric(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		validateHttp(monitorInfo);

		HttpCheckInfo checkInfo = monitorInfo.getHttpCheckInfo();

		// input validate
		validateNumeric(monitorInfo, checkInfo.getTimeout());

	}

	/**
	 * HTTP監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateHttp(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateHttp() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		HttpCheckInfo checkInfo = monitorInfo.getHttpCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("HTTP Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_HTTP.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is HTTP Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// requestUrl
		if(checkInfo.getRequestUrl() == null || "".equals(checkInfo.getRequestUrl())){
			throw new InvalidSetting(Messages.getString("message.http.1"));
		}else{
			String url = checkInfo.getRequestUrl();
			// format check
			if (url.length() > 0 && (!url.startsWith("http://") && !url.startsWith("https://"))) {
				throw new InvalidSetting(Messages.getString("message.http.5"));
			}
			else if((url.startsWith("http://") && url.length() == 7) || (url.startsWith("https://") && url.length() == 8)){
				throw new InvalidSetting(Messages.getString("message.http.5"));
			}
		}
		CommonValidator.validateString(Messages.getString("request.url"),
				checkInfo.getRequestUrl(), true, 8, 2083);

		// urlReplace : not implemented

		// timeout : not implemented
		if(checkInfo.getTimeout() == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.timeout.undef"));
		}
		CommonValidator.validateInt(Messages.getString("time.out"),
				checkInfo.getTimeout(), 1, DataRangeConstant.SMALLINT_HIGH);

		// proxySet : not implemented

		// proxyHost : not implemented

		// proxyPort : not implemented
	}

	/**
	 * ログファイル監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateLogfile(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateLogfile() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		LogfileCheckInfo checkInfo = monitorInfo.getLogfileCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Logfile Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Logfile Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// logfile
		if(checkInfo.getLogfile() == null || "".equals(checkInfo.getLogfile())){
			throw new InvalidSetting(Messages.getString("message.logfile.1"));
		}
		CommonValidator.validateString(Messages.getString("file.path"), checkInfo.getLogfile(), true, 1, 1024);
	}

	/**
	 * リソース監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validatePerformance(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validatePerformance() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		PerfCheckInfo checkInfo = monitorInfo.getPerfCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Performance Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Performance Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// itemCode
		if(checkInfo.getItemCode() == null || "".equals(checkInfo.getItemCode())){
			throw new InvalidSetting(Messages.getString("message.monitor.57"));
		}else{
			try{
				CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(checkInfo.getItemCode());
			} catch (NamingException e) {
				throw new HinemosUnknown("add monitor unknown error. itemCode = " + checkInfo.getItemCode(), e);
			} catch (FinderException e) {
				throw new InvalidSetting("itemCode is not exist. itemCode = " + checkInfo.getItemCode());
			}
		}

		// deviceDisplayName : not implemented
		if(checkInfo.getDeviceDisplayName() == null){
			throw new InvalidSetting("Target Display Name is not defined.");
		}
		// breakdownFlg : not implemented

		validateNumeric(monitorInfo, -1);
	}

	/**
	 * ping監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validatePing(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validatePing() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		PingCheckInfo checkInfo = monitorInfo.getPingCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Ping Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_PING.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Ping Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// runCount : not implemented

		// runInterval : not implemented

		// timeout : not implemented
		if(checkInfo.getTimeout() == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.timeout.undef"));
		}
		CommonValidator.validateInt(Messages.getString("time.out"),
				checkInfo.getTimeout(), 1, DataRangeConstant.SMALLINT_HIGH);

		// input validate
		int runInterval = monitorInfo.getRunInterval();
		int runCount = checkInfo.getRunCount();
		int interval = checkInfo.getRunInterval();
		int timeout = checkInfo.getTimeout();

		// 間隔よりチェック設定の「回数×タイムアウト＋間隔」が大きい場合
		double total = runCount * ((double)timeout / 1000) + ((double)interval / 1000);
		if (runInterval <= (int)total) {
			throw new InvalidSetting(Messages.getString("message.monitor.52"));
		}
		validateNumeric(monitorInfo, timeout);

		// パケット紛失(%)は0-100の間
		CommonValidator.validateDouble(Messages.getString("ping.reach"),
				monitorInfo.getNumericValueInfo().get(0).getThresholdUpperLimit(),
				0f,100f);
		CommonValidator.validateDouble(Messages.getString("ping.reach"),
				monitorInfo.getNumericValueInfo().get(1).getThresholdUpperLimit(),
				0f,100f);
	}

	/**
	 * ポート監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validatePort(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validatePort() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		PortCheckInfo checkInfo = monitorInfo.getPortCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Port Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_PORT.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Port Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// portNo : not implemented
		if(checkInfo.getPortNo() == null) {
			throw new InvalidSetting(Messages.getString("message.port.8"));
		}
		CommonValidator.validateInt(Messages.getString("port.number"), checkInfo.getPortNo(), 1, 65535);

		// runCount : not implemented

		// runInterval : not implemented
		if(checkInfo.getRunInterval() == null) {
			throw new InvalidSetting(Messages.getString("message.port.2"));
		}

		// timeout : not implemented
		if(checkInfo.getTimeout() == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.timeout.undef"));
		}
		CommonValidator.validateInt(Messages.getString("time.out"),
				checkInfo.getTimeout(), 1, DataRangeConstant.SMALLINT_HIGH);

		// serviceId : not implemented
		if(checkInfo.getServiceId() == null || "".equals(checkInfo.getServiceId())){
			throw new InvalidSetting(Messages.getString("message.port.7"));
		}else{
			try{
				MonitorProtocolMasterUtil.getLocalHome().findByPrimaryKey(checkInfo.getServiceId());
			} catch (NamingException e) {
				throw new HinemosUnknown("add monitor unknown error. ServiceId = " + checkInfo.getServiceId(), e);
			} catch (FinderException e) {
				throw new InvalidSetting("ServiceId is not exist. ServiceId = " + checkInfo.getServiceId());
			}
		}

		// input validate
		int runInterval = monitorInfo.getRunInterval();
		int runCount = checkInfo.getRunCount();
		int interval = checkInfo.getRunInterval();
		int timeout = checkInfo.getTimeout();

		// 間隔よりチェック設定の「回数×タイムアウト＋間隔」が大きい場合
		double total = runCount * ((double)timeout / 1000) + ((double)interval / 1000);
		if (runInterval <= (int)total) {
			throw new InvalidSetting(Messages.getString("message.monitor.52"));
		}

		validateNumeric(monitorInfo, timeout);
	}

	/**
	 * プロセス監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateProcess(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateProcess() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		ProcessCheckInfo checkInfo = monitorInfo.getProcessCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Process Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_PROCESS.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Process Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// command
		if(checkInfo.getCommand() == null || "".equals(checkInfo.getCommand())){
			throw new InvalidSetting(Messages.getString("message.process.1"));
		}
		CommonValidator.validateString(Messages.getString("command"),
				checkInfo.getCommand(), true, 1, 256);
		try{
			Pattern.compile(checkInfo.getCommand());
		}
		catch(PatternSyntaxException e){
			throw new InvalidSetting(Messages.getString("message.process.2"), e);
		}

		// param
		if (checkInfo.getParam() != null) {
			CommonValidator.validateString(Messages.getString("param"),
					checkInfo.getParam(), false, 1, 256);
			try {
				Pattern.compile(checkInfo.getParam());
			} catch(PatternSyntaxException e) {
				throw new InvalidSetting(Messages.getString("message.process.3"));
			}
		}

		validateNumeric(monitorInfo, -1);
	}

	/**
	 * SNMPトラップ用監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSnmptrap(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateSnmptrap() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		TrapCheckInfo checkInfo = monitorInfo.getTrapCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("SNMP Trap Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is SNMP Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// communityName
		if(checkInfo.getCommunityCheck() == MonitorTrapConstant.COMMUNITY_CHECK_ON){
			if(checkInfo.getCommunityName() == null || "".equals(checkInfo.getCommunityName())){
				throw new InvalidSetting(Messages.getString("message.snmptrap.1"));
			}
			CommonValidator.validateString(Messages.getString("community.name"),
					checkInfo.getCommunityName(), true, 1, 64);
		}

		// checkMode : not implemented

		// communityCheck : not implemented

		// charsetConvert : not implemented
		CommonValidator.validateString(Messages.getString("charset.snmptrap.code"), checkInfo.getCharsetName(),
				false, 1, 64);

		// charsetName
		if(checkInfo.getCharsetConvert() == MonitorTrapConstant.CHARSET_CONVERT_ON){
			if(checkInfo.getCharsetName() == null || "".equals(checkInfo.getCharsetName())){
				throw new InvalidSetting(Messages.getString("message.snmptrap.4"));
			}
		}
	}

	/**
	 * SNMP監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSnmpNumeric(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		validateSnmp(monitorInfo);

		validateNumeric(monitorInfo, -1);
	}

	/**
	 * SNMP監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSnmp(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateSnmp() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		SnmpCheckInfo checkInfo = monitorInfo.getSnmpCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("SNMP Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_SNMP.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is SNMP Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// communityName : not implemented

		// convertFlg
		if(checkInfo.getConvertFlg() == null) {
			throw new InvalidSetting(Messages.getString("message.snmp.3"));
		}

		// snmpOid
		if(checkInfo.getSnmpOid() == null || "".equals(checkInfo.getSnmpOid())){
			throw new InvalidSetting(Messages.getString("message.snmp.2"));
		}
		CommonValidator.validateString(Messages.getString("oid"),
				checkInfo.getSnmpOid(), true, 1, 1024);

		// snmpPort : not implemented

		// snmpVersion : not implemented
	}

	/**
	 * SQL監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSqlNumeric(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		validateSql(monitorInfo);

		validateNumeric(monitorInfo, -1);
	}

	/**
	 * SQL監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSql(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateSql() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		SqlCheckInfo checkInfo = monitorInfo.getSqlCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("SQL Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_SQL.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is SQL Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// connectionUrl
		String url = checkInfo.getConnectionUrl();
		if(url == null || "".equals(url) || url.length() < 6 ||
				!url.startsWith("jdbc:")){
			throw new InvalidSetting(Messages.getString("message.sql.8"));
		}
		CommonValidator.validateString(Messages.getString("connection.url"),
				checkInfo.getConnectionUrl(), true, 1, 256);

		// user
		if(checkInfo.getUser() == null || "".equals(checkInfo.getUser())){
			throw new InvalidSetting(Messages.getString("message.sql.2"));
		}
		CommonValidator.validateString(Messages.getString("user"),
				checkInfo.getUser(), true, 1, 64);

		// password
		if(checkInfo.getPassword() == null || "".equals(checkInfo.getPassword())){
			throw new InvalidSetting(Messages.getString("message.sql.3"));
		}
		CommonValidator.validateString(Messages.getString("password"),
				checkInfo.getPassword(), true, 1, 64);

		// query
		if(checkInfo.getQuery() == null || checkInfo.getQuery().length() < 7){
			throw new InvalidSetting(Messages.getString("message.sql.5"));
		}else {
			String work = checkInfo.getQuery().substring(0, 6);
			if(!work.equalsIgnoreCase("SELECT")){
				throw new InvalidSetting(Messages.getString("message.sql.5"));
			}
		}
		CommonValidator.validateString(Messages.getString("sql.string"),
				checkInfo.getQuery(), true, 1, 1024);

		// jdbcDriver
		if(checkInfo.getJdbcDriver() == null || "".equals(checkInfo.getJdbcDriver())){
			throw new InvalidSetting(Messages.getString("message.sql.1"));
		}
	}

	/**
	 * システムログ監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateSystemlog(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateSystemlog() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo : not impletemted( check info is not exists)
		m_log.debug("validateSystemlog() is not needed. ");

		// monitorType
		if(!HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Systemlog Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}
	}

	/**
	 * コマンド監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateCustom(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateCustom() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		CustomCheckInfo checkInfo = monitorInfo.getCustomCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Custom Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Custom Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// execType
		if(checkInfo.getCommandExecType() == null &&
				(checkInfo.getCommandExecType().equals(CustomConstant.CommandExecType.SELECTED) || checkInfo.getCommandExecType().equals(CustomConstant.CommandExecType.INDIVIDUAL))){
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.type.undef"));
		}

		// selectedFacilityId
		if(checkInfo.getCommandExecType().equals(CustomConstant.CommandExecType.SELECTED)){
			if(checkInfo.getSelectedFacilityId() == null || "".equals(checkInfo.getSelectedFacilityId())){
				throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.node.undef"));
			}

			try{
				NodeUtil.getLocalHome().findByPrimaryKey(checkInfo.getSelectedFacilityId());
			} catch (NamingException e) {
				throw new HinemosUnknown("add command monitor unknown error. SelectedFacilityId = " + checkInfo.getSelectedFacilityId(), e);
			} catch (FinderException e) {
				throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.node.undef"));
			}
		}
		// effectiveUser
		if(checkInfo.getEffectiveUser() == null || "".equals(checkInfo.getEffectiveUser())){
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.effectiveuser.undef"));
		}
		CommonValidator.validateString(Messages.getString("effective.user"), checkInfo.getEffectiveUser(),
				true, 1, 64);

		// command
		if(checkInfo.getCommand() == null || "".equals(checkInfo.getCommand())){
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.command.undef"));
		}
		CommonValidator.validateString(Messages.getString("command"), checkInfo.getCommand(),
				true, 1, 1024);

		// timeout
		if(checkInfo.getTimeout() == null) {
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.timeout.undef"));
		}
		CommonValidator.validateInt(Messages.getString("time.out"),
				checkInfo.getTimeout(), 1, DataRangeConstant.SMALLINT_HIGH);
		int timeout = checkInfo.getTimeout();
		if (monitorInfo.getRunInterval() * 1000 < timeout) {
			throw new InvalidSetting(Messages.getString("message.monitor.custom.msg.timeout.toolarge"));
		}

		validateNumeric(monitorInfo, -1);
	}

	/**
	 * Windowsサービス監視設定(MonitorInfo)の基本設定の妥当性チェック（関連テーブルへのリンク & NULL CHECK）
	 * @param monitorInfo
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	private static void validateWinService(MonitorInfo monitorInfo) throws InvalidSetting, HinemosUnknown{
		if(monitorInfo == null){
			throw new InvalidSetting("MonitorInfo is not defined.");
		}
		m_log.debug("validateWinService() monitorId = " + monitorInfo.getMonitorId());

		// CheckInfo
		WinServiceCheckInfo checkInfo = monitorInfo.getWinServiceCheckInfo();
		if(checkInfo == null){
			throw new InvalidSetting("Windows Service Monitor Setting is not defined. monitorId = " + monitorInfo.getMonitorId());
		}

		// monitorType
		if(!HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorInfo.getMonitorTypeId())){
			throw new InvalidSetting("This is Windows Service Monitor Setting. But MonitorTypeId = " + monitorInfo.getMonitorTypeId());
		}

		// serviceName
		if(checkInfo.getServiceName() == null || "".equals(checkInfo.getServiceName())){
			throw new InvalidSetting(Messages.getString("message.winservice.1"));
		}
		CommonValidator.validateString(Messages.getString("winservice.name"),
				checkInfo.getServiceName(), true, 1, 1024);
	}
}
