/*

Copyright (C) 2009 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.process.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.process.ejb.entity.MonitorProcessMethodMstLocal;
import com.clustercontrol.process.ejb.entity.MonitorProcessMethodMstPK;
import com.clustercontrol.process.ejb.entity.MonitorProcessMethodMstUtil;
import com.clustercontrol.process.ejb.entity.MonitorProcessPollingMstData;
import com.clustercontrol.process.ejb.entity.MonitorProcessPollingMstPK;
import com.clustercontrol.process.factory.ProcessMasterCache;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ポーリングに関する情報をDBから取得するクラス<BR>
 *
 * @version 4.0.0
 * @since 3.1.0
 */
public class PollingDataManager {

	private static Log m_log = LogFactory.getLog( PollingDataManager.class );

	private String m_platformId = "";
	private String m_subPlatformId = "";

	/**
	 * コンストラクタ
	 * 
	 * @param ファシリティID
	 */
	public PollingDataManager(String facilityId){
		m_log.debug("PollingDataManager() facilityId = " + facilityId);

		RepositoryControllerLocal repositoryLocal;
		try {
			repositoryLocal = RepositoryControllerUtil.getLocalHome().create();

			// プラットフォームIDとサブプラットフォームIDを問い合わせる
			NodeInfo info = repositoryLocal.getNode(facilityId);

			m_platformId = info.getPlatformFamily();
			m_subPlatformId = info.getVirtualizationSolution();

			// サブプラットフォームIDのnullチェック
			if(m_subPlatformId == null)
				m_subPlatformId = "";

		} catch (CreateException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			m_log.error(e.getMessage(), e);
		}
	}

	private static ConcurrentHashMap<MonitorProcessMethodMstPK, String> methodCache =
		new ConcurrentHashMap<MonitorProcessMethodMstPK, String> ();

	static {
		try {
			Collection<MonitorProcessMethodMstLocal> c = MonitorProcessMethodMstUtil.getLocalHome().findAll();
			for (MonitorProcessMethodMstLocal local : c) {
				methodCache.put((MonitorProcessMethodMstPK)local.getPrimaryKey(), local.getCollectMethod());
			}
		} catch (Exception e) {
			m_log.error("create MonitorProcessMethodMst cache failed", e);
		}

	}

	/**
	 * 収集方法（SNMP,WBEMなど）の取得
	 * 
	 * @return 収集方法
	 */

	public String getCollectMethod(){

		String collectMethod = "";

		m_log.debug("getColledtMethod() m_platformId : " + m_platformId + ", m_subPlatformId : " + m_subPlatformId);
		// 収集方法を取得する
		MonitorProcessMethodMstPK processPk = new MonitorProcessMethodMstPK(
				m_platformId,
				m_subPlatformId);

		collectMethod = methodCache.get(processPk);
		if (collectMethod == null) {
			m_log.error("getCollectMethod() : collectMethod is null");
		}

		return collectMethod;
	}

	/**
	 * ポーリング対象の取得
	 * 
	 * @return ポーリング対象のリスト
	 */
	public List<String> getPollingTargets(String collectMethod){
		m_log.debug("getPollingTargets() collectMethod = " + collectMethod);

		List<String> pollingTargets = new ArrayList<String>();

		MonitorProcessPollingMstData data = null;
		data = ProcessMasterCache.getMonitorProcessPollingMst(new MonitorProcessPollingMstPK(
				collectMethod, m_platformId, m_subPlatformId,
		"name"));
		if (data != null) {
			pollingTargets.add(data.getPollingTarget());
		}

		data = ProcessMasterCache.getMonitorProcessPollingMst(new MonitorProcessPollingMstPK(
				collectMethod, m_platformId, m_subPlatformId,
				"param"));
		if (data != null) {
			pollingTargets.add(data.getPollingTarget());
		}

		data = ProcessMasterCache.getMonitorProcessPollingMst(new MonitorProcessPollingMstPK(
				collectMethod, m_platformId, m_subPlatformId,
				"path"));
		if (data != null) {
			pollingTargets.add(data.getPollingTarget());
		}
		if (pollingTargets.size() == 0) {
			m_log.error("pollingTargets.size == 0");
		}

		return pollingTargets;
	}


	/**
	 * インスタンス生成時に与えたファシリティIDのプラットフォームIDを取得する
	 * @return プラットフォームID
	 */
	public String getPlatformId() {
		return m_platformId;
	}

	/**
	 * インスタンス生成時に与えたファシリティIDのサブプラットフォームIDを取得する
	 * @return サブプラットフォームID
	 */
	public String getSubPlatformId() {
		return m_subPlatformId;
	}
}
