/*

 Copyright (C) 2012 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.factory;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpOctetString;
import org.opennms.protocols.snmp.SnmpSyntax;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerLocal;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTrapValueInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.snmptrap.bean.MonitorTrapConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapV1;
import com.clustercontrol.snmptrap.bean.TrapCheckInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterPK;
import com.clustercontrol.snmptrap.util.SnmpTrapNotifier;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * SNMPTRAPに対する監視処理実装
 *
 * @since 4.0
 */
public class SnmpTrapFilter {

	private static final Log _log = LogFactory.getLog(SnmpTrapFilter.class);

	private final ExecutorService _executor;

	public static final String _defaultCharsetName = "UTF-8";
	private static final Charset _defaultCharset;

	public static final String _taskQueueSizeKey = "monitor.snmptrap.filter.queue.size";
	public static final int _taskQueueSizeDefault = 3000;
	public static final int _taskQueueSize;

	public static final String _taskThreadSizeKey = "monitor.snmptrap.filter.thread.size";
	public static final int _taskThreadSizeDefault = 8;
	public static final int _taskThreadSize;

	private final SnmpTrapNotifier _notifier;

	static {
		String charStr = "";
		for (String c : Charset.availableCharsets().keySet()) {
			charStr += c + ", ";
		}
		_log.info("supported charset : " + charStr);

		// initialize charset
		Charset charset = Charset.defaultCharset();
		try {
			charset = Charset.forName(_defaultCharsetName);
		} catch (Exception e) { }
		_defaultCharset = charset;

		// initialize queue size
		String taskQueueSizeStr = HinemosProperties.getProperty(_taskQueueSizeKey);
		int taskQueueSize = _taskQueueSizeDefault;
		try {
			if (taskQueueSizeStr != null) {
				taskQueueSize = Integer.parseInt(taskQueueSizeStr);
			}
		} catch (Exception e) { }
		_taskQueueSize = taskQueueSize;

		// initialize thread size
		String taskThreadSizeStr = HinemosProperties.getProperty(_taskThreadSizeKey);
		int taskThreadSize = _taskThreadSizeDefault;
		try {
			if (taskThreadSizeStr != null) {
				taskThreadSize = Integer.parseInt(taskThreadSizeStr);
			}
		} catch (Exception e) { }
		_taskThreadSize = taskThreadSize;

		_log.info("initialized SnmpTrapFilter : defaultCharset = " + _defaultCharset.name() + ", queueSize = " + _taskQueueSize + ", threads = " + _taskThreadSize);
	}

	public SnmpTrapFilter(SnmpTrapNotifier notifier) {
		_executor = new ThreadPoolExecutor(_taskThreadSize, _taskThreadSize,
				0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(_taskQueueSize),
				new ThreadFactory() {
			private volatile int _count = 0;

			@Override
			public Thread newThread(Runnable r) {
				String threadName = "SnmpTrapFilter-" + _count++;
				return new Thread(r, threadName);
			}
		}, new SnmpTrapRejectionHandler());
		this._notifier = notifier;
	}

	private class SnmpTrapRejectionHandler extends ThreadPoolExecutor.DiscardPolicy {

		@Override
		public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
			if (r instanceof SnmpTrapFilterTask) {
				_log.warn("too many snmptrap, filtering task queue is full : " + ((SnmpTrapFilterTask)r).snmptrap);
			}
		}

	}

	public void shutdown() {
		_executor.shutdownNow();
		_notifier.shutdown();
	}

	public synchronized void put(SnmpTrapV1 snmptrap) {

		_executor.execute(new SnmpTrapFilterTask(snmptrap));

	}

	/**
	 * フィルタリング処理を実装したタスククラス
	 */
	private class SnmpTrapFilterTask implements Runnable {

		public final SnmpTrapV1 snmptrap;

		public SnmpTrapFilterTask(SnmpTrapV1 snmptrap) {
			this.snmptrap = snmptrap;
		}

		@Override
		public void run() {

			try {
				MonitorSettingControllerLocal monSettingCtrl = MonitorSettingControllerUtil.getLocalHome().create();
				RepositoryControllerLocal repositoryCtrl = RepositoryControllerUtil.getLocalHome().create();
				CalendarControllerLocal calendarCtrl = CalendarControllerUtil.getLocalHome().create();

				List<MonitorInfo> monitorList = monSettingCtrl.getTrapList();

				if (monitorList == null) {
					if (_log.isDebugEnabled()) {
						_log.debug("snmptrap monitor not found. skip filtering. [" + snmptrap + "]");
					}
					return;
				}

				for (MonitorInfo monitor : monitorList) {
					TrapCheckInfo check = monitor.getTrapCheckInfo();

					// 無効となっている設定はスキップする
					if (monitor.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
						if (_log.isDebugEnabled()) {
							_log.debug("snmptrap monitor " + monitor.getMonitorId()
									+ " is not enabled, skip filtering. [" + snmptrap + "]");
						}
						continue;
					}

					// カレンダの有効期間外の場合、スキップする
					String calendarId = monitor.getCalendarId();
					if (calendarId != null && ! "".equals(calendarId)) {
						try {
							if (! calendarCtrl.isRun(calendarId, snmptrap.receivedTime)) {
								if (_log.isDebugEnabled()) {
									_log.debug("calendar " + calendarId + " is not enabled term. [" + snmptrap + "]");
								}
								continue;
							}
						} catch (CalendarNotFound e) {
							// カレンダが未定義の場合は、スキップせずに継続する（予期せぬロストの回避）
							_log.warn("calendar " + calendarId
									+ " is not found, skip calendar check. [" + snmptrap + "]");
						}
					}

					// コミュニティのチェック
					String community = check.getCommunityName();
					if (check.getCommunityCheck() != MonitorTrapConstant.COMMUNITY_CHECK_OFF) {
						if (! community.equals(snmptrap.community)) {
							if (_log.isDebugEnabled()) {
								_log.debug("community " + community + " is not matched. [" + snmptrap + "]");
							}
							continue;
						}
					}

					// varbindの文字列変換
					List<SnmpSyntax> varbinds = snmptrap.getVarBinds();
					String[] varbind = new String[varbinds.size()];
					String charsetName = check.getCharsetConvert() == MonitorTrapConstant.CHARSET_CONVERT_ON ? check.getCharsetName() : null;
					for (int i = 0; i < varbinds.size(); i++) {
						varbind[i] = getDecodedString(varbinds.get(i), charsetName);
					}
					String varbindStr = null;
					for (String value : varbind) {
						if (varbindStr == null) {
							varbindStr = "varBind = " + value;
						} else {
							varbindStr += ", " + value;
						}
					}

					// snmptrapのagent address(joesnmpではpacketのsrc address)から該当するファシリティ一覧を取得
					List<String> matchedFacilities = null;
					matchedFacilities = repositoryCtrl.getFacilityIdByIpAddress(snmptrap.agentAddr);
					if (matchedFacilities == null || matchedFacilities.size() == 0) {
						matchedFacilities = new ArrayList<String>();
						matchedFacilities.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);
					}
					if (_log.isDebugEnabled()) {
						_log.debug("matched facilities : " + matchedFacilities + " [" + snmptrap + "]");
					}

					// 監視対象のファシリティID一覧を取得する
					List<String> targetFacilities = null;
					if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(monitor.getFacilityId())) {
						targetFacilities = new ArrayList<String>();
						targetFacilities.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);
					} else {
						targetFacilities = repositoryCtrl.getExecTargetFacilityIdList(monitor.getFacilityId());
					}
					if (_log.isDebugEnabled()) {
						_log.debug("target facilities : " + targetFacilities + " [" + snmptrap + "]");
					}

					// 通知対象のファシリティID一覧を絞り込む
					List<String> notifyFacilities = new ArrayList<String>(matchedFacilities);
					notifyFacilities.retainAll(targetFacilities);
					if (notifyFacilities.size() == 0) {
						if (_log.isDebugEnabled()) {
							_log.debug("notification facilities not found [" + snmptrap + "]");
						}
						continue;
					}

					// OIDのチェック
					String facilityPath = "";
					SelectMibMaster mibSelector = new SelectMibMaster();
					if (check.getCheckMode() == MonitorTrapConstant.ALL_OID) {
						// マスタ登録済みの全てのOIDが対象の場合
						SnmpTrapMasterInfo mibMst = mibSelector.findMasterInfo(snmptrap.enterpriseId, snmptrap.genericId, snmptrap.specificId);

						if (mibMst != null && mibMst.getMib() != null) {

							String msg = mibMst.getLogmsg();
							String msgOrig = Messages.getString("communityName") + "=" + snmptrap.community
							+ "  " + mibMst.getDescr() + "\n" + varbindStr;

							msg = getBindedString(msg, varbind);
							msgOrig = getBindedString(msgOrig, varbind);

							String[] args = { snmptrap.enterpriseId, mibMst.getUei() };
							msgOrig = Messages.getString("message.snmptrap.3", args) + "\n" + msgOrig;

							int priority = mibMst.getPriority();

							for (String facilityId : notifyFacilities) {
								if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
									facilityPath = snmptrap.agentAddr.getHostAddress();
								} else {
									facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
								}

								_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
										priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
							}

						} else {
							//一般トラップが1件もマッチしない場合は、再検索
							if (snmptrap.genericId != 6) {
								//SNMPv1からSNMPv2形式の変換において、genericId=6以外(標準トラップ)の場合はOIDはTrapSnmp.GENERIC_TRAPSに従う
								mibMst = mibSelector.findMasterInfo(SnmpTrapReceiver._genericTraps.get(snmptrap.genericId).toString(),
										snmptrap.genericId, snmptrap.specificId);

								if (mibMst != null && mibMst.getMib() != null) {
									String msg = mibMst.getLogmsg();
									String msgOrig = Messages.getString("communityName") + "=" + snmptrap.community
									+ "  " + mibMst.getDescr() + "\n" + varbindStr;

									msg = getBindedString(msg, varbind);
									msgOrig = getBindedString(msgOrig, varbind);

									String[] args = { snmptrap.enterpriseId, mibMst.getUei() };
									msgOrig = Messages.getString("message.snmptrap.3", args) + "\n" + msgOrig;

									int priority = mibMst.getPriority();

									for (String facilityId : notifyFacilities) {
										if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
											facilityPath = snmptrap.agentAddr.getHostAddress();
										} else {
											facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
										}

										_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
												priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
									}
								}
							}
						}


					} else if (check.getCheckMode() == MonitorTrapConstant.UNREGISTERED_OID) {
						// マスタ未登録てのOIDが対象の場合

						if (snmptrap.genericId == 6) {
							SnmpTrapMasterInfo mibMst = mibSelector.findMasterInfo(snmptrap.enterpriseId, snmptrap.genericId, snmptrap.specificId);

							if (mibMst != null && mibMst.getMib() == null) {
								String msg = "communityName  : " + snmptrap.community + ", oid  : " + snmptrap.enterpriseId
								+ ", specificId : " + snmptrap.specificId
								+ ", generic_id : " + snmptrap.genericId;
								String msgOrig = msg + "\n" + varbindStr;

								int priority = PriorityConstant.TYPE_UNKNOWN;

								for (String facilityId : notifyFacilities) {
									if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
										facilityPath = snmptrap.agentAddr.getHostAddress();
									} else {
										facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
									}

									_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
											priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
								}
							}

						}


					} else {
						// ユーザ指定のOIDが対象の場合

						List<MonitorTrapValueInfo> oidList = monitor.getTrapValueInfo();
						int matched = 0;

						for (MonitorTrapValueInfo oid : oidList) {
							if (oid.isValidFlg()
									&& snmptrap.enterpriseId.equals(oid.getTrapOid())
									&& snmptrap.genericId == oid.getGenericId()
									&& snmptrap.specificId == oid.getSpecificId()) {
								String mib = oid.getMib();

								SnmpTrapMasterPK pk = new SnmpTrapMasterPK(
										mib,
										snmptrap.enterpriseId,
										snmptrap.genericId,
										snmptrap.specificId);
								SnmpTrapMasterInfo mibMst = mibSelector.getMasterInfo(pk);

								String msg = oid.getLogmsg();
								String msgOrig = Messages.getString("communityName") + "=" + snmptrap.community
								+ "  " + oid.getDescr() + "\n" + varbindStr;

								msg = getBindedString(msg, varbind);
								msgOrig = getBindedString(msgOrig, varbind);

								String[] args = { snmptrap.enterpriseId, mibMst.getUei() };
								msgOrig = Messages.getString("message.snmptrap.3", args) + "\n" + msgOrig;

								int priority = oid.getPriority();

								for (String facilityId : notifyFacilities) {
									if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
										facilityPath = snmptrap.agentAddr.getHostAddress();
									} else {
										facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
									}

									_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
											priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
								}

								matched++;
								break;
							}
						}

						if (matched == 0 && snmptrap.genericId != 6) {
							for (MonitorTrapValueInfo oid : oidList) {
								if (oid.isValidFlg() && oid.getTrapOid().equals(SnmpTrapReceiver._genericTraps.get(snmptrap.genericId).toString())) {

									String mib = oid.getMib();

									SnmpTrapMasterPK pk = new SnmpTrapMasterPK(
											mib,
											SnmpTrapReceiver._genericTraps.get(snmptrap.genericId).toString(),
											snmptrap.genericId,
											snmptrap.specificId);
									SnmpTrapMasterInfo mibMst = mibSelector.getMasterInfo(pk);

									String msg = oid.getLogmsg();
									String msgOrig = Messages.getString("communityName") + "=" + snmptrap.community
									+ "  " + oid.getDescr() + "\n" + varbindStr;

									msg = getBindedString(msg, varbind);
									msgOrig = getBindedString(msgOrig, varbind);

									String[] args = { snmptrap.enterpriseId, mibMst.getUei() };
									msgOrig = Messages.getString("message.snmptrap.3", args) + "\n" + msgOrig;

									int priority = oid.getPriority();

									for (String facilityId : notifyFacilities) {
										if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
											facilityPath = snmptrap.agentAddr.getHostAddress();
										} else {
											facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
										}

										_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
												priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
									}

								}
							}
						}
					}

				}


			} catch (Exception e) {
				_log.warn("unexpected internal error. [" + snmptrap + "]", e);

				// Internal Event
				AplLogger apllog = new AplLogger("TRAP", "trap");
				String[] args = {snmptrap.enterpriseId, String.valueOf(snmptrap.genericId), String.valueOf(snmptrap.specificId)};
				apllog.put("SYS", "009", args);
			}
		}
	}

	private String getDecodedString(SnmpSyntax value, String charsetName) {
		if (value == null) {
			if (_log.isDebugEnabled()) {
				_log.debug("varbind value is null, use as 0-length string.");
			}
			return "";
		}

		if (! (value instanceof SnmpOctetString)) {
			if (_log.isDebugEnabled()) {
				_log.debug("not octet string, skip decoding : " + value.toString());
			}
			return value.toString();
		}

		String str = "";
		if (charsetName == null) {
			// charsetNameを定義していない場合は、UTF-8でデコードする(文字化けは仕方がない)
			str = new String(((SnmpOctetString)value).getString(), _defaultCharset);
		} else {
			boolean supportedCharset = false;
			Charset charset = null;

			try {
				supportedCharset = Charset.isSupported(charsetName);
				charset = Charset.forName(charsetName);
			} catch (Exception e) {
				_log.warn("");
			}

			if (supportedCharset) {
				str = new String(((SnmpOctetString)value).getString(), charset);
			} else {
				str = new String(((SnmpOctetString)value).getString(), _defaultCharset);
			}
		}

		if (_log.isDebugEnabled()) {
			_log.debug("decoded string : orig = " + ((SnmpOctetString)value).getString() + ", decoded = " + str);
		}

		return str;
	}

	private String getBindedString(String str, String[] varbinds) {
		if (str == null) {
			return "";
		}
		if (varbinds == null) {
			return str;
		}

		for (int i = 0; i < varbinds.length; i++) {
			if (_log.isDebugEnabled()) {
				_log.debug("binding : " + str + "  " + "%parm[#" + (i + 1) + "]% to " + varbinds[i] + "]");
			}
			str = str.replace("%parm[#" + (i + 1) + "]%", varbinds[i]);
			if (_log.isDebugEnabled()) {
				_log.debug("binded : " + str);
			}
		}

		return str;
	}

}
