/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.CalendarDuplicate;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.calendar.bean.CalendarAddInfo;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.ScheduleInfo;
import com.clustercontrol.calendar.bean.WeekdayInfo;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocalHome;

/**
 * カレンダー用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://calendar.ws.clustercontrol.com")
public class CalendarEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( CalendarEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * カレンダ用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private CalendarControllerLocal getCalendarControllerLocal() throws HinemosUnknown, InvalidUserPass {
		CalendarControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			CalendarControllerLocalHome localHome =
				(CalendarControllerLocalHome)
				namingContext.lookup(CalendarControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getCalendarControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getCalendarControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * カレンダ一覧を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @return カレンダ情報のリスト
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CalendarInfo> getCalendarList() throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getUserInfoListAll");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Calendar, Method=getCalendarList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getCalendarControllerLocal().getCalendarList();
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ情報を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @param id
	 * @return カレンダ情報
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public CalendarInfo getCalendar(String id) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getCalendar");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Calendar, Method=getCalendar, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getCalendar(id);
	}

	/**
	 * カレンダ（基本）情報を登録します。<BR>
	 * 
	 * 引数のDTOの内容をマネージャに登録します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param info　登録するカレンダー（基本）情報
	 * @return
	 * @throws HinemosUnknown
	 * @throws CalendarDuplicate
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void addCalendar(CalendarInfo info) throws CalendarDuplicate, HinemosUnknown, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("addCalendar");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Add Calendar, Method=addCalendar, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().addCalendar(info);
	}

	/**
	 * カレンダ（基本）情報を変更します。<BR>
	 * 
	 * 引数のプロパティの内容で更新します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param info
	 * @return
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void modifyCalendar(CalendarInfo info) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("modifyCalendar");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Change Calendar, Method=modifyCalendar, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().modifyCalendar(info);
	}

	/**
	 * カレンダ(基本）情報を 削除します。<BR>
	 * 
	 * 引数のIDに対応するカレンダー（基本）情報を削除します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id カレンダーID
	 * @return
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void deleteCalendar(String id) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteCalendar");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Delete Calendar, Method=deleteCalendar, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getCalendarControllerLocal().deleteCalendar(id);
	}

	/**
	 * カレンダ曜日別情報一覧を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @return カレンダ曜日別情報のリスト
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<WeekdayInfo> getWeekdayList(String id) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getWeekdayList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Rule, Method=getWeekdayList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getWeekdayList(id);
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ曜日別情報を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @return カレンダ曜日別情報
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public WeekdayInfo getWeekday(String id, Integer weekday, Long from, Long to) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getWeekday");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		msg.append(", Weekday=");
		msg.append(weekday);
		msg.append(", From=");
		msg.append(from==null?null:sdf.format(new Date(from)));
		msg.append(", To=");
		msg.append(to==null?null:sdf.format(new Date(to)));
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Rule, Method=getWeekday, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getWeekday(id, weekday, from, to);
	}

	/**
	 * カレンダ曜日別情報を登録します。<BR>
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id
	 * @param info
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarNotFound
	 * @throws CalendarDuplicate
	 */
	public void addWeekday(String id, WeekdayInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarDuplicate, CalendarNotFound,InvalidSetting {
		m_log.debug("addWeekday");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(id);
			msg.append(", Weekday=");
			msg.append(info.getWeekday());
			msg.append(", From=");
			msg.append(info.getTimeFrom()==null?null:sdf.format(new Date(info.getTimeFrom())));
			msg.append(", To=");
			msg.append(info.getTimeTo()==null?null:sdf.format(new Date(info.getTimeTo())));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Add Rule, Method=addWeekday, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().addWeekday(id, info);
	}

	/**
	 * カレンダ曜日別情報を変更します。<BR>
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @param info
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 */
	public void modifyWeekday(String id, Integer weekday, Long from, Long to, WeekdayInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound, CalendarDuplicate,InvalidSetting {
		m_log.debug("modifyWeekday");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(id);
			msg.append(", Weekday=");
			msg.append(weekday);
			msg.append(", From=");
			msg.append(from==null?null:sdf.format(new Date(from)));
			msg.append(", To=");
			msg.append(to==null?null:sdf.format(new Date(to)));
			msg.append(", NewWeekday=");
			msg.append(info.getWeekday());
			msg.append(", NewFrom=");
			msg.append(info.getTimeFrom()==null?null:sdf.format(new Date(info.getTimeFrom())));
			msg.append(", NewTo=");
			msg.append(info.getTimeTo()==null?null:sdf.format(new Date(info.getTimeTo())));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Change Rule, Method=modifyWeekday, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().modifyWeekday(id, weekday, from, to, info);
	}

	/**
	 * カレンダ曜日別情報を削除します。<BR>
	 * 
	 * idで指定されたカレンダー（実行規則）情報を削除します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id カレンダーID
	 * @param weekday 曜日
	 * @param from 時間帯
	 * @param to 時間帯
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarNotFound
	 */
	public void deleteWeekday(String id, Integer weekday, Long from, Long to) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound {
		m_log.debug("deleteWeekday");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		msg.append(", Weekday=");
		msg.append(weekday);
		msg.append(", From=");
		msg.append(from==null?null:sdf.format(new Date(from)));
		msg.append(", To=");
		msg.append(to==null?null:sdf.format(new Date(to)));
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Delete Rule, Method=deleteWeekday, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getCalendarControllerLocal().deleteWeekday(id, weekday, from, to);
	}

	/**
	 * カレンダ追加情報一覧を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @return カレンダ追加情報のリスト
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CalendarAddInfo> getAddList(String id) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getAddList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Exception, Method=getAddList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getAddList(id);
	}

	/**
	 * カレンダ追加情報用プロパティを取得します。<BR>
	 * 
	 * 引数のカレンダーID、日時区間に対応するカレンダー（実行例外）情報を取得します。
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @return カレンダ追加情報
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public CalendarAddInfo getCalendarAdd(String id, Long from, Long to) throws CalendarNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getCalendarAdd");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		msg.append(", From=");
		msg.append(from==null?null:sdf.format(new Date(from)));
		msg.append(", To=");
		msg.append(to==null?null:sdf.format(new Date(to)));
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Exception, Method=getCalendarAdd, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getCalendarAdd(id, from, to);
	}

	/**
	 * カレンダ追加情報を登録します。<BR>
	 * 
	 * idとプロパティの情報を元にカレンダー（実行例外）情報をマネージャに登録します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id
	 * @param info
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 */
	public void addCalendarAdd(String id, CalendarAddInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound, CalendarDuplicate,InvalidSetting {
		m_log.debug("addCalendarAdd");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(id);
			msg.append(", From=");
			msg.append(info.getTimeFrom()==null?null:sdf.format(new Date(info.getTimeFrom())));
			msg.append(", To=");
			msg.append(info.getTimeTo()==null?null:sdf.format(new Date(info.getTimeTo())));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Add Exception, Method=addAddInfo, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().addCalendarAdd(id, info);
	}

	/**
	 * カレンダ追加情報を変更します。<BR>
	 * 
	 * idで指定されたカレンダー（例外実行）情報を変更します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @param info
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 */
	public void modifyCalendarAdd(String id, Long from, Long to, CalendarAddInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound, CalendarDuplicate,InvalidSetting {
		m_log.debug("modifyCalendarAdd");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			StringBuffer msg = new StringBuffer();
			msg.append(", CalendarID=");
			msg.append(id);
			msg.append(", From=");
			msg.append(from==null?null:sdf.format(new Date(from)));
			msg.append(", To=");
			msg.append(to==null?null:sdf.format(new Date(to)));
			msg.append(", NewFrom=");
			msg.append(info.getTimeFrom()==null?null:sdf.format(new Date(info.getTimeFrom())));
			msg.append(", NewTo=");
			msg.append(info.getTimeTo()==null?null:sdf.format(new Date(info.getTimeTo())));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Change Exception, Method=modifyAddInfo, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getCalendarControllerLocal().modifyCalendarAdd(id, from, to, info);
	}

	/**
	 * カレンダ追加情報を削除します。<BR>
	 * 
	 * idで指定されたカレンダー（実行例外）情報を削除します。
	 * 
	 * CalendarWrite権限が必要
	 * 
	 * @param id カレンダーID
	 * @param from 時間帯
	 * @param to 時間帯
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarNotFound
	 */
	public void deleteCalendarAdd(String id, Long from, Long to) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound {
		m_log.debug("deleteCalendarAdd");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", CalendarID=");
		msg.append(id);
		msg.append(", From=");
		msg.append(from==null?null:sdf.format(new Date(from)));
		msg.append(", To=");
		msg.append(to==null?null:sdf.format(new Date(to)));
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Delete Exception, Method=deleteAddInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getCalendarControllerLocal().deleteCalendarAdd(id, from, to);
	}

	/**
	 * 各カレンダの実行予定を取得します。<BR>
	 * 
	 * CalendarRead権限が必要
	 * 
	 * @return カレンダ実行予定のリスト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarNotFound
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<ScheduleInfo> getSchedule(Long base) throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound {
		m_log.debug("getSchedule");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", BaseDate=");
		msg.append(base==null?null:sdf.format(new Date(base)));
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_CALENDAR + " Get Schedule, Method=getSchedule, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getCalendarControllerLocal().getSchedule(base);
	}
}