/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.ui.views;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchPart;

import com.clustercontrol.action.FacilityTree;
import com.clustercontrol.bean.FacilityImageConstant;
import com.clustercontrol.cloud.Activator;
import com.clustercontrol.cloud.presenter.AllPropertyObserver;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.cloud.presenter.CollectionObserver2;
import com.clustercontrol.cloud.presenter.ErrorCodeConstants;
import com.clustercontrol.cloud.presenter.IAccountResource;
import com.clustercontrol.cloud.presenter.IAccountResourceManager;
import com.clustercontrol.cloud.presenter.IElement;
import com.clustercontrol.cloud.presenter.IFacility;
import com.clustercontrol.cloud.presenter.IFacilityRoot;
import com.clustercontrol.cloud.presenter.INode;
import com.clustercontrol.cloud.presenter.IScope;
import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.repository.bean.FacilityConstant;

public class ScopeView extends AbstractCloudViewPart {
	public static final String Id = "com.clustercontrol.cloud.ui.views.ScopeView";
	
	private class FacilityRootUpdateService {
		private FacilityTreeComposite listener;
		
		public FacilityRootUpdateService() {
			listener = new FacilityTreeComposite(composite, SWT.None, null) {
				@Override
				public void update() {
					composite.getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
							ScopeView.this.update();
						}
					});
				}
				public boolean isDisposed () {
					return false;
				}
			};
			listener.dispose();
			new FacilityTree().addComposite(listener);
		}
		
		public void dispose() {
			new FacilityTree().delComposite(listener);
		}
	}

	private CollectionObserver2<IAccountResource> accountResourceObserver = new CollectionObserver2<IAccountResource>() {
		@Override
		public void elementAdded(ElementAddedEvent<IAccountResource> event) {
		}
		@Override
		public void elementRemoved(ElementRemovedEvent<IAccountResource> event) {
			IFacility cloudFacility = null;
			for(IFacility facility: input.getScopes()){
				if(facility.getFacilityId().equals("Cloud")){
					cloudFacility = facility;
					break;
				}
			}
			TreeSelection selection = new TreeSelection(new TreePath(new Object[]{input, cloudFacility}));
			IWorkbenchPart activePart = ScopeView.this.getSite().getPage().getActivePart();
			ScopeView.this.getSite().getPage().activate(ScopeView.this);
			ScopeView.this.getSite().getSelectionProvider().setSelection(selection);
			activePart.getSite().getPage().activate(activePart);
		}
	};

	private AllPropertyObserver observer = new AllPropertyObserver() {
		@Override
		public void elementAdded(ElementAddedEvent event) {
			if (event.getAddedElement() instanceof IElement) {
				((IElement)event.getAddedElement()).addPropertyObserver2(IElement.allProperty, this);
			}
			
			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}

		@Override
		public void elementRemoved(ElementRemovedEvent event) {
			if (event.getRemovedElement() instanceof IElement) {
				((IElement)event.getRemovedElement()).removePropertyObserver2(IElement.allProperty, this);
			}

			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}

		@Override
		public void propertyChanged(ValueChangedEvent event) {
			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}
	};
	

	private FacilityRootUpdateService service;
	
	private class TreeContentProvider implements IStructuredContentProvider, ITreeContentProvider{
		
		public Object[] getChildren(Object parentElement) {
			if(parentElement instanceof IScope){
				IScope scope = (IScope)parentElement;
				List<IFacility> facilities = new ArrayList<IFacility>();
				facilities.addAll(Arrays.asList(scope.getScopes()));
				facilities.addAll(Arrays.asList(scope.getNodes()));
				
				return facilities.toArray();
			}
			return null;
		}

		public Object getParent(Object element) {
			if(element instanceof IFacility){
				IFacility facility = (IFacility)element;
				return facility.getParent() != null ? facility.getParent(): facility.getFacilityRoot();
			}
			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof IScope){
				IScope scope = (IScope)element;
				return scope.getScopes().length != 0 || scope.getNodes().length != 0;
			}
			return false;	
		}

		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof IFacilityRoot) {
				return ((IFacilityRoot)inputElement).getScopes();
			}
			return null;
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	private TreeViewer treeViewer;
	private IFacilityRoot input;
	private Composite composite;
	
	public ScopeView() {
	}

	@Override
	protected void internalCreatePartControl(Composite parent) {
		composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new FillLayout(SWT.HORIZONTAL));

		Composite composite_1 = new Composite(composite, SWT.NONE);
		TreeColumnLayout tcl_composite = new TreeColumnLayout();
		composite_1.setLayout(tcl_composite);

		treeViewer = new TreeViewer(composite_1, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);

		treeViewer.setContentProvider(new TreeContentProvider());
		treeViewer.setLabelProvider(new LabelProvider(){
			@Override
			public Image getImage(Object element) {
				if (element instanceof IScope) {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_SCOPE);
				}
				else if (element instanceof INode) {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);					
				}
				else {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);					
				}
			}

			@Override
			public String getText(Object element) {
				if (element instanceof IFacility){
					IFacility facility = (IFacility)element;
					return facility.getName() + "(" + facility.getFacilityId() + ")";
				}
				return null;
			}
		});
		
		Activator.getDefault().getCloudResourceManager().getAccountResourceManager().addPropertyObserver2(IAccountResourceManager.p2.accountResources, accountResourceObserver);
		
		this.getSite().setSelectionProvider(treeViewer);

		treeViewer.setContentProvider(new TreeContentProvider());

		update();

		service = new FacilityRootUpdateService();
	}

	private void setUp(IFacilityRoot faciiltyRoot) {
		faciiltyRoot.addPropertyObserver2(IElement.allProperty, observer);
		for (IScope scope: faciiltyRoot.getScopes()) {
			recursiveSetUp(scope);
		}
	}
	
	private void recursiveSetUp(IScope scope) {
		scope.addPropertyObserver2(IElement.allProperty, observer);
		for (IScope child: scope.getScopes()) {
			recursiveSetUp(child);
		}
		for (INode node: scope.getNodes()) {
			node.addPropertyObserver2(IElement.allProperty, observer);
		}
	}
	
	private void setDown(IFacilityRoot faciiltyRoot) {
		faciiltyRoot.removePropertyObserver2(IElement.allProperty, observer);
		for (IScope scope: faciiltyRoot.getScopes()) {
			recursiveSetDown(scope);
		}
	}
	
	private void recursiveSetDown(IScope scope) {
		scope.removePropertyObserver2(IElement.allProperty, observer);
		for (IScope child: scope.getScopes()) {
			recursiveSetDown(child);
		}
		for (INode node: scope.getNodes()) {
			node.removePropertyObserver2(IElement.allProperty, observer);
		}
	}

	@Override
	public void dispose() {
		if (service != null) {
			service.dispose();
		}
		if (input != null) {
			setDown(input);
		}
		
		// ログオフ時の動作としてモデルのリリースをこのビューで行う。
		Activator.getDefault().releaseCloudManager();
		
		super.dispose();
	}

	public void update() {
		if (input == null) {
			try {
				input = Activator.getDefault().getCloudResourceManager().getHinemosService().getFacilityRoot();
				if (input != null) {
					input.update();

					setUp(input);
					treeViewer.setInput(input);
					treeViewer.expandToLevel(4);
				}
			}
			catch (CloudModelException e) {
				Logger logger = Logger.getLogger(ScopeView.class);
				logger.error(e.getMessage(), e);
			}
		}
		else {
			try {
				input.update();
			}
			catch (CloudModelException e) {
				if (ErrorCodeConstants.FACILITYROOT_INVALID_ROOT_SCOPE_NOT_FOUND.equals(e.getErrorCode())) {
					//　ビュー表示内容の消去。
					setDown(input);
					input = null;
					treeViewer.setInput(null);
				}
				else {
					Logger logger = Logger.getLogger(ScopeView.class);
					logger.warn(e.getMessage(), e);
				}
			}
		}
	}

	public Logger getLogger() {
		return Logger.getLogger(ScopeView.class);
	}

	@Override
	protected StructuredViewer getViewer() {
		return treeViewer;
	}
}
