package com.clustercontrol.cloud.aws.factory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.AttachVolumeRequest;
import com.amazonaws.services.ec2.model.AttachVolumeResult;
import com.amazonaws.services.ec2.model.AvailabilityZone;
import com.amazonaws.services.ec2.model.CreateImageRequest;
import com.amazonaws.services.ec2.model.CreateImageResult;
import com.amazonaws.services.ec2.model.CreateSnapshotRequest;
import com.amazonaws.services.ec2.model.CreateSnapshotResult;
import com.amazonaws.services.ec2.model.CreateVolumeRequest;
import com.amazonaws.services.ec2.model.CreateVolumeResult;
import com.amazonaws.services.ec2.model.DeleteSnapshotRequest;
import com.amazonaws.services.ec2.model.DeleteVolumeRequest;
import com.amazonaws.services.ec2.model.DeregisterImageRequest;
import com.amazonaws.services.ec2.model.DescribeAvailabilityZonesResult;
import com.amazonaws.services.ec2.model.DescribeInstanceAttributeRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeInstancesResult;
import com.amazonaws.services.ec2.model.DescribeVolumesRequest;
import com.amazonaws.services.ec2.model.DescribeVolumesResult;
import com.amazonaws.services.ec2.model.DetachVolumeRequest;
import com.amazonaws.services.ec2.model.DetachVolumeResult;
import com.amazonaws.services.ec2.model.GroupIdentifier;
import com.amazonaws.services.ec2.model.InstanceAttributeName;
import com.amazonaws.services.ec2.model.InstanceBlockDeviceMapping;
import com.amazonaws.services.ec2.model.InstanceStateName;
import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.Placement;
import com.amazonaws.services.ec2.model.Reservation;
import com.amazonaws.services.ec2.model.RunInstancesRequest;
import com.amazonaws.services.ec2.model.RunInstancesResult;
import com.amazonaws.services.ec2.model.StartInstancesRequest;
import com.amazonaws.services.ec2.model.StartInstancesResult;
import com.amazonaws.services.ec2.model.StopInstancesRequest;
import com.amazonaws.services.ec2.model.StopInstancesResult;
import com.amazonaws.services.ec2.model.TerminateInstancesRequest;
import com.amazonaws.services.ec2.model.Volume;
import com.amazonaws.services.ec2.model.VolumeType;
import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.elasticloadbalancing.model.DescribeLoadBalancersRequest;
import com.amazonaws.services.elasticloadbalancing.model.DescribeLoadBalancersResult;
import com.amazonaws.services.elasticloadbalancing.model.LoadBalancerDescription;
import com.amazonaws.services.elasticloadbalancing.model.RegisterInstancesWithLoadBalancerRequest;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.Storage.StorageAttachment;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.aws.AWSOptionPropertyConstants;
import com.clustercontrol.cloud.aws.util.AWSConstants;
import com.clustercontrol.cloud.aws.util.AWSErrorCode;
import com.clustercontrol.cloud.aws.util.AWSUtil;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.bean.PlatformKind;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StorageAttachmentStateKind;
import com.clustercontrol.cloud.bean.StorageStateKind;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.bean.Zone;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.cloud.persistence.Transactional;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;

@Transactional(Transactional.TransactionType.Supported)
public class AWSResourceManagement implements IResourceManagement, AWSConstants {
	public static String RT_Instance = "EC2";
	public static String RT_Image = "AMI";
	public static String RT_Storage = "EBSVolume";
	public static String RT_Snapshot = "EBSSnapshot";
	public static String RT_LoadBalancer = "ELB";
	public static String RT_InstanceBackup = "InstanceBackup";
	public static String RT_StorageBackup = "StorageBackup";

	public static class EbsBlockDevice {
		public Integer volumeSize;
		public Boolean deleteOnTermination;
		public String volumeType;
		public Integer iops;
	}

	public static class BackupedInstanceDetail {
		public String instanceId;
		public String name;
		public String flavor;
		public String zone;
		public List<Tag> tags;
		public Date createTime;
		public InstanceDetail instanceDetail;
	}

	public static class InstanceDetail {
		public String subnetId;
		public List<String> securityGroupIds = new ArrayList<>();
		public String keyName;
		public Boolean monitoring;
		public Boolean disableApiTermination;
		public String instanceInitiatedShutdownBehavior;
		public Boolean ebsOptimized;
		public EbsBlockDevice rootBlockDevice;
	}

	public static class BackupedStorageDetail {
		public String storageId;
		public String name;
		public String flavor;
		public String zone;
		public Integer size;
		public Date createTime;
		public StorageDetail storageDetail;
	}

	public static class StorageDetail {
		public Integer iops;
	}

	private IRegion region;
	private ICredential credential;
	private IStore store;

	@Override
	public void setAccessDestination(ICredential credential, IRegion region) {
		this.region = region;
		this.credential = credential;
	}

	@Override
	public void setStore(IStore store) {
		this.store = store;
	}

	private InstanceStateKind instanceStateKind(String state) {
		for (InstanceStateKind stateKind: InstanceStateKind.values()) {
			if (stateKind.label().equals(state)) {
				return stateKind;
			}
		}
		throw new InternalManagerError();
	}

	@Override
	public ICredential getCledential() {
		return credential;
	}

	@Override
	public IRegion getRegion() {
		return region;
	}

	@Override
	public void disconnect() {
	}

	@Override
	public Instance createInstance(String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
		com.amazonaws.services.ec2.model.Instance awsInstance = null;
		boolean success = false;
		try {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(InstanceDetail.class);
			InstanceDetail detail = or.readValue(instanceDetail);

			// 後続の処理でエラーが発生すると、EC2 インスタンスの削除を行いますが、
			// AWS のマネージメントコンソールでは、削除状態でしばらく残ります。
			RunInstancesRequest awsRequest = new RunInstancesRequest()
			.withInstanceType(flavor)
			.withEbsOptimized(detail.ebsOptimized)
			.withSecurityGroupIds(detail.securityGroupIds)
			.withInstanceInitiatedShutdownBehavior(detail.instanceInitiatedShutdownBehavior)
			.withImageId(imageId)
			.withMonitoring(detail.monitoring)
			.withKeyName(detail.keyName)
			.withPlacement(new Placement().withAvailabilityZone(zone))
			.withDisableApiTermination(detail.disableApiTermination)
			.withSubnetId(detail.subnetId)
			.withMinCount(1)
			.withMaxCount(1);

			// ルートノードの関連付け。
			if (detail.rootBlockDevice != null) {
				awsRequest.withBlockDeviceMappings(
						new com.amazonaws.services.ec2.model.BlockDeviceMapping().withEbs(
								new com.amazonaws.services.ec2.model.EbsBlockDevice()
								.withDeleteOnTermination(detail.rootBlockDevice.deleteOnTermination)
								.withIops(detail.rootBlockDevice.iops)
								.withVolumeSize(detail.rootBlockDevice.volumeSize)
								.withVolumeType(detail.rootBlockDevice.volumeType)
								)
								.withDeviceName("/dev/sda1")
						);
			}
			else {
				awsRequest.withBlockDeviceMappings(
						new com.amazonaws.services.ec2.model.BlockDeviceMapping().withEbs(
								new com.amazonaws.services.ec2.model.EbsBlockDevice()
								.withDeleteOnTermination(true)
								)
								.withDeviceName("/dev/sda1")
						);
			}

			RunInstancesResult result = ec2.runInstances(awsRequest);

			Reservation reservation = result.getReservation();
			assert reservation.getInstances().size() == 1;

			awsInstance = reservation.getInstances().get(0);

			// AWS インスタンスの名前を追加。
			List<com.amazonaws.services.ec2.model.Tag> awsTags = new ArrayList<>();
			tags.add(new Tag(KEY_NAME, name));
			for (int i =0; i < Math.min(tags.size(), 9); ++i) {
				awsTags.add(new com.amazonaws.services.ec2.model.Tag(tags.get(i).getKey(), tags.get(i).getValue()));
			}
			AWSUtil.addTags(ec2, awsInstance.getInstanceId(), awsTags);

			final Instance instance = new Instance();
			instance.setResourceType(RT_Instance);
			instance.setInstanceId(awsInstance.getInstanceId());
			instance.setName(name);
			instance.setFlavor(flavor);
			instance.setZone(awsInstance.getPlacement().getAvailabilityZone());
			instance.setImageId(imageId);
			instance.setPlatform(awsInstance.getPlatform() == null ? PlatformKind.linux: PlatformKind.windows);
			instance.setState(instanceStateKind(awsInstance.getState().getName()));

			if (AWSOptionPropertyConstants.aws_node_ip.match(AWSOptionPropertyConstants.qublic)) {
				instance.setHostName(awsInstance.getPublicDnsName());
				instance.setIpAddress(awsInstance.getPublicIpAddress());
			}
			else {
				instance.setHostName(awsInstance.getPrivateDnsName());
				instance.setIpAddress(awsInstance.getPrivateIpAddress());
			}

			List<Instance.BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
			for (com.amazonaws.services.ec2.model.InstanceBlockDeviceMapping awsBlock: awsInstance.getBlockDeviceMappings()) {
				if (awsBlock.getEbs() != null) {
					blockDeviceMappings.add(new Instance.BlockDeviceMapping(awsBlock.getEbs().getVolumeId(), awsBlock.getDeviceName(), awsBlock.getEbs().getStatus()));
				}
			}
			instance.setBlockDeviceMappings(blockDeviceMappings);

			List<Tag> cloudTags = new ArrayList<>();
			for (com.amazonaws.services.ec2.model.Tag tag: awsTags) {
				cloudTags.add(new Tag(tag.getKey(), tag.getValue()));
			}
			instance.setTags(cloudTags);

			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								ec2.terminateInstances(new TerminateInstancesRequest().withInstanceIds(instance.getInstanceId()));
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});

			success = true;

			return instance;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		finally {
			// 失敗したか確認。
			if (!success) {
				// EC2 インスタンス削除。
				if (awsInstance != null) {
					try {
						Thread.sleep(1000);
						ec2.terminateInstances(new TerminateInstancesRequest().withInstanceIds(awsInstance.getInstanceId()));
					}
					catch (InterruptedException e) {
					}
					catch (Exception e) {
						Logger logger = Logger.getLogger(this.getClass());
						logger.warn(e.getMessage(), e);
					}
				}
			}
		}
	}

	@Override
	public void deleteInstance(String instanceId) throws CloudManagerFault {
		try {
			// 関連している EC2 インスタンスが存在するので削除。
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			TerminateInstancesRequest terminateInstancesRequest = new TerminateInstancesRequest().withInstanceIds(instanceId);
			ec2.terminateInstances(terminateInstancesRequest);
		}
		catch (AmazonServiceException e) {
			// 既にインスタンスが存在しない場合は、無視。
			if (
					!AWSErrorCode.InvalidInstanceID_NotFound.label().equals(e.getErrorCode()) &&
					!AWSErrorCode.OperationNotPermitted.label().equals(e.getErrorCode())
					) {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
			else {
				Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
			}
		}
	}

	private Instance convertInstance(com.amazonaws.services.ec2.model.Instance awsInstance) {
		Instance instance = new Instance();
		instance.setResourceType(RT_Instance);
		instance.setInstanceId(awsInstance.getInstanceId());

		for (com.amazonaws.services.ec2.model.Tag t: awsInstance.getTags()) {
			if (KEY_NAME.equals(t.getKey())) {
				instance.setName(t.getValue());
				break;
			}
		}

		instance.setFlavor(awsInstance.getInstanceType());
		instance.setZone(awsInstance.getPlacement().getAvailabilityZone());
		instance.setImageId(awsInstance.getImageId());
		instance.setPlatform(awsInstance.getPlatform() == null ? PlatformKind.linux: PlatformKind.windows);
		instance.setState(instanceStateKind(awsInstance.getState().getName()));

		if (AWSOptionPropertyConstants.aws_node_ip.match(AWSOptionPropertyConstants.qublic)) {
			instance.setHostName(awsInstance.getPublicDnsName());
			instance.setIpAddress(awsInstance.getPublicIpAddress());
		}
		else {
			instance.setHostName(awsInstance.getPrivateDnsName());
			instance.setIpAddress(awsInstance.getPrivateIpAddress());
		}

		List<Instance.BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
		for (com.amazonaws.services.ec2.model.InstanceBlockDeviceMapping awsBlock: awsInstance.getBlockDeviceMappings()) {
			if (awsBlock.getEbs() != null) {
				blockDeviceMappings.add(new Instance.BlockDeviceMapping(awsBlock.getEbs().getVolumeId(), awsBlock.getDeviceName(), awsBlock.getEbs().getStatus()));
			}
		}
		instance.setBlockDeviceMappings(blockDeviceMappings);

		List<Tag> cloudTags = new ArrayList<>();
		for (com.amazonaws.services.ec2.model.Tag tag: awsInstance.getTags()) {
			cloudTags.add(new Tag(tag.getKey(), tag.getValue()));
		}
		instance.setTags(cloudTags);

		return instance;
	}

	@Override
	public Instance getInstance(String instanceId) throws CloudManagerFault {
		// AWS のボリューム情報を取得。
		com.amazonaws.services.ec2.model.Instance awsInstance;
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			com.amazonaws.services.ec2.model.DescribeInstancesRequest request = new com.amazonaws.services.ec2.model.DescribeInstancesRequest().withInstanceIds(instanceId);
			awsInstance = ec2.describeInstances(request).getReservations().get(0).getInstances().get(0);
		}
		catch (AmazonServiceException e) {
			// AWS ボリュームが存在しない場合は、処理を継続。
			if (!AWSErrorCode.InvalidVolume_NotFound.match(e.getErrorCode())) {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}

			throw ErrorCode.Resource_InvalidInstanceID_NotFound.cloudManagerFault(instanceId);
		}
		return convertInstance(awsInstance);
	}

	@Override
	public List<Instance> getInstances(String...instanceIds) throws CloudManagerFault {
		return getInstances(Arrays.asList(instanceIds));
	}

	@Override
	public List<Instance> getInstances(List<String> instanceIds) throws CloudManagerFault {
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			DescribeInstancesResult result = ec2.describeInstances(new DescribeInstancesRequest().withInstanceIds(instanceIds));

			List<Instance> instances = new ArrayList<>();
			for (Reservation r: result.getReservations()) {
				for (com.amazonaws.services.ec2.model.Instance awsInstance: r.getInstances()) {
					if (!InstanceStateName.ShuttingDown.toString().equals(awsInstance.getState()) || !InstanceStateName.Terminated.toString().equals(awsInstance.getState().getName())) {
						instances.add(convertInstance(awsInstance));
					}
				}
			}

			return instances;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public InstanceStateChange startInstance(String instanceId) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		StartInstancesRequest request = new StartInstancesRequest().withInstanceIds(instanceId);
		StartInstancesResult result = ec2.startInstances(request);

		try {
			com.amazonaws.services.ec2.model.InstanceStateChange awsChange = result.getStartingInstances().get(0);

			InstanceStateChange change = new InstanceStateChange();
			change.setInstanceId(instanceId);
			change.setCurrentState(InstanceStateKind.byLabel(awsChange.getCurrentState().getName()));
			change.setPreviousState(InstanceStateKind.byLabel(awsChange.getPreviousState().getName()));

			return change;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public InstanceStateChange stopInstance(String instanceId) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		StopInstancesRequest request = new StopInstancesRequest().withInstanceIds(instanceId);
		StopInstancesResult result = ec2.stopInstances(request);

		try {
			com.amazonaws.services.ec2.model.InstanceStateChange awsChange = result.getStoppingInstances().get(0);

			InstanceStateChange change = new InstanceStateChange();
			change.setInstanceId(instanceId);
			change.setCurrentState(InstanceStateKind.byLabel(awsChange.getCurrentState().getName()));
			change.setPreviousState(InstanceStateKind.byLabel(awsChange.getPreviousState().getName()));

			return change;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public List<String> getInstanceFlavors() throws CloudManagerFault {
		List<String> instanceTypes = new ArrayList<>();
		for (InstanceType type: InstanceType.values()) {
			instanceTypes.add(type.toString());
		}
		return instanceTypes;
	}

	@Override
	public List<Zone> getZones() throws CloudManagerFault {
		try {
			List<Zone> zones = new ArrayList<>();
			DescribeAvailabilityZonesResult result = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation()).describeAvailabilityZones();
			for (AvailabilityZone awsZone: result.getAvailabilityZones()) {
				Zone zone = new Zone();
				zone.setName(awsZone.getZoneName());
				zones.add(zone);
			}
			return zones;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public void attachStorage(final String instanceId, final String storageId, String deviceName) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		boolean success = false;
		AttachVolumeResult result = null;
		try {
			// AWS としてアタッチ。
			AttachVolumeRequest attachVolumeRequest = new AttachVolumeRequest()
			.withInstanceId(instanceId)
			.withVolumeId(storageId)
			.withDevice(deviceName);
			result = ec2.attachVolume(attachVolumeRequest);

			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								ec2.detachVolume(new DetachVolumeRequest().withInstanceId(instanceId).withVolumeId(storageId));
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});

			success = true;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		finally {
			// 失敗したか確認。
			if (!success) {
				// EC2 インスタンス削除。
				if (result != null) {
					try {
						DetachVolumeRequest detachVolumeRequest = new DetachVolumeRequest()
						.withInstanceId(instanceId)
						.withVolumeId(storageId);
						ec2.detachVolume(detachVolumeRequest);
					}
					catch (Exception e) {
						Logger logger = Logger.getLogger(this.getClass());
						logger.warn(e.getMessage(), e);
					}
				}
			}
		}
	}

	@Override
	public void detachStorage(final String instanceId, final String storageId) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		boolean success = false;
		DetachVolumeResult result = null;
		try {
			// AWS としてアタッチ。
			DetachVolumeRequest detachVolumeRequest = new DetachVolumeRequest()
			.withInstanceId(instanceId)
			.withVolumeId(storageId);
			result = ec2.detachVolume(detachVolumeRequest);

			final String deviceName = result.getAttachment().getDevice();
			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								ec2.attachVolume(new AttachVolumeRequest().withInstanceId(instanceId).withVolumeId(storageId).withDevice(deviceName));
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});

			success = true;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		finally {
			// 失敗したか確認。
			if (!success) {
				// EC2 インスタンス削除。
				if (result != null) {
					try {
						AttachVolumeRequest attachVolumeRequest = new AttachVolumeRequest()
						.withInstanceId(instanceId)
						.withVolumeId(storageId)
						.withDevice(result.getAttachment().getDevice());
						ec2.attachVolume(attachVolumeRequest);
					}
					catch (Exception e) {
						Logger logger = Logger.getLogger(this.getClass());
						logger.warn(e.getMessage(), e);
					}
				}
			}
		}
	}

	@Override
	public Storage createStorage(String name, String flavor, int size, String snapshotId, String zone, String storageDetail) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		Volume volume = null;
		boolean success = false;
		try {
			StorageDetail detail = null;
			if (storageDetail != null) {
				try {
					ObjectMapper om = new ObjectMapper();
					ObjectReader or = om.reader(StorageDetail.class);
					detail = or.readValue(storageDetail);
				}
				catch (Exception e) {
					throw new InternalManagerError(e);
				}
			}

			CreateVolumeRequest awsRequest = new CreateVolumeRequest()
			.withAvailabilityZone(zone)
			.withIops(detail != null ? detail.iops: null)
			.withSize(size)
			.withSnapshotId(snapshotId)
			.withVolumeType(flavor);

			CreateVolumeResult result = ec2.createVolume(awsRequest);

			volume = result.getVolume();
			AWSUtil.addTag(ec2, volume.getVolumeId(), KEY_NAME, name);

			final Storage storage = new Storage();
			storage.setResourceType(RT_Storage);
			storage.setStorageId(result.getVolume().getVolumeId());
			storage.setName(name);
			storage.setSnapshotId(result.getVolume().getSnapshotId());
			storage.setSize(result.getVolume().getSize());
			storage.setZone(result.getVolume().getAvailabilityZone());
			storage.setFlavor(result.getVolume().getVolumeType());
			storage.setState(StorageStateKind.byLabel(volume.getState()));

			for (com.amazonaws.services.ec2.model.VolumeAttachment va: volume.getAttachments()) {
				StorageAttachment sa = new StorageAttachment();
				sa.setInstanceId(va.getInstanceId());
				sa.setDevice(va.getDevice());
				sa.setState(StorageAttachmentStateKind.byLabel(va.getState()));
				sa.setAttachTime(va.getAttachTime());
				storage.setStorageAttachment(sa);
				break;
			}

			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								ec2.deleteVolume(new DeleteVolumeRequest().withVolumeId(storage.getStorageId()));
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});

			success = true;

			return storage;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		finally {
			// 失敗したか確認。
			if (!success) {
				// EC2 インスタンス削除。
				if (volume != null) {
					try {
						Thread.sleep(1000);
						ec2.deleteVolume(new DeleteVolumeRequest().withVolumeId(volume.getVolumeId()));
					}
					catch (InterruptedException e) {
					}
					catch (Exception e) {
						Logger logger = Logger.getLogger(this.getClass());
						logger.warn(e.getMessage(), e);
					}
				}
			}
		}
	}

	@Override
	public void deleteStorage(String storageId) throws CloudManagerFault {
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			DeleteVolumeRequest awsRequest = new DeleteVolumeRequest().withVolumeId(storageId);
			ec2.deleteVolume(awsRequest);
		}
		catch (AmazonServiceException e) {
			if (
					AWSErrorCode.InvalidVolume_NotFound.match(e.getErrorCode()) ||
					AWSErrorCode.OperationNotPermitted.match(e.getErrorCode())
					) {
				throw ErrorCode.Resource_InvalidStorage_NotFound.cloudManagerFault(storageId);
			}
			else {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
	}

	@Override
	public Storage getStorage(String storageId) throws CloudManagerFault {
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			DescribeVolumesRequest request = new DescribeVolumesRequest().withVolumeIds(storageId);
			DescribeVolumesResult result = ec2.describeVolumes(request);
			Volume volume = result.getVolumes().get(0);

			Storage storage = new Storage();
			storage.setResourceType(RT_Storage);
			storage.setStorageId(volume.getVolumeId());
			for (com.amazonaws.services.ec2.model.Tag t: volume.getTags()) {
				if (KEY_NAME.equals(t.getKey())) {
					storage.setName(t.getValue());
					break;
				}
			}
			storage.setSnapshotId(volume.getSnapshotId());
			storage.setSize(volume.getSize());
			storage.setZone(volume.getAvailabilityZone());
			storage.setFlavor(volume.getVolumeType());
			storage.setState(StorageStateKind.byLabel(volume.getState()));

			for (com.amazonaws.services.ec2.model.VolumeAttachment va: volume.getAttachments()) {
				StorageAttachment sa = new StorageAttachment();
				sa.setInstanceId(va.getInstanceId());
				sa.setDevice(va.getDevice());
				sa.setState(StorageAttachmentStateKind.byLabel(va.getState()));
				sa.setAttachTime(va.getAttachTime());
				storage.setStorageAttachment(sa);
				break;
			}

			storage.setCreateTime(volume.getCreateTime());

			return storage;
		}
		catch (AmazonServiceException e) {
			// AWS ボリュームが存在しない場合は、処理を継続。
			if (!AWSErrorCode.InvalidVolume_NotFound.match(e.getErrorCode())) {
				throw ErrorCode.Resource_InvalidStorage_NotFound.cloudManagerFault(storageId);
			}
			else {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
	}

	@Override
	public List<Storage> getStorages(String... storageIds) throws CloudManagerFault {
		return getStorages(Arrays.asList(storageIds));
	}

	@Override
	public List<Storage> getStorages(List<String> storageIds) throws CloudManagerFault {
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			DescribeVolumesResult result = ec2.describeVolumes(new DescribeVolumesRequest().withVolumeIds(storageIds));

			List<Storage> storages = new ArrayList<>();
			for (Volume volume: result.getVolumes()) {
				Storage storage = new Storage();
				storage.setStorageId(volume.getVolumeId());

				String name = null;
				for (com.amazonaws.services.ec2.model.Tag t: volume.getTags()) {
					if (KEY_NAME.equals(t.getKey())) {
						name = t.getValue();
						break;
					}
				}
				storage.setName(name);

				storage.setSnapshotId(volume.getSnapshotId());
				storage.setSize(volume.getSize());
				storage.setZone(volume.getAvailabilityZone());
				storage.setFlavor(volume.getVolumeType());
				storage.setState(StorageStateKind.byLabel(volume.getState()));

				for (com.amazonaws.services.ec2.model.VolumeAttachment va: volume.getAttachments()) {
					StorageAttachment sa = new StorageAttachment();
					sa.setInstanceId(va.getInstanceId());
					sa.setDevice(va.getDevice());
					sa.setState(StorageAttachmentStateKind.byLabel(va.getState()));
					sa.setAttachTime(va.getAttachTime());
					storage.setStorageAttachment(sa);
					break;
				}
				storage.setCreateTime(volume.getCreateTime());
				storages.add(storage);
			}

			return storages;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	public List<String> getStorageFlavors() throws CloudManagerFault {
		List<String> storageTypes = new ArrayList<>();
		for (VolumeType type: VolumeType.values()) {
			storageTypes.add(type.toString());
		}
		return storageTypes;
	}

	@Override
	public StorageBackup createStorageBackup(String storageId, String name, String description, String backupOption) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
		String snapshotId = null;
		//		boolean success = false;
		try {
			CreateSnapshotRequest createSnapshotRequest = new CreateSnapshotRequest().withVolumeId(storageId).withDescription(description);
			final CreateSnapshotResult createSnapshotResult = ec2.createSnapshot(createSnapshotRequest);
			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								ec2.deleteSnapshot(new DeleteSnapshotRequest().withSnapshotId(createSnapshotResult.getSnapshot().getSnapshotId()));
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}						}
					}
					);

			snapshotId = createSnapshotResult.getSnapshot().getSnapshotId();
			AWSUtil.addTags(ec2, createSnapshotResult.getSnapshot().getSnapshotId(), new com.amazonaws.services.ec2.model.Tag(KEY_NAME, name), new com.amazonaws.services.ec2.model.Tag("SouceStorageId", storageId));

			com.amazonaws.services.ec2.model.Volume volume = ec2.describeVolumes(new com.amazonaws.services.ec2.model.DescribeVolumesRequest().withVolumeIds(storageId)).getVolumes().get(0);

			BackupedStorageDetail backupedData = new BackupedStorageDetail();
			backupedData.storageId = storageId;
			for (com.amazonaws.services.ec2.model.Tag t: volume.getTags()) {
				if (KEY_NAME.equals(t.getKey())) {
					backupedData.name = t.getValue();
					break;
				}
			}
			backupedData.flavor = volume.getVolumeType();
			backupedData.zone = volume.getAvailabilityZone();
			backupedData.size = volume.getSize();
			backupedData.createTime = volume.getCreateTime();
			backupedData.storageDetail = new StorageDetail();
			backupedData.storageDetail.iops = volume.getIops();

			ObjectMapper om = new ObjectMapper();
			ObjectWriter ow = om.writerWithType(BackupedStorageDetail.class);
			String backupedDetail = ow.writeValueAsString(backupedData);
			store.put(RT_StorageBackup, snapshotId, backupedDetail);

			final StorageBackup storageBackup = new StorageBackup();
			storageBackup.setResourceType(RT_StorageBackup);
			storageBackup.setStorageBackupId(snapshotId);
			storageBackup.setName(name);
			storageBackup.setDescription(description);
			storageBackup.setStorageId(volume.getVolumeId());
			storageBackup.setCreateTime(backupedData.createTime);
			storageBackup.setBackupedData(new StorageBackup.BackupedData());
			storageBackup.getBackupedData().setName(backupedData.name);
			storageBackup.getBackupedData().setFlavor(backupedData.flavor);
			storageBackup.getBackupedData().setZone(backupedData.zone);
			storageBackup.getBackupedData().setSize(backupedData.size);
			storageBackup.getBackupedData().setStorageDetail(om.writerWithType(StorageDetail.class).writeValueAsString(backupedData.storageDetail));

			//			success = true;

			return storageBackup;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		finally {
			//			if (!success) {
			//				if (snapshotId != null) {
			//					ec2.deleteSnapshot(new DeleteSnapshotRequest().withSnapshotId(snapshotId));
			//				}
			//			}
		}
	}

	@Override
	public void deleteStorageBackup(String storageBackupId) throws CloudManagerFault {
		try {
			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			DeleteSnapshotRequest awsRequest = new DeleteSnapshotRequest().withSnapshotId(storageBackupId);
			ec2.deleteSnapshot(awsRequest);
		}
		catch (AmazonServiceException e) {
			if (
					AWSErrorCode.InvalidSnapshot_NotFound.match(e.getErrorCode()) ||
					AWSErrorCode.OperationNotPermitted.match(e.getErrorCode())
					) {
				throw ErrorCode.Resource_InvalidSnapshot_NotFound.cloudManagerFault(storageBackupId);
			}
			else {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}


		try {
			store.remove(RT_StorageBackup, storageBackupId);
		}
		catch (Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	public List<Snapshot> getSnapshots(String...snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshots(List<String> snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(Filter...filters) throws CloudManagerFault {
		return getSnapshotsWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(List<Filter> filters) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		List<com.amazonaws.services.ec2.model.Filter> awsFilters = new ArrayList<>();
		for (Filter filter: filters) {
			awsFilters.add(new com.amazonaws.services.ec2.model.Filter().withName(filter.getName()).withValues(filter.getValues()));
		}

		com.amazonaws.services.ec2.model.DescribeSnapshotsRequest request = new com.amazonaws.services.ec2.model.DescribeSnapshotsRequest().withFilters(awsFilters);
		com.amazonaws.services.ec2.model.DescribeSnapshotsResult result = ec2.describeSnapshots(request);

		List<Snapshot> snapshots = new ArrayList<>();
		for(com.amazonaws.services.ec2.model.Snapshot awsSnapshot: result.getSnapshots()){
			Snapshot snapshot = new Snapshot();
			snapshot.setResourceType(RT_Snapshot);
			for (com.amazonaws.services.ec2.model.Tag t: awsSnapshot.getTags()) {
				if (KEY_NAME.equals(t.getKey())) {
					snapshot.setName(t.getValue());
					break;
				}
			}
			snapshot.setSnapshotId(awsSnapshot.getSnapshotId());
			snapshot.setDescription(awsSnapshot.getDescription());
			snapshots.add(snapshot);
		}
		return snapshots;
	}

	@Override
	public InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean withVolumes, Boolean noReboot, String backupOption) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
		String imageId = null;
		//		boolean success = false;
		try {
			// AWS イメージ作成。
			CreateImageRequest createImageRequest = new CreateImageRequest()
			.withInstanceId(instanceId)
			.withName(name)
			.withDescription(description)
			.withNoReboot(noReboot != null ? noReboot: false);

			// ルートデバイス以外のストレージをバックアップ対象にするか判定。
			if (!Boolean.TRUE.equals(withVolumes)) {
				// ルートデバイス名取得。
				com.amazonaws.services.ec2.model.DescribeInstancesRequest request = new com.amazonaws.services.ec2.model.DescribeInstancesRequest().withInstanceIds(instanceId);
				com.amazonaws.services.ec2.model.Instance instance = ec2.describeInstances(request).getReservations().get(0).getInstances().get(0);

				// ルートデバイス以外のデバイスの情報を取得。
				List<com.amazonaws.services.ec2.model.BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
				for (InstanceBlockDeviceMapping instanceBlockDeviceMapping: instance.getBlockDeviceMappings()) {
					if (instanceBlockDeviceMapping.getDeviceName().equals(instance.getRootDeviceName())) {
						continue;
					}

					com.amazonaws.services.ec2.model.BlockDeviceMapping blockDeviceMapping = new com.amazonaws.services.ec2.model.BlockDeviceMapping();
					blockDeviceMapping.setDeviceName(instanceBlockDeviceMapping.getDeviceName());
					blockDeviceMapping.setNoDevice("");
					blockDeviceMappings.add(blockDeviceMapping);
				}
				createImageRequest.setBlockDeviceMappings(blockDeviceMappings);
			}

			// AWS イメージ作成。
			CreateImageResult createImageResult = ec2.createImage(createImageRequest);
			imageId = createImageResult.getImageId();
			AWSUtil.addTags(ec2, createImageResult.getImageId(), new com.amazonaws.services.ec2.model.Tag(KEY_NAME, imageId), new com.amazonaws.services.ec2.model.Tag("SouceInstanceId", instanceId));

			final com.amazonaws.services.ec2.model.Image image = ec2.describeImages(new com.amazonaws.services.ec2.model.DescribeImagesRequest().withImageIds(imageId)).getImages().get(0);
			if (image.getState().equals("failed")) {
				ec2.deregisterImage(new DeregisterImageRequest().withImageId(imageId));
				throw ErrorCode.Resource_Instance_Backup_Image_State_Failed.cloudManagerFault();
			}

			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
								com.amazonaws.services.ec2.model.DeregisterImageRequest deregisterImageRequest = new com.amazonaws.services.ec2.model.DeregisterImageRequest().withImageId(image.getImageId());
								ec2.deregisterImage(deregisterImageRequest);
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					}
					);

			com.amazonaws.services.ec2.model.Instance instance = ec2.describeInstances(new com.amazonaws.services.ec2.model.DescribeInstancesRequest().withInstanceIds(instanceId)).getReservations().get(0).getInstances().get(0);

			BackupedInstanceDetail backupedData = new BackupedInstanceDetail();
			backupedData.instanceId = instanceId;
			for (com.amazonaws.services.ec2.model.Tag t: instance.getTags()) {
				if (KEY_NAME.equals(t.getKey())) {
					backupedData.name = t.getValue();
					break;
				}
			}
			backupedData.flavor = instance.getInstanceType();
			backupedData.zone = instance.getPlacement().getAvailabilityZone();
			backupedData.tags = new ArrayList<>();
			backupedData.createTime = new Date();
			for (com.amazonaws.services.ec2.model.Tag t: instance.getTags()) {
				if (KEY_NAME.equals(t.getKey())) continue;
				backupedData.tags.add(new Tag(t.getKey(), t.getValue()));
			}
			backupedData.instanceDetail = new InstanceDetail();
			backupedData.instanceDetail.subnetId = instance.getSubnetId();
			backupedData.instanceDetail.securityGroupIds = new ArrayList<>(instance.getSecurityGroups().size());
			for (GroupIdentifier gi: instance.getSecurityGroups()) {
				backupedData.instanceDetail.securityGroupIds.add(gi.getGroupId());
			}
			backupedData.instanceDetail.keyName = instance.getKeyName();
			backupedData.instanceDetail.monitoring = instance.getMonitoring().getState().equals("enabled");
			backupedData.instanceDetail.disableApiTermination = ec2.describeInstanceAttribute(new DescribeInstanceAttributeRequest().withInstanceId(instanceId).withAttribute(InstanceAttributeName.DisableApiTermination)).getInstanceAttribute().getDisableApiTermination();
			backupedData.instanceDetail.instanceInitiatedShutdownBehavior = ec2.describeInstanceAttribute(new DescribeInstanceAttributeRequest().withInstanceId(instanceId).withAttribute(InstanceAttributeName.InstanceInitiatedShutdownBehavior)).getInstanceAttribute().getInstanceInitiatedShutdownBehavior();
			backupedData.instanceDetail.ebsOptimized = instance.isEbsOptimized();
			backupedData.instanceDetail.rootBlockDevice = new EbsBlockDevice();
			for (InstanceBlockDeviceMapping device: instance.getBlockDeviceMappings()) {
				if (device.getDeviceName().equals(instance.getRootDeviceName()) &&  device.getEbs() != null) {
					Volume v = ec2.describeVolumes(new DescribeVolumesRequest().withVolumeIds(device.getEbs().getVolumeId())).getVolumes().get(0);
					backupedData.instanceDetail.rootBlockDevice.volumeType = v.getVolumeType();
					backupedData.instanceDetail.rootBlockDevice.volumeSize = v.getSize();
					backupedData.instanceDetail.rootBlockDevice.iops = v.getIops();
					backupedData.instanceDetail.rootBlockDevice.deleteOnTermination = device.getEbs().getDeleteOnTermination();
				}
			}

			ObjectMapper om = new ObjectMapper();
			ObjectWriter ow = om.writerWithType(BackupedInstanceDetail.class);
			String backupedDetail = ow.writeValueAsString(backupedData);
			store.put(RT_InstanceBackup, imageId, backupedDetail);

			final InstanceBackup instanceBackup = new InstanceBackup();
			instanceBackup.setResourceType(RT_InstanceBackup);
			instanceBackup.setInstanceBackupId(imageId);
			instanceBackup.setName(name);
			instanceBackup.setDescription(description);
			instanceBackup.setInstanceId(instanceId);
			instanceBackup.setPlatform(instance.getPlatform() == null ? PlatformKind.linux: PlatformKind.windows);
			instanceBackup.setCreateTime(backupedData.createTime);
			instanceBackup.setBackupedData(new InstanceBackup.BackupedData());
			instanceBackup.getBackupedData().setName(backupedData.name);
			instanceBackup.getBackupedData().setFlavor(backupedData.flavor);
			instanceBackup.getBackupedData().setZone(backupedData.zone);
			instanceBackup.getBackupedData().setTags(backupedData.tags);
			instanceBackup.getBackupedData().setInstanceDetail(om.writerWithType(InstanceDetail.class).writeValueAsString(backupedData.instanceDetail));

			//			success = true;

			return instanceBackup;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		finally {
			//			if (!success) {
			//				if (imageId != null) {
			//					com.amazonaws.services.ec2.model.DeregisterImageRequest deregisterImageRequest = new com.amazonaws.services.ec2.model.DeregisterImageRequest().withImageId(imageId);
			//					ec2.deregisterImage(deregisterImageRequest);
			//				}
			//			}
		}
	}

	@Override
	public void deleteInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		com.amazonaws.services.ec2.model.Image image = null;
		try {
			com.amazonaws.services.ec2.model.DescribeImagesResult result = ec2.describeImages(new com.amazonaws.services.ec2.model.DescribeImagesRequest().withImageIds(instanceBackupId));
			if (!result.getImages().isEmpty()) {
				image = result.getImages().get(0);
				ec2.deregisterImage(new DeregisterImageRequest(instanceBackupId));
			}
		}
		catch (AmazonServiceException e) {
			if (!AWSErrorCode.InvalidAMIID_NotFound.label().equals(e.getErrorCode())) {
				throw e;
			}
		}

		// 紐付いているスナップショットも削除。
		if (image != null) {
			for (com.amazonaws.services.ec2.model.BlockDeviceMapping bdm: image.getBlockDeviceMappings()) {
				if (bdm.getEbs() == null) continue;

				try {
					ec2.deleteSnapshot(new DeleteSnapshotRequest(bdm.getEbs().getSnapshotId()));
				}
				catch (AmazonServiceException e) {
					if (!AWSErrorCode.InvalidSnapshot_NotFound.label().equals(e.getErrorCode())) {
						// イメージは消しているので、メッセージだけ記録して処理続行。
						Logger logger = Logger.getLogger(this.getClass());
						logger.error(e.getMessage(), e);
					}
				}
			}
		}

		try {
			store.remove(RT_InstanceBackup, instanceBackupId);
		}
		catch (Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	public List<Image> getImages(String...imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImages(List<String> imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImagesWithFilter(Filter...filters) throws CloudManagerFault {
		return getImagesWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault {
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		List<com.amazonaws.services.ec2.model.Filter> awsFilters = new ArrayList<>();
		for (Filter filter: filters) {
			awsFilters.add(new com.amazonaws.services.ec2.model.Filter().withName(filter.getName()).withValues(filter.getValues()));
		}

		com.amazonaws.services.ec2.model.DescribeImagesRequest request = new com.amazonaws.services.ec2.model.DescribeImagesRequest().withFilters(awsFilters);
		com.amazonaws.services.ec2.model.DescribeImagesResult result = ec2.describeImages(request);

		List<Image> images = new ArrayList<>();
		for(com.amazonaws.services.ec2.model.Image awsImage: result.getImages()){
			Image image = new Image();
			image.setResourceType(RT_Image);
			image.setName(awsImage.getName());
			image.setImageId(awsImage.getImageId());
			image.setDescription(awsImage.getDescription());
			images.add(image);
		}
		return images;
	}

	@Override
	public Snapshot getSnapshot(String snapshotId) throws CloudManagerFault {
		List<Snapshot> snapshots = getSnapshots(snapshotId);
		if (snapshots.isEmpty()) {
			throw ErrorCode.Resource_InvalidSnapshot_NotFound.cloudManagerFault(snapshotId);
		}
		return snapshots.get(0);
	}

	@Override
	public Image getImage(String imageId) throws CloudManagerFault {
		List<Image> images = getImages(imageId);
		if (images.isEmpty()) {
			throw ErrorCode.Resource_InvalidImageID_NotFound.cloudManagerFault(imageId);
		}
		return images.get(0);
	}

	@Override
	public Instance restoreInstance(String instanceBackupId, String name, String flavor, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		try {
			String value = store.get(RT_InstanceBackup, instanceBackupId);

			ObjectMapper om = new ObjectMapper();
			BackupedInstanceDetail backuped = om.reader(BackupedInstanceDetail.class).readValue(value);

			backuped.name = name != null ? name: backuped.name;
			backuped.flavor = flavor != null ? flavor: backuped.flavor;
			backuped.zone = zone != null ? zone: backuped.zone;

			if (instanceDetail != null) {
				InstanceDetail override = om.reader(InstanceDetail.class).readValue(instanceDetail);

				if (backuped.instanceDetail == null) {
					backuped.instanceDetail = override;
				}
				else {
					backuped.instanceDetail.subnetId = override.subnetId != null ? override.subnetId: backuped.instanceDetail.subnetId;
					backuped.instanceDetail.keyName = override.keyName != null ? override.keyName: backuped.instanceDetail.keyName;
					backuped.instanceDetail.monitoring = override.monitoring != null ? override.monitoring: backuped.instanceDetail.monitoring;
					backuped.instanceDetail.disableApiTermination = override.disableApiTermination != null ? override.disableApiTermination: backuped.instanceDetail.disableApiTermination;
					backuped.instanceDetail.instanceInitiatedShutdownBehavior = override.instanceInitiatedShutdownBehavior != null ? override.instanceInitiatedShutdownBehavior: backuped.instanceDetail.instanceInitiatedShutdownBehavior;
					backuped.instanceDetail.ebsOptimized = override.ebsOptimized != null ? override.ebsOptimized: backuped.instanceDetail.ebsOptimized;
					if (backuped.instanceDetail.rootBlockDevice == null) {
						backuped.instanceDetail.rootBlockDevice = override.rootBlockDevice;
					}
					else if (override.rootBlockDevice != null) {
						backuped.instanceDetail.rootBlockDevice.volumeType = override.rootBlockDevice.volumeType != null ? override.rootBlockDevice.volumeType: backuped.instanceDetail.rootBlockDevice.volumeType;
						backuped.instanceDetail.rootBlockDevice.volumeSize = override.rootBlockDevice.volumeSize != null ? override.rootBlockDevice.volumeSize: backuped.instanceDetail.rootBlockDevice.volumeSize;
						backuped.instanceDetail.rootBlockDevice.iops = override.rootBlockDevice.iops != null ? override.rootBlockDevice.iops: backuped.instanceDetail.rootBlockDevice.iops;
						backuped.instanceDetail.rootBlockDevice.deleteOnTermination = override.rootBlockDevice.deleteOnTermination != null ? override.rootBlockDevice.deleteOnTermination: backuped.instanceDetail.rootBlockDevice.deleteOnTermination;
					}
					backuped.instanceDetail.securityGroupIds = override.securityGroupIds != null ? override.securityGroupIds: backuped.instanceDetail.securityGroupIds;
				}
			}

			List<Tag> tagsTemp = new ArrayList<>(tags);
			for (Tag t: backuped.tags) {
				if (tagsTemp.isEmpty()) {
					break;
				}
				for (Iterator<Tag> iter = tagsTemp.iterator(); iter.hasNext(); ) {
					Tag override = iter.next();
					if (t.getKey().equals(override.getKey())) {
						t.setValue(override.getValue());
						iter.remove();
						break;
					}
				}
			}
			for (Tag t: tagsTemp) {
				backuped.tags.add(t);
			}

			String backupedDetail = om.writerWithType(InstanceDetail.class).writeValueAsString(backuped.instanceDetail);

			return createInstance(backuped.name, backuped.flavor, instanceBackupId, backuped.zone, backupedDetail, backuped.tags);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public Storage restoreStorage(String storageBackupId, String name,String flavor, Integer size, String zone, String storageDetail) throws CloudManagerFault {
		try {
			String value = store.get(RT_StorageBackup, storageBackupId);

			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedStorageDetail.class);
			BackupedStorageDetail backuped = or.readValue(value);

			backuped.name = name != null ? name: backuped.name;
			backuped.flavor = flavor != null ? flavor: backuped.flavor;
			backuped.zone = zone != null ? zone: backuped.zone;
			backuped.size = size != null ? size: backuped.size;

			if (storageDetail != null) {
				StorageDetail override = or.readValue(storageDetail);

				if (backuped.storageDetail == null) {
					backuped.storageDetail = override;
				}
				else {
					backuped.storageDetail.iops = override.iops != null ? override.iops: backuped.storageDetail.iops;
				}
			}

			ObjectWriter ow = om.writerWithType(InstanceDetail.class);
			String backupedDetail = ow.writeValueAsString(backuped.storageDetail);

			return createStorage(backuped.name, backuped.flavor, backuped.size, storageBackupId, backuped.zone, backupedDetail);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public StorageBackup getStorageBackup(String storageBackupId) throws CloudManagerFault {
		try {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedStorageDetail.class);
			ObjectWriter ow = om.writerWithType(StorageDetail.class);

			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			com.amazonaws.services.ec2.model.DescribeSnapshotsResult result = ec2.describeSnapshots(new com.amazonaws.services.ec2.model.DescribeSnapshotsRequest().withSnapshotIds(storageBackupId));
			com.amazonaws.services.ec2.model.Snapshot snapshot = result.getSnapshots().get(0);

			if (!"error".equals(snapshot.getState())) {
				String backupDetail = store.get(RT_StorageBackup, snapshot.getSnapshotId());
				BackupedStorageDetail backuped = or.readValue(backupDetail);

				StorageBackup storageBackup = new StorageBackup();
				storageBackup.setResourceType(RT_StorageBackup);
				storageBackup.setStorageBackupId(snapshot.getSnapshotId());
				for (com.amazonaws.services.ec2.model.Tag t: snapshot.getTags()) {
					if (KEY_NAME.equals(t.getKey())) {
						storageBackup.setName(t.getValue());
						break;
					}
				}
				storageBackup.setStorageId(backuped.storageId);
				storageBackup.setDescription(snapshot.getDescription());
				storageBackup.setCreateTime(backuped.createTime);
				storageBackup.setBackupedData(new StorageBackup.BackupedData());
				storageBackup.getBackupedData().setName(backuped.name);
				storageBackup.getBackupedData().setFlavor(backuped.flavor);
				storageBackup.getBackupedData().setZone(backuped.zone);
				storageBackup.getBackupedData().setStorageDetail(ow.writeValueAsString(backuped.storageDetail));

				return storageBackup;
			}
			else {
				try {
					store.remove(RT_StorageBackup, snapshot.getVolumeId());
				}
				catch (Exception e1) {
					Logger.getLogger(this.getClass()).warn(e1.getMessage(), e1);
				}
				return null;
			}
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public List<StorageBackup> getStorageBackups(String... storageBackupId) throws CloudManagerFault {
		return getStorageBackups(Arrays.asList(storageBackupId));
	}

	@Override
	public List<StorageBackup> getStorageBackups(List<String> storageBackupIds) throws CloudManagerFault {
		try {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedStorageDetail.class);
			ObjectWriter ow = om.writerWithType(StorageDetail.class);

			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			com.amazonaws.services.ec2.model.DescribeSnapshotsResult result = ec2.describeSnapshots(new com.amazonaws.services.ec2.model.DescribeSnapshotsRequest().withSnapshotIds(storageBackupIds));

			List<String> storageBackupIdsList = storageBackupIds.isEmpty() ? store.getIds(RT_StorageBackup): new ArrayList<>(storageBackupIds);
			List<StorageBackup> storageBackups = new ArrayList<>();
			for (com.amazonaws.services.ec2.model.Snapshot snapshot: result.getSnapshots()) {
				for (Iterator<String> iter = storageBackupIdsList.iterator(); iter.hasNext();) {
					if (iter.next().equals(snapshot.getSnapshotId())) {
						iter.remove();
						break;
					}
				}

				if (!"error".equals(snapshot.getState())) {
					String backupDetail = store.get(RT_StorageBackup, snapshot.getSnapshotId());
					BackupedStorageDetail backuped = or.readValue(backupDetail);

					StorageBackup storageBackup = new StorageBackup();
					storageBackup.setResourceType(RT_StorageBackup);
					storageBackup.setStorageBackupId(snapshot.getSnapshotId());
					for (com.amazonaws.services.ec2.model.Tag t: snapshot.getTags()) {
						if (KEY_NAME.equals(t.getKey())) {
							storageBackup.setName(t.getValue());
							break;
						}
					}
					storageBackup.setStorageId(backuped.storageId);
					storageBackup.setDescription(snapshot.getDescription());
					storageBackup.setCreateTime(backuped.createTime);
					storageBackup.setBackupedData(new StorageBackup.BackupedData());
					storageBackup.getBackupedData().setName(backuped.name);
					storageBackup.getBackupedData().setFlavor(backuped.flavor);
					storageBackup.getBackupedData().setZone(backuped.zone);
					storageBackup.getBackupedData().setSize(backuped.size);
					storageBackup.getBackupedData().setStorageDetail(ow.writeValueAsString(backuped.storageDetail));

					storageBackups.add(storageBackup);
				}
				else {
					//　削除中、削除後のインスタンスなので関連する情報を削除。
					try {
						store.remove(RT_StorageBackup, snapshot.getVolumeId());
					}
					catch (Exception e1) {
						Logger.getLogger(this.getClass()).warn(e1.getMessage(), e1);
					}
				}
			}

			for (String storageBackupId: storageBackupIdsList) {
				//　削除中、削除後のインスタンスなので関連する情報を削除。
				try {
					store.remove(RT_StorageBackup, storageBackupId);
				}
				catch (Exception e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}

			return storageBackups;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public InstanceBackup getInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		try {

			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			com.amazonaws.services.ec2.model.DescribeImagesResult result = ec2.describeImages(new com.amazonaws.services.ec2.model.DescribeImagesRequest().withImageIds(instanceBackupId));
			com.amazonaws.services.ec2.model.Image image = result.getImages().get(0);

			if (!"deregistered".equals(image.getState())) {
				ObjectMapper om = new ObjectMapper();

				String backupDetail = store.get(RT_InstanceBackup, image.getImageId());
				ObjectReader or = om.reader(BackupedInstanceDetail.class);
				BackupedInstanceDetail backuped = or.readValue(backupDetail);

				InstanceBackup instanceBackup = new InstanceBackup();
				instanceBackup.setResourceType(RT_InstanceBackup);
				instanceBackup.setInstanceBackupId(image.getImageId());
				for (com.amazonaws.services.ec2.model.Tag t: image.getTags()) {
					if (KEY_NAME.equals(t.getKey())) {
						instanceBackup.setName(t.getValue());
						break;
					}
				}
				instanceBackup.setInstanceId(backuped.instanceId);
				instanceBackup.setDescription(image.getDescription());
				instanceBackup.setPlatform(image.getPlatform() == null ? PlatformKind.linux: PlatformKind.windows);
				instanceBackup.setCreateTime(backuped.createTime);
				instanceBackup.setBackupedData(new InstanceBackup.BackupedData());
				instanceBackup.getBackupedData().setName(backuped.name);
				instanceBackup.getBackupedData().setFlavor(backuped.flavor);
				instanceBackup.getBackupedData().setZone(backuped.zone);
				instanceBackup.getBackupedData().setTags(backuped.tags);

				ObjectWriter ow = om.writerWithType(InstanceDetail.class);
				instanceBackup.getBackupedData().setInstanceDetail(ow.writeValueAsString(backuped.instanceDetail));

				return instanceBackup;
			}
			else {
				//　削除中、削除後のインスタンスなので関連する情報を削除。
				try {
					store.remove(RT_InstanceBackup, image.getImageId());
				}
				catch (Exception e1) {
					Logger.getLogger(this.getClass()).warn(e1.getMessage(), e1);
				}
				return null;
			}
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public List<InstanceBackup> getInstanceBackups(String... instanceBackupIds) throws CloudManagerFault {
		return getInstanceBackups(Arrays.asList(instanceBackupIds));
	}

	@Override
	public List<InstanceBackup> getInstanceBackups(List<String> instanceBackupIds) throws CloudManagerFault {
		try {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedInstanceDetail.class);
			ObjectWriter ow = om.writerWithType(InstanceDetail.class);

			AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());
			com.amazonaws.services.ec2.model.DescribeImagesResult result = ec2.describeImages(new com.amazonaws.services.ec2.model.DescribeImagesRequest().withImageIds(instanceBackupIds));

			List<String> instanceBackupIdsList = instanceBackupIds.isEmpty() ? store.getIds(RT_InstanceBackup): new ArrayList<>(instanceBackupIds);
			List<InstanceBackup> instanceBackups = new ArrayList<>();
			for (com.amazonaws.services.ec2.model.Image image: result.getImages()) {
				for (Iterator<String> iter = instanceBackupIdsList.iterator(); iter.hasNext();) {
					if (iter.next().equals(image.getImageId())) {
						iter.remove();
						break;
					}
				}

				if (!"deregistered".equals(image.getState())) {
					String backupDetail = store.get(RT_InstanceBackup, image.getImageId());
					BackupedInstanceDetail backuped = or.readValue(backupDetail);

					InstanceBackup instanceBackup = new InstanceBackup();
					instanceBackup.setResourceType(RT_InstanceBackup);
					instanceBackup.setInstanceBackupId(image.getImageId());
					for (com.amazonaws.services.ec2.model.Tag t: image.getTags()) {
						if (KEY_NAME.equals(t.getKey())) {
							instanceBackup.setName(t.getValue());
							break;
						}
					}
					instanceBackup.setInstanceId(backuped.instanceId);
					instanceBackup.setDescription(image.getDescription());
					instanceBackup.setPlatform(image.getPlatform() == null ? PlatformKind.linux: PlatformKind.windows);
					instanceBackup.setCreateTime(backuped.createTime);
					instanceBackup.setBackupedData(new InstanceBackup.BackupedData());
					instanceBackup.getBackupedData().setName(backuped.name);
					instanceBackup.getBackupedData().setFlavor(backuped.flavor);
					instanceBackup.getBackupedData().setZone(backuped.zone);
					instanceBackup.getBackupedData().setTags(backuped.tags);
					instanceBackup.getBackupedData().setInstanceDetail(ow.writeValueAsString(backuped.instanceDetail));

					instanceBackups.add(instanceBackup);
				}
				else {
					//　削除中、削除後のインスタンスなので関連する情報を削除。
					try {
						store.remove(RT_InstanceBackup, image.getImageId());
					}
					catch (Exception e1) {
						Logger.getLogger(this.getClass()).warn(e1.getMessage(), e1);
					}
				}
			}

			for (String instanceBackupId: instanceBackupIdsList) {
				//　削除中、削除後のインスタンスなので関連する情報を削除。
				try {
					store.remove(RT_InstanceBackup, instanceBackupId);
				}
				catch (Exception e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}

			return instanceBackups;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public List<LoadBalancer> getLoadBalancers() throws CloudManagerFault {
		AmazonElasticLoadBalancing elb = AWSUtil.createAmazonElasticLoadBalancing(credential, region.getEndpoint(EP_TYPE_ELB).getLocation());
		AmazonEC2 ec2 = AWSUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		DescribeLoadBalancersResult elbResult = elb.describeLoadBalancers();
		DescribeInstancesResult instanceResult = ec2.describeInstances();

		List<LoadBalancer> lbs = new ArrayList<>();
		for (LoadBalancerDescription description: elbResult.getLoadBalancerDescriptions()) {
			LoadBalancer lb = new LoadBalancer();
			lb.setResourceType(RT_LoadBalancer);
			lb.setLbId(description.getLoadBalancerName());
			lb.setRegisterdInstanceIds(new ArrayList<String>());
			for (com.amazonaws.services.elasticloadbalancing.model.Instance i: description.getInstances()) {
				lb.getRegisterdInstanceIds().add(i.getInstanceId());
			}

			lb.setAvailableInstanceIds(new ArrayList<String>());
			for (Reservation rev: instanceResult.getReservations()) {
				if (description.getVPCId() != null) {
					for (com.amazonaws.services.ec2.model.Instance i: rev.getInstances()) {
						if (lb.getRegisterdInstanceIds().contains(i.getInstanceId())) {
							continue;
						}

						for (String subnetId: description.getSubnets()) {
							if (subnetId.equals(i.getSubnetId())) {
								lb.getAvailableInstanceIds().add(i.getInstanceId());
							}
						}
					}
				}
				else {
					for (com.amazonaws.services.ec2.model.Instance i: rev.getInstances()) {
						if (lb.getRegisterdInstanceIds().contains(i.getInstanceId())) {
							continue;
						}

						if (i.getVpcId() == null) {
							for (String zone: description.getAvailabilityZones()) {
								if (zone.equals(i.getPlacement().getAvailabilityZone())) {
									lb.getAvailableInstanceIds().add(i.getInstanceId());
								}
							}
						}
					}
				}
			}
			lbs.add(lb);
		}
		return lbs;
	}

	@Override
	public void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		AmazonElasticLoadBalancing elb = AWSUtil.createAmazonElasticLoadBalancing(credential, region.getEndpoint(EP_TYPE_ELB).getLocation());
		DescribeLoadBalancersResult elbResult = elb.describeLoadBalancers(new DescribeLoadBalancersRequest().withLoadBalancerNames(lbId));
		if (elbResult.getLoadBalancerDescriptions().isEmpty()) {
			// 該当する elb が存在しないので終了。
			return;
		}

		RegisterInstancesWithLoadBalancerRequest regRequest = new RegisterInstancesWithLoadBalancerRequest().withLoadBalancerName(lbId);
		for (LoadBalancerDescription description:  elbResult.getLoadBalancerDescriptions()) {
			for (com.amazonaws.services.elasticloadbalancing.model.Instance i: description.getInstances()) {
				if (i.getInstanceId().equals(instanceId)) {
					// 既に追加されているので終了。
					return;
				}
				regRequest.getInstances().add(i);
			}
			break;
		}

		regRequest.getInstances().add(new com.amazonaws.services.elasticloadbalancing.model.Instance(instanceId));
		elb.registerInstancesWithLoadBalancer(regRequest);
	}

	@Override
	public void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		AmazonElasticLoadBalancing elb = AWSUtil.createAmazonElasticLoadBalancing(credential, region.getEndpoint(EP_TYPE_ELB).getLocation());
		DescribeLoadBalancersResult elbResult = elb.describeLoadBalancers(new DescribeLoadBalancersRequest().withLoadBalancerNames(lbId));
		if (elbResult.getLoadBalancerDescriptions().isEmpty()) {
			// 該当する elb が存在しないので終了。
			return;
		}

		LoadBalancerDescription description = elbResult.getLoadBalancerDescriptions().get(0);
		RegisterInstancesWithLoadBalancerRequest regRequest = new RegisterInstancesWithLoadBalancerRequest().withLoadBalancerName(lbId);
		for (com.amazonaws.services.elasticloadbalancing.model.Instance i: description.getInstances()) {
			if (i.getInstanceId().equals(instanceId)) {
				continue;
			}
			regRequest.getInstances().add(i);
		}

		if (description.getInstances().size() == regRequest.getInstances().size()) {
			// 削除対象のインスタンスが登録されていないので終了。
			return;
		}

		regRequest.getInstances().add(new com.amazonaws.services.elasticloadbalancing.model.Instance(instanceId));
		elb.registerInstancesWithLoadBalancer(regRequest);
	}
}