package com.clustercontrol.ws.cloud;

import java.io.FileInputStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;

import javax.annotation.Resource;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import javax.xml.ws.Endpoint;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.spi.Invoker;
import javax.xml.ws.spi.Provider;

import org.apache.log4j.Logger;

import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.PluginFault;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.commons.HinemosProperties;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.cloud.validation.ValidationUtil;
import com.clustercontrol.plugin.impl.WebServicePlugin;
import com.clustercontrol.ws.cloud.security.HinemosAccessRight;
import com.clustercontrol.ws.cloud.security.HinemosAccessRights;
import com.sun.net.httpserver.HttpsConfigurator;
import com.sun.net.httpserver.HttpsServer;

public class Publisher implements AutoCloseable {
	public interface MethodChecker {
		void check(Method method, Object... args) throws PluginFault;
	}
	
    public interface UncaughtExceptionHandler {
    	PluginFault uncaughtException(Throwable e);
    }
	
	private class CloudInvoker extends Invoker {
		private IWebServiceBase implementor;
		private Field resourceField;
		private Endpoint endpoint;
		private WebServiceContext wsctx;
		
		public CloudInvoker(IWebServiceBase implementor) {
			assert implementor != null;
			for (Field f: implementor.getClass().getDeclaredFields()) {
				if (f.getAnnotation(Resource.class) != null ) {
					this.resourceField = f;
					break;
				}
			}
			this.implementor = implementor;
		}
		
		@Override
		public void inject(WebServiceContext wsctx) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException {
			this.wsctx = wsctx;
			
			if (resourceField != null) {
				boolean flag = resourceField.isAccessible();
				resourceField.setAccessible(true);
				try {
					resourceField.set(implementor, wsctx);
				}
				finally {
					resourceField.setAccessible(flag);
				}
			}
		}

		@Override
		public Object invoke(Method method, Object... args) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException {
			Logger logger = Logger.getLogger(Publisher.class);
			logger.debug("calling " + method.getName());

			try {
				long before = System.currentTimeMillis();

				context.set(wsctx);

				HinemosAccessRight accessRight = method.getAnnotation(HinemosAccessRight.class);
				if (accessRight != null) {
					for (SystemPrivilegeMode systemPrivilege: accessRight.right()) {
						HinemosUtil.authCheck(wsctx, new SystemPrivilegeInfo(accessRight.roleName(), systemPrivilege));
					}
				}
				else {
					HinemosAccessRights ars = method.getAnnotation(HinemosAccessRights.class);
					if (ars != null) {
						List<SystemPrivilegeInfo> spis = new ArrayList<SystemPrivilegeInfo>();
						for (HinemosAccessRight ar: ars.value()) {
							for (SystemPrivilegeMode systemPrivilege: ar.right()) {
								HinemosUtil.authCheck(wsctx, new SystemPrivilegeInfo(ar.roleName(), systemPrivilege));
							}
						}
						if (!spis.isEmpty()) {
							HinemosUtil.authCheck(wsctx, spis.toArray(new SystemPrivilegeInfo[0]));
						}
					}
				}
				
				ValidationUtil.getMethodValidator().validate(method, args);

				List<MethodChecker> checkerList = new ArrayList<>(checkers);
				for (MethodChecker checker: checkerList) {
					checker.check(method, args);
				}
				
				Object returnValue = method.invoke(implementor, args);
				
				context.set(null);
				
				long after = System.currentTimeMillis();

				StringBuilder sb = new StringBuilder();
				sb.append("Called ").append(method.getName()).append(" : ").append("elapsedTime=").append(after - before).append("ms");
				if (args != null) {
					for (Object arg: args) {
						sb.append(", arg=").append(arg.toString());
					}
				}
				logger.info(sb.toString());
				
				return returnValue;
			}
			catch (InvocationTargetException e) {
				logger.error(e.getTargetException().getMessage(), e.getTargetException());
				throw e;
			}
			catch (IllegalAccessException | IllegalArgumentException e) {
				logger.error(e.getMessage(), e);
				throw e;
			}
			catch (Exception e) {
				logger.error(e.getMessage(), e);

				if (handler != null) {
					PluginFault fault = handler.uncaughtException(e);
					if (fault != null) {
						throw new InvocationTargetException(fault);
					}
				}

				if (e instanceof InternalManagerError) {
					throw (InternalManagerError)e;
				}
				else {
					throw new InternalManagerError(e);
				}
			}
			finally {
				SessionService.current().close();
			}
		}

		@SuppressWarnings("unused")
		public Endpoint getEndpoint() {
			return endpoint;
		}

		public void setEndpoint(Endpoint endpoint) {
			this.endpoint = endpoint;
		}

		public IWebServiceBase getImplementor() {
			return implementor;
		}
	};
	
	private List<CloudInvoker> endpoints = Collections.synchronizedList(new ArrayList<CloudInvoker>());
	private List<MethodChecker> checkers = Collections.synchronizedList(new ArrayList<MethodChecker>());
	private  UncaughtExceptionHandler handler;
	
	private static ThreadLocal<WebServiceContext> context  = new ThreadLocal<WebServiceContext>() {
		protected WebServiceContext initialValue()
		{
			return null;
		}
	};
	
	public static WebServiceContext getCurrentContext() {
		return context.get();
	}
	
	public Publisher() {
	}
	
	@SuppressWarnings("unchecked")
	public void publish(String address, IWebServiceBase implementor) {
		URL url = null;
		try {
			url = new URL(address);
		}	
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		
		if("https".equals(url.getProtocol())){
			Map<URL, HttpsServer> httpsServerMap = null;
			{
				Field field = null;
				Boolean accesible = null;
				try {
					field = WebServicePlugin.class.getDeclaredField("httpsServerMap");
					
					accesible = field.isAccessible();
					field.setAccessible(true);
					httpsServerMap = (Map<URL, HttpsServer>)field.get(null);
				}
				catch(Exception e) {
					throw new IllegalStateException(e);
				}
				finally {
					if (accesible != null) {
						field.setAccessible(accesible);
					}
				}
			}

			ThreadPoolExecutor _threadPool = null;
			{
				Field field = null;
				Boolean accesible = null;
				try {
					field = WebServicePlugin.class.getDeclaredField("_threadPool");
					
					accesible = field.isAccessible();
					field.setAccessible(true);
					_threadPool = (ThreadPoolExecutor)field.get(null);
				}
				catch(Exception e) {
					throw new IllegalStateException(e);
				}
				finally {
					if (accesible != null) {
						field.setAccessible(accesible);
					}
				}
			}

			CloudInvoker invoker = new CloudInvoker(implementor);
			implementor.start();
			Endpoint e =Provider.provider().createEndpoint(null, implementor.getClass(), invoker);
			e.setExecutor(_threadPool);
			
			URL urlPrefix;
			try {
				urlPrefix = new URL("https://" + url.getHost() + ":" + url.getPort());
			}
			catch (Exception e1) {
				throw new InternalManagerError(e1);
			}

			HttpsServer server = httpsServerMap.get(urlPrefix);
			if (server == null) {
				try {
					// HTTPS Serverの作成（HTTPSサーバの開始は、後で一括して行うため、ここではインスタンスの生成のみに留める
					String protocol = HinemosProperties.getProperty("common.ws.https.protocol", "TLS");
					String keystorePath = HinemosProperties.getProperty("common.ws.https.keystore.path", "/root/keystore");
					String keystorePassword = HinemosProperties.getProperty("common.ws.https.keystore.password", "hinemos");
					String keystoreType = HinemosProperties.getProperty("common.ws.https.keystore.type", "PKCS12");
					SSLContext ssl = SSLContext.getInstance(protocol);
					KeyManagerFactory keyFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
					KeyStore store = KeyStore.getInstance(keystoreType);
					store.load(new FileInputStream(keystorePath), keystorePassword.toCharArray());
					keyFactory.init(store, keystorePassword.toCharArray());
					TrustManagerFactory trustFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
					trustFactory.init(store);
					ssl.init(keyFactory.getKeyManagers(), trustFactory.getTrustManagers(), new SecureRandom());
					HttpsConfigurator configurator = new HttpsConfigurator(ssl);
	
					// 新規にHTTPSSeverを作って、Hashmapに登録する
					server = HttpsServer.create(new InetSocketAddress(urlPrefix.getHost(), urlPrefix.getPort()), 0);
					server.setHttpsConfigurator(configurator);
					httpsServerMap.put(urlPrefix, server);
				}
				catch (Exception e1) {
					throw new InternalManagerError(e1);
				}
			}
							
			e.publish(server.createContext(url.getPath()));
			invoker.setEndpoint(e);
			endpoints.add(invoker);
		}
		else {
			CloudInvoker invoker = new CloudInvoker(implementor);
			implementor.start();
			Endpoint e =Provider.provider().createEndpoint(null, implementor.getClass(), invoker);
			e.publish(address);
			invoker.setEndpoint(e);
			endpoints.add(invoker);
		}
	}
	
	@Override
	public void close() throws Exception {
		synchronized (endpoints) {
			for (CloudInvoker ep: endpoints) {
				try {
//					ep.getEndpoint().stop();
				}
				catch (Exception e) {
					Logger logger = Logger.getLogger(Publisher.class);
					logger.error(e.getMessage(), e);
				}
				try {
					ep.getImplementor().stop();
				}
				catch (Exception e) {
					Logger logger = Logger.getLogger(Publisher.class);
					logger.error(e.getMessage(), e);
				}
			}
		}
	}
	
	public void addChecker(MethodChecker checker) {
		checkers.add(checker);
	}
	
	public void setUncaughtExceptionHandler(UncaughtExceptionHandler handler) {
		this.handler = handler;
	}
}