package com.clustercontrol.cloud.factory;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityExistsException;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.bean.CreateCloudUserRequest;
import com.clustercontrol.cloud.bean.ModifyCloudUserRequest;
import com.clustercontrol.cloud.dao.CloudAccountResourceDao;
import com.clustercontrol.cloud.dao.CloudUserDao;
import com.clustercontrol.cloud.dao.CloudUserDao.CloudUserType;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.registry.IObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectRegistryService;

@Transactional
public class CloudUserOperator implements ICloudUserOperator {
	public CloudUserOperator() {
	}
	
	@Override
	public CloudUser createCloudAccount(CreateCloudUserRequest request) throws CloudManagerFault {
		return createInternalCloudUser(request, CloudUserType.account);
	}

	@Override
	public CloudUser createCloudUser(CreateCloudUserRequest request) throws CloudManagerFault {
		return createInternalCloudUser(request, CloudUserType.user);
	}

	private CloudUser createInternalCloudUser(CreateCloudUserRequest request, CloudUserType type) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudAccountResourceDao arDao = em.find(CloudAccountResourceDao.class, request.getAccountResourceId());
		if (arDao == null) {
			throw ErrorCode.ACCOUNTRESOURCE_INVALID_ACCOUNTRESOURCE_NOT_FOUND.cloudManagerFault(request.getAccountResourceId());
		}

		CloudUserDao cud = new CloudUserDao();
		cud.setCloudUserId(request.getCloudUserId());
		cud.setCloudUserName(request.getCloudUserName());
		cud.setAccountResource(arDao);
		cud.setCloudUserType(type);
		cud.setDescription(request.getDescription());
		cud.setAccessKey(request.getAccessKey());
		cud.setSecretKey(request.getSecretKey());
		cud.setRoleId(request.getRoleId());

		CloudUser cu = null;
		try {
			em.persist(cud);
			cu = new CloudUser(cud);
		}
		catch (EntityExistsException e) {
			throw ErrorCode.CLOUDUSER_ALREADY_EXIST.cloudManagerFault(cud.getCloudUserId());
		}

		try {
			ObjectRegistryService.registry().get(IObjectChangedService.class).firePostAddedEvent(null, CloudUser.class, cu);
			return cu;
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public void removeCloudUser(String cloudUserId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudUserDao dao = em.find(CloudUserDao.class, cloudUserId);
		if (dao == null) {
			throw ErrorCode.CLOUDUSER_NOT_FOUND.cloudManagerFault(cloudUserId);
		}
		
		if (dao.getCloudUserType() == CloudUserType.account) {
			dao.getAccountResource().setAccount(null);
		}

		try {
			ObjectRegistryService.registry().get(IObjectChangedService.class).firePreRemovedEvent(null, CloudUser.class, new CloudUser(dao));
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}

		em.remove(dao);
	}

	@Override
	public CloudUser modifyCloudUser(ModifyCloudUserRequest request) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudUserDao cud = em.find(CloudUserDao.class, request.getCloudUserId());
		if (cud == null) {
			throw ErrorCode.CLOUDUSER_NOT_FOUND.cloudManagerFault(request.getCloudUserId());
		}
		if (request.getCloudUserName() != null) {
			cud.setCloudUserName(request.getCloudUserName());
		}
		if (request.getDescription() != null) {
			cud.setDescription(request.getDescription());
		}
		if (request.getAccessKey() != null) {
			cud.setAccessKey(request.getAccessKey());
		}
		if (request.getSecretKey() != null) {
			cud.setSecretKey(request.getSecretKey());
		}
		return new CloudUser(cud);
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public CloudUser findCloudUser(String cloudUserId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudUserDao cud = em.find(CloudUserDao.class, cloudUserId);
		if (cud == null) {
			throw ErrorCode.CLOUDUSER_NOT_FOUND.cloudManagerFault(cloudUserId);
		}
		return new CloudUser(cud);
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<CloudUser> findCloudUserByAccountResource(String accountResourceId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		List<CloudUserDao> culs = em.findByFilter(CloudUserDao.class, new Filter("accountResource.accountResourceId", accountResourceId));

		List<CloudUser> cus = new ArrayList<CloudUser>();
		for (CloudUserDao cud: culs) {
			cus.add(new CloudUser(cud));
		}
		return cus;
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public CloudUser findCloudUserByUserId(String hinemosUserId, String roleId) throws CloudManagerFault {
		try {
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
//			Query query = em.createQuery("select c from CloudUserDao as c join RoleEntity as r on c.roleId = r.roleId, UserEntity as h where r.roleId = '" + roleId + "' and r member of h.roleEntities and h.userId = '" + hinemosUserId + "'");
			Query query = em.createQuery("SELECT c FROM UserEntity u JOIN u.roleEntities r JOIN CloudUserDao AS c ON c.roleId = r.roleId WHERE u.userId = :userId AND r.roleId = :roleId");
			query.setParameter("userId", hinemosUserId);
			query.setParameter("roleId", roleId);
			return new CloudUser((CloudUserDao)query.getSingleResult());
		}
		catch (NoResultException e) {
			throw ErrorCode.CLOUDUSER_INVALID_NOT_RELAITED_HINEMOS_ACCOUNT.cloudManagerFault(hinemosUserId, roleId);
		}
	}

	@Override
	public CloudUser findCloudUserByRoleId(String roleId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		List<CloudUserDao> cuds = em.findByFilter(CloudUserDao.class, new Filter("roleId", roleId));
		if (cuds.isEmpty()) {
			throw ErrorCode.CLOUDUSER_NOT_FOUND.cloudManagerFault(roleId);
		}
		return new CloudUser(cuds.get(0));
	}
}