/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.bean.CloudInstance;
import com.clustercontrol.cloud.bean.CloudInstanceBackup;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CreateInstanceBackupRequest;
import com.clustercontrol.cloud.bean.CreateInstanceByTemplateRequest;
import com.clustercontrol.cloud.bean.CreateInstanceRequest;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.LoadBalancer;
import com.clustercontrol.cloud.bean.RegistNodeRequest;
import com.clustercontrol.cloud.bean.RestoreInstanceRequest;
import com.clustercontrol.cloud.bean.StartInstanceRequest;
import com.clustercontrol.cloud.bean.StopInstanceRequest;
import com.clustercontrol.cloud.dao.CloudInstanceDao;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;

public interface IInstanceOperator extends IResourceOperatorBase {
	
	public class UpdateAllResult {
		// AWS および Hinemos DB に登録されているインスタンスの対応情報保持。
		public static class InstanceMapping {
			public InstanceMapping(IResourceManagement.Instance instance, CloudInstanceDao cloudInstance) {
				this.instance = instance;
				this.cloudInstance = cloudInstance;
			}
			public IResourceManagement.Instance instance;
			public CloudInstanceDao cloudInstance;
		}
		
		public final List<InstanceMapping> both;
		public final List<InstanceMapping> onlyAws;
		public final List<CloudInstanceDao> onlyCloud;

		public UpdateAllResult(List<InstanceMapping> both, List<InstanceMapping> onlyAws, List<CloudInstanceDao> onlyCloud) {
			this.both = Collections.unmodifiableList(both);
			this.onlyAws = Collections.unmodifiableList(onlyAws);
			this.onlyCloud = Collections.unmodifiableList(onlyCloud);
		}
		
		public List<CloudInstance> getCloudInstances() {
			try {
				Logger logger = Logger.getLogger(UpdateAllResult.class);
				
				List<CloudInstance> cloudInstaces = new ArrayList<CloudInstance>();
	
				// AWS および Hinemos DB の両方に存在する。 
				for (InstanceMapping mapping: both) {
					ICloudServiceOperator operator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
					CloudService service = operator.findCloudService(mapping.cloudInstance.getCloudServiceId());
					cloudInstaces.add(new CloudInstance(mapping.cloudInstance, service.getCloudTypeId(), mapping.instance, HinemosUtil.getFacilityName(mapping.cloudInstance.getFacilityId())));
					logger.debug("Both : facilityId=" + mapping.cloudInstance.getFacilityId() + ", instanceId=" + mapping.cloudInstance.getInstanceId());
				}
				
				// AWS のみに存在する。 
				for (InstanceMapping mapping: onlyAws) {
					ICloudServiceOperator operator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
					CloudService service = operator.findCloudService(mapping.cloudInstance.getCloudServiceId());
					cloudInstaces.add(new CloudInstance(mapping.cloudInstance, service.getCloudTypeId(), mapping.instance, HinemosUtil.getFacilityName(mapping.cloudInstance.getFacilityId())));
					logger.debug("AWS : instanceId=" + mapping.cloudInstance.getInstanceId());
				}
	
				// Hinemos DB のみに存在する。 
				for (CloudInstanceDao ci: onlyCloud) {
					ICloudServiceOperator operator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
					CloudService service = operator.findCloudService(ci.getCloudServiceId());
					cloudInstaces.add(new CloudInstance(ci, service.getCloudTypeId(), HinemosUtil.getFacilityName(ci.getFacilityId())));
					logger.debug("Cloud : facilityId=" + ci.getFacilityId() + ", instanceId=" + ci.getInstanceId());
				}
				
				return cloudInstaces;
			}
			catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}
	
	public class UpdateResult {
		public final IResourceManagement.Instance awsInstance;
		public final CloudInstanceDao cloudInstance;

		public UpdateResult(IResourceManagement.Instance awsInstance, CloudInstanceDao cloudInstance) {
			this.awsInstance = awsInstance;
			this.cloudInstance = cloudInstance;
		}
		
		public CloudInstance getCloudInstance() {
			if (cloudInstance != null) {
				try {
					ICloudServiceOperator operator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
					CloudService service = operator.findCloudService(cloudInstance.getCloudServiceId());
					if (awsInstance != null) {
						// クラウドインスタンスがないので、ダミーを作成し返す。
						return new CloudInstance(cloudInstance, service.getCloudTypeId(), awsInstance, HinemosUtil.getFacilityName(cloudInstance.getFacilityId()));
					}
					else {
						// クラウドインスタンスも EC2 インスタンスも存在しないので例外を返す。
						return new CloudInstance(cloudInstance, service.getCloudTypeId(), HinemosUtil.getFacilityName(cloudInstance.getFacilityId()));
					}
				}
				catch (Exception e) {
					throw new InternalManagerError(e);
				}
			}
			
			return null;
		}
	}

	void setFlags(boolean update, boolean regist, boolean mount);
	
	CloudInstance createInstance(CreateInstanceRequest request) throws CloudManagerFault, InvalidRole;
	CloudInstance createInstanceByTemplate(CreateInstanceByTemplateRequest request) throws CloudManagerFault, InvalidRole;
	CloudInstanceBackup createInstanceBackup(CreateInstanceBackupRequest request) throws CloudManagerFault, InvalidRole;
	CloudInstance restoreInstance(RestoreInstanceRequest request) throws CloudManagerFault, InvalidRole;
	void removeInstance(String instanceId) throws CloudManagerFault;	
	void removeInstanceBackup(String imageId) throws CloudManagerFault;
	InstanceStateChange startInstance(StartInstanceRequest request) throws CloudManagerFault;
	InstanceStateChange stopInstance(StopInstanceRequest request) throws CloudManagerFault;	
	UpdateResult updateInstance(String instanceId) throws CloudManagerFault, InvalidRole;
	UpdateAllResult updateAllInstance() throws CloudManagerFault, InvalidRole;
	CloudInstance registNode(final RegistNodeRequest request) throws CloudManagerFault, InvalidRole;	
	void unregistNode(String instanceId) throws CloudManagerFault, InvalidRole;
	CloudInstance getInstance(String instanceId) throws CloudManagerFault, InvalidRole;
	CloudInstance getInstanceByFacilityId(String facilityId) throws CloudManagerFault, InvalidRole;
	List<CloudInstance> getAllInstances() throws CloudManagerFault, InvalidRole;

	CloudInstanceBackup getInstanceBackup(String imageId) throws CloudManagerFault;
	List<CloudInstanceBackup> getAllInstanceBackup() throws CloudManagerFault;
	void updateAllInstanceBackup() throws CloudManagerFault, InvalidRole;
	List<CloudInstanceBackup> getInstanceBackupByInstanceId(String instanceId) throws CloudManagerFault;

	public List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	List<LoadBalancer> getLoadBalancers() throws CloudManagerFault;
	void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault;
	void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault;
	
//	List<String> getFlavors() throws CloudManagerFault;
}