/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory.monitors;

import static com.clustercontrol.cloud.util.CloudMessageUtil.getExceptionStackTrace;

import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.session.AccessControllerBean;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.HinemosCredential;
import com.clustercontrol.cloud.ICloudOption;
import com.clustercontrol.cloud.Messages;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudAccountResource;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.commons.CloudConstants;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.factory.IAccountResourceOperator;
import com.clustercontrol.cloud.factory.ICloudServiceOperator;
import com.clustercontrol.cloud.factory.ICloudUserOperator;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.CloudCredential;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.cloud.util.RepositoryControllerBeanWrapper;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.monitor.plugin.model.MonitorPluginInfoEntity;
import com.clustercontrol.monitor.plugin.model.MonitorPluginStringInfoEntity;
import com.clustercontrol.monitor.plugin.util.QueryUtil;
import com.clustercontrol.monitor.run.bean.MonitorRunResultInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.session.NotifyControllerBean;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class CloudServiceBillingMonitor extends RunMonitorNumericValueType {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( RunMonitor.class );

//	private static final String pluginId = "MON_BLL"; 
	public static final String monitorTypeId = "MON_BILLING";
	public static final int monitorType = MonitorTypeConstant.TYPE_NUMERIC;
	
	public static final String key_target = "target";
	
	/** 監視情報 */
	private MonitorPluginInfoEntity m_plugin = null;

	/** 文字列情報リスト */
	private List<MonitorPluginStringInfoEntity> m_monitorPluginStringInfoList;

//	/** 数値情報リスト */
//	private List<MonitorPluginNumericInfoEntity> m_monitorPluginNumericInfoList;
	
	/** メッセージ **/
	private String m_message = null;

	/** オリジナルメッセージ */
	private String m_messageOrg = null;

	/**
	 * コンストラクタ
	 */
	public CloudServiceBillingMonitor() {
		super();
	}

	@Override
	public boolean collect(String facilityId) {
		try {
			// Cron 経由で起動した場合は、処理を行うユーザーをセッション情報として設定する必要がある。
			// 以下はそのおまじない。
			String userId = CloudPropertyConstants.internal_thread_admin_user.value();
			HinemosSessionContext.instance().setProperty(HinemosSessionContext.LOGIN_USER_ID, userId);
			SessionService.current().setHinemosCredential(new HinemosCredential(userId));
			HinemosSessionContext.instance().setProperty(HinemosSessionContext.IS_ADMINISTRATOR, new AccessControllerBean().isAdministrator());
			
			NodeInfo node = RepositoryControllerBeanWrapper.bean().getNode(facilityId);
			String accountResourceId = node.getCloudAccountResouce();
			if (accountResourceId == null) {
				m_messageOrg = m_message = ErrorCode.MONITOR_MISSING_ACCOUNTRESOURCEID.getMessage(accountResourceId);
				return false;
			}

			IAccountResourceOperator accountResourceOperator = ObjectRegistryService.registry().get(IAccountResourceOperator.class);
			CloudAccountResource accountResource = accountResourceOperator.findCloudAccountResource(accountResourceId);

			ICloudServiceOperator serviceOperator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
			CloudService cloudService = serviceOperator.findCloudService(accountResource.getCloudServiceId());
			
			String target = null;
			for (MonitorPluginStringInfoEntity entry: m_monitorPluginStringInfoList) {
				if (key_target.equals(entry.getId().getKey())) {
					target = entry.getValue();
					break;
				}
			}

			ICloudUserOperator userOperator = ObjectRegistryService.registry().get(ICloudUserOperator.class);
			CloudUser cloudUser = userOperator.findCloudUser(accountResource.getAccountId());

			// オプション取得。
			ICloudOption cloudOption = ObjectRegistryService.registry().get(ICloudOption.class, cloudService.getCloudTypeId());
			cloudOption.setCredential(new CloudCredential(cloudUser));
			
			// 監視結果取得。
			ICloudOption.Billing billing = cloudOption.getBillingForCloudService(target);

			// 取得結果を確認。
			if (billing != null) {
				m_message = Messages.getString("cloud.billingalarm.message.notify", new Object[]{String.valueOf(billing.getPrice())});
				m_messageOrg = Messages.getString("cloud.billingalarm.message.notify.org", new Object[]{String.valueOf(billing.getPrice()), CloudConstants.time_format.format(billing.getUpdateDate().getTime())});
				m_nodeDate = billing.getUpdateDate().getTime();
				m_value = billing.getPrice();
				
				return true;
			}
			else {
				m_messageOrg = m_message = Messages.getString("cloud.billingalarm.message.notify.no_data", new Object[]{target});
				if (m_now != null) {
					m_nodeDate = m_now.getTime();
				}

				return false;
			}
		}
		catch (Exception e) {
			m_message = e.getMessage();
			m_messageOrg = e.getMessage() + "\n" + getExceptionStackTrace(e);

			// 例外が発生したが、結果を返しているので、とりあえず true を返す。
			return false;
		}
	}

	@Override
	protected void setCheckInfo() throws MonitorNotFound {
		// 監視情報を取得
		m_plugin = QueryUtil.getMonitorPluginInfoPK(m_monitorId);
		// 監視情報を設定
//		m_monitorPluginNumericInfoList = m_plugin.getMonitorPluginNumericInfoEntities();
		m_monitorPluginStringInfoList = m_plugin.getMonitorPluginStringInfoEntities();
	}

	/**
	 *  マルチスレッドを実現するCallableTaskに渡すためのインスタンスを作成するメソッド
	 */
	@Override
	protected RunMonitor createMonitorInstance() throws HinemosUnknown {
		return new CloudServiceBillingMonitor();
	}

	/**
	 * メッセージID
	 */
	@Override
	public String getMessageId(int key) {
		HinemosUtil.Priority priority = HinemosUtil.Priority.priority(key);
		return priority == null ? HinemosUtil.MESSAGE_ID_UNKNOWN: priority.messageId;
	}
	
	/**
	 * メッセージ
	 */
	@Override
	public String getMessage(int key) {
		return m_message;
	}
	
	/**
	 * オリジナルメッセージ
	 */
	@Override
	public String getMessageOrg(int key) {
		return m_messageOrg;
	}
	
	protected void notify(
			boolean isNode,
			String facilityId,
			int result,
			Date generationDate,
			MonitorRunResultInfo resultInfo) throws HinemosUnknown {

		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() isNode = " + isNode + ", facilityId = " + facilityId
					+ ", result = " + result + ", generationDate = " + generationDate.toString()
					+ ", resultInfo = " + resultInfo.getMessage());
		}

		Integer monitorFlg = 0;

		monitorFlg = m_monitor.getMonitorFlg();

		// 監視無効の場合、通知しない
		if(monitorFlg == ValidConstant.TYPE_INVALID){
			m_log.debug("notify() isNode = " + isNode + ", facilityId = " + facilityId
					+ ", result = " + result + ", generationDate = " + generationDate.toString()
					+ ", resultInfo = " + resultInfo.getMessage() + ", monitorFlg is false");
			return;
		}

		// 通知IDが指定されていない場合、通知しない
		String notifyGroupId = resultInfo.getNotifyGroupId();
		if(notifyGroupId == null || "".equals(notifyGroupId)){
			return;
		}

		// 通知情報を設定
		OutputBasicInfo notifyInfo = new OutputBasicInfo();
		notifyInfo.setPluginId(m_monitorTypeId);
		notifyInfo.setMonitorId(m_monitorId);
		notifyInfo.setApplication(m_monitor.getApplication());

		String facilityPath = new RepositoryControllerBean().getFacilityPath(facilityId, null);
		notifyInfo.setFacilityId(facilityId);
		notifyInfo.setScopeText(facilityPath);

		int priority = -1;
		String messageId = "";
		String message = "";
		String messageOrg = "";

		if(isNode){
			// ノードの場合
			priority = resultInfo.getPriority();
			messageId = resultInfo.getMessageId();
			message = resultInfo.getMessage();
			messageOrg = resultInfo.getMessageOrg();
		}
		else{
			// スコープの場合
			priority = result;
			messageId = getMessageIdForScope(result);
			message = getMessageForScope(result);
			messageOrg = getMessageOrgForScope(result);
		}
		notifyInfo.setPriority(priority);
		notifyInfo.setMessageId(messageId);
//		// 通知抑制用のサブキーを設定。
//		if(resultInfo.getDisplayName() != null && !"".equals(resultInfo.getDisplayName())){
//			// 監視結果にデバイス名を含むものは、デバイス名をサブキーとして設定。
//			notifyInfo.setSubKey(resultInfo.getDisplayName());
//		} else if(resultInfo.getPatternText() != null){
//			// 監視結果にパターンマッチ文字列を含むものは、デバイス名をサブキーとして設定。
//			notifyInfo.setSubKey(resultInfo.getPatternText());
//		}
		
		String target = null;
		for (MonitorPluginStringInfoEntity entry: m_monitorPluginStringInfoList) {
			if (key_target.equals(entry.getId().getKey())) {
				target = entry.getValue();
				break;
			}
		}
		notifyInfo.setSubKey(target);
		notifyInfo.setMessage(message);
		notifyInfo.setMessageOrg(messageOrg);
		if (generationDate != null) {
			notifyInfo.setGenerationDate(generationDate.getTime());
		}
		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() priority = " + priority
					+ " , messageId = " + messageId
					+ " , message = " + message
					+ " , messageOrg = " + messageOrg
					+ ", generationDate = " + generationDate);
		}

		// ログ出力情報を送信
		if (m_log.isDebugEnabled()) {
			m_log.debug("sending message"
					+ " : priority=" + notifyInfo.getPriority()
					+ " generationDate=" + notifyInfo.getGenerationDate() + " pluginId=" + notifyInfo.getPluginId()
					+ " monitorId=" + notifyInfo.getMonitorId() + " facilityId=" + notifyInfo.getFacilityId()
					+ " subKey=" + notifyInfo.getSubKey()
					+ ")");
		}
		new NotifyControllerBean().notify(notifyInfo, notifyGroupId);
	}
}
