/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.ws.cloud;

import java.util.List;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.bean.AvailableRole;
import com.clustercontrol.cloud.bean.CloudAccountResource;
import com.clustercontrol.cloud.bean.CloudInstance;
import com.clustercontrol.cloud.bean.CloudInstanceBackup;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudStorage;
import com.clustercontrol.cloud.bean.CloudStorageBackup;
import com.clustercontrol.cloud.bean.CloudTemplate;
import com.clustercontrol.cloud.bean.CloudType;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.bean.CreateAccountResourceRequest;
import com.clustercontrol.cloud.bean.CreateCloudUserRequest;
import com.clustercontrol.cloud.bean.CreateInstanceBackupRequest;
import com.clustercontrol.cloud.bean.CreateInstanceByTemplateRequest;
import com.clustercontrol.cloud.bean.CreateInstanceRequest;
import com.clustercontrol.cloud.bean.CreateStorageBackupRequest;
import com.clustercontrol.cloud.bean.CreateStorageRequest;
import com.clustercontrol.cloud.bean.CreateTemplateRequest;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.ModifyAccountBillingDetailRequest;
import com.clustercontrol.cloud.bean.ModifyAccountResourceRequest;
import com.clustercontrol.cloud.bean.ModifyCloudUserRequest;
import com.clustercontrol.cloud.bean.ModifyTemplateRequest;
import com.clustercontrol.cloud.bean.RegistNodeRequest;
import com.clustercontrol.cloud.bean.RestoreInstanceRequest;
import com.clustercontrol.cloud.bean.RestoreStorageRequest;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StartInstanceRequest;
import com.clustercontrol.cloud.bean.StopInstanceRequest;
import com.clustercontrol.cloud.bean.Zone;
import com.clustercontrol.cloud.validation.ValidationFault;
import com.clustercontrol.cloud.validation.annotation.Into;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;

/**
 * クラウド管理オプションの Web サービス API 定義。
 * 
 *
 */
@WebService(targetNamespace = "http://cloud.ws.clustercontrol.com")
public interface CloudEndpoint {
	/**
	 * クラウドインスタンスの作成を AWS に指示し、該当するノードを作成します。
	 * 
	 * @param request クラウドインスタンス作成情報。
	 * @return クラウド管理オプションに登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstance addInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * テンプレートに従ったクラウドインスタンスの作成を AWS に指示し、該当するノードを作成します。
	 * なお、エージェントが自動起動するよう設定された AMI イメージを用いてインスタンスを作成した場合、テンプレートに指定されたジョブを実行します。
	 * 
	 * @param request テンプレート付きのクラウドインスタンス作成情報。
	 * @return クラウド管理オプションに登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstance addInstanceByTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceByTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 作成済みのクラウドインスタンスバックアップ情報から、クラウドインスタンスを復元します。
	 * クラウドインスタンスバックアップ情報は、{@link #addInstanceBackup(CreateInstanceBackupRequest request) addInstanceBackup} にて作成します。
	 * 
	 * @param request クラウドインスタンスバックアップ情報。
	 * @return クラウド管理オプションに登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstance restoreInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RestoreInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * AWS からインスタンスを削除し、ノードが紐づいている場合は、併せて削除されます。
	 * 
	 * @param region 削除するクラウドインスタンスが所属するリージョン名。
	 * @param instanceId 削除するクラウドインスタンス Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 作成済みのクラウドインスタンス情報を取得します。
	 * 
	 * @param regionName クラウドインスタンス情報が所属するリージョン名。
	 * @param instanceId クラウドインスタンス Id。
	 * @return クラウドインスタンス情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstance getInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンに作成されているインスタンス情報のリストを取得します。
	 * 取得できるリストは、実行ユーザに紐づくクラウドユーザ(AWS アカウント、IMA ユーザ)の参照可能なインスタンスに限ります。
	 * 
	 * @param regionName リージョン名。
	 * @return クラウドインスタンス情報の一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<CloudInstance> getInstancesByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドインスタンスを起動します。
	 * エージェントが自動起動可能なノードは、<br>
	 * クラウドインスタンス作成時に{@link #addInstanceByTemplate(CreateInstanceByTemplateRequest request) テンプレートを指定した場合、｝
	 * クラウドインスタンス起動後に、テンプレートに既定にて指定されているテンプレートジョブが実施されます。
	 * また、テンプレートに指定されている既定のテンプレートジョブを切り替えたり、テンプレートが指定されていないエージェントでもテンプレートジョブを指定して起動することもできます。
	 * 
	 * @param request クラウドインスタンス起動情報。
	 * @return 起動後のクラウドインスタンスの状態情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	InstanceStateChange startInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") StartInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドインスタンスを停止します。
	 * クラウドインスタンス作成時に{@link #addInstanceByTemplate(CreateInstanceByTemplateRequest request) テンプレートを指定した場合、｝
	 * クラウドインスタンス停止前に、テンプレートに既定にて指定されているテンプレートジョブが実施されます。
	 * また、テンプレートに指定されている既定のテンプレートジョブを切り替えたり、テンプレートが指定されていないエージェントでもテンプレートジョブを指定して停止することもできます。
	 * 
	 * @param request クラウドインスタンス停止情報。
	 * @return 停止後のクラウドインスタンスの状態情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	InstanceStateChange stopInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") StopInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドユーザ(AWS アカウント、IMA ユーザ)を追加します。
	 * クラウドユーザと Hinemos ユーザは一対一の関係となり、クラウドユーザに紐づいていない Hinemos ユーザが存在しない場合は、例外が発生します。
	 * なお、クラウドユーザと紐づいた Hinemos ユーザーは、クラウドユーザを削除しないと削除することができません。
	 * 
	 * @param cloudUser クラウドユーザー情報。
	 * @return 作成したクラウドユーザーの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudUser addCloudUser(@NotNull @Into @WebParam(name = "request") CreateCloudUserRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザ(AWS アカウント、IMA ユーザ)を取得します。
	 * 
	 * @param userId ユーザーId。
	 * @return クラウドユーザー情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudUser getCloudUser(@WebParam(name = "userId") String userId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザ(AWS アカウント、IMA ユーザ)を削除します。
	 * 削除対象のクラウドユーザが課金アラートを作成している場合は、クラウドユーザを削除するには、作成された課金アラートの削除が必要になります。
	 * 
	 * @param userId ユーザーId。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeCloudUser(@WebParam(name = "userId") String userId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 全てのクラウドユーザ(AWS アカウント、IMA ユーザ)を取得します。
	 * 
	 * @return クラウドユーザー情報一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudUser> getCloudUsersByAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザ(AWS アカウント、IMA ユーザ)の設定を変更します。
	 * クラウドユーザID 以外は、null 指定が可能となり、値が指定された項目のみ更新されます。
	 * 
	 * @param cloudUser クラウドユーザー情報。
	 * @return クラウドユーザー情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudUser modifyCloudUser(@WebParam(name = "request") ModifyCloudUserRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドタイプを取得します。
	 * 現在は、AWS のみ返します。
	 * 
	 * @return クラウドタイプ種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudType> getAllSupportedCloudTypes() throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * AWS の全てのリージョンを返します。
	 * 
	 * @return リージョン情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudRegion> getRegionsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに所属するアベイラビリティゾーンを返します。
	 * 
	 * @param regionName リージョン名。
	 * @return アベイラビリティゾーン
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<Zone> getZones(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンおよびフィルター条件に該当する AMI イメージを返します。
	 * フィルターの設定方法は、以下のリンクを参考にしてください。
	 * <a http://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeImages.html /> 
	 * 
	 * @param regionName リージョン名。
	 * @param filter フィルター情報。
	 * @return AMI イメージ。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<Image> getImagesWithFilter(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "filters") List<Filter> filters) throws CloudManagerFault, InvalidUserPass, InvalidRole;

//	/**
//	 * 指定したリージョンおよびフィルター条件に該当する AMI イメージを返します。
//	 * フィルターの設定方法は、以下のリンクを参考にしてください。
//	 * <a http://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeImages.html /> 
//	 * 
//	 * @param regionName リージョン名。
//	 * @param filter フィルター情報。
//	 * @return AMI イメージ。
//	 * @throws CloudManagerFault
//	 * @throws InvalidUserPass
//	 * @throws InvalidRole
//	 * @throws ValidationFault
//	 */
//	@WebMethod
//	List<Snapshot> getAllImagesByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 選択可能なインスタンス種別の一覧を取得します。
	 * 
	 * @return インスタンス種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<String> getInstanceFlavorsByService(@WebParam(name = "serviceId") String serviceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンおよびフィルターに該当するスナップショットの一覧を取得します。
	 * フィルターの設定方法は、以下のリンクを参考にしてください。
	 * <a http://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeSnapshots.html /> 
	 * 
	 * @param regionName リージョン名。
	 * @param filter フィルター一覧。
	 * @return スナップショット。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<Snapshot> getSnapshotsWithFilter(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "filter") List<Filter> filter) throws CloudManagerFault, InvalidUserPass, InvalidRole;

//	/**
//	 * 指定したリージョンおよびフィルターに該当するスナップショットの一覧を取得します。
//	 * フィルターの設定方法は、以下のリンクを参考にしてください。
//	 * <a http://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeSnapshots.html /> 
//	 * 
//	 * @param regionName リージョン名。
//	 * @param filter フィルター一覧。
//	 * @return スナップショット。
//	 * @throws CloudManagerFault
//	 * @throws InvalidUserPass
//	 * @throws InvalidRole
//	 * @throws ValidationFault
//	 */
//	@WebMethod
//	List<Snapshot> getAllSnapshotsByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したインスタンスおよびストレージをアタッチします。
	 * アタッチが完了すると　EC2 インスタンスへ EBS ボリュームがアタッチされ、該当するノードへは、<br>
	 * EBS ボリュームの情報が反映されたディスク情報を追加されます。 
	 * 
	 * @param regionName リージョン名。
	 * @param instanceId インスタンス Id。
	 * @param storageId ストレージ Id。
	 * @param device デバイス名。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void attachStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId, @WebParam(name = "storageId") String storageId, @WebParam(name = "device") String device) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * 指定したインスタンスからストレージをデタッチします。
	 * アタッチが完了すると　EC2 インスタンスへ EBS ボリュームがアタッチされ、該当するノードへは、<br>
	 * EBS ボリュームの情報が反映されたディスク情報を追加されます。 
	 * 
	 * @param regionName リージョン名。
	 * @param instanceId インスタンス Id。
	 * @param storageId ストレージ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void detachStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId, @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * ストレージを追加します。
	 * 
	 * @param request 
	 * @return
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudStorage addStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateStorageRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * 作成済みのストレージバックアップ情報から、ストレージを復元します。
	 * ストレージバックアップ情報は、{@link #addStorageBackup(CreateStorageBackupRequest request) addStorageBackup} にて作成します。
	 * 
	 * @param request
	 * @return
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudStorage restoreStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RestoreStorageRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージを削除します。
	 * 
	 * @param regionName リージョン名。
	 * @param storageId ストレージ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージを取得します。
	 * 
	 * @param regionName リージョン名。
	 * @param storageId ストレージ Id。
	 * @return ストレージ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudStorage getStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region,  @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンに登録されているストレージの一覧を取得します。
	 * 
	 * @param regionName リージョン名。
	 * @return ストレージ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<CloudStorage> getStoragesByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 選択可能なストレージ種別の一覧を取得します。
	 * 
	 * @return ストレージ種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<String> getStorageFlavorsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージのバックアップ情報を取得します。
	 * 
	 * @param request ストレージバックアップ作成情報。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudStorageBackup addStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateStorageBackupRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージのバックアップ情報を削除します。
	 * 
	 * @param regionName リージョン名。
	 * @param snapshotId スナップショット Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageBackupId") String storageBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したストレージバックアップの情報を取得します。
	 * 
	 * @param regionName リージョン名。
	 * @param snapshotId スナップショット Id。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudStorageBackup getStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageBackupId") String storageBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに登録されているストレージバックアップ情報の一覧を取得します。
	 * 
	 * @param regionName リージョン名。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<CloudStorageBackup> getStorageBackupsByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したインスタンスのバックアップ情報を取得します。
	 * 
	 * @param request クラウドインスタンスバックアップ作成情報。
	 * @return クラウドインスタンスバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstanceBackup addInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceBackupRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したインスタンスのバックアップ情報を削除します。
	 * 
	 * @param regionName リージョン名。
	 * @param imageId イメージ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceBackupId") String instanceBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したインスタンスのバックアップ情報を取得します。
	 * 
	 * @param regionName リージョン名。
	 * @param imageId イメージ Id。
	 * @return クラウドインスタンスバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudInstanceBackup getInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceBackupId") String instanceBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに登録されているインスタンスバックアップ情報のリストを取得します。
	 * 
	 * @param regionName リージョン名。
	 * @return クラウドインスタンスバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	List<CloudInstanceBackup> getInstanceBackupsByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * ノードと紐づいていないインスタンスにノードを紐づけます。
	 * 
	 * @param request ノード登録情報。
	 * @return クラウドインスタンス情報。
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 * @throws CloudManagerFault
	 */
	@WebMethod
	CloudInstance registNode(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RegistNodeRequest request) throws InvalidUserPass, InvalidRole, CloudManagerFault;

	/**
	 * ノードに紐づいたインスタンスで、既に EC2 インスタンスが削除されているインスタンスを削除します。
	 * 
	 * @param regionName リージョン名。
	 * @param instanceId インスタンス Id。
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 * @throws CloudManagerFault
	 */
	@WebMethod
	void unregistNode(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws InvalidUserPass, InvalidRole, CloudManagerFault;
	
	/**
	 * テンプレートを追加します。
	 * 
	 * @param template テンプレート情報。
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudTemplate addTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "request") CreateTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したテンプレートを削除します。
	 * 
	 * @param templateId テンプレート Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	void removeTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "templateId") String templateId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したテンプレートを更新します。
	 * 
	 * @param template テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudTemplate modifyTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "request") ModifyTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したテンプレートを取得します。
	 * 
	 * @param templateId テンプレート Id。
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws ValidationFault
	 */
	@WebMethod
	CloudTemplate getTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "templateId") String templateId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 実行ユーザが参照可能な全てのテンプレートを取得します。
	 * 
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudTemplate> getTemplates(@WebParam(name = "roleId") String roleId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * テンプレートに設定可能なスクリプトの一覧を取得します。
	 * 
	 * @return スクリプト。
	 * @throws CloudManagerFault
	 */
	@WebMethod
	List<String> getTemplateScripts(@WebParam(name = "roleId") String roleId) throws CloudManagerFault;

	/**
	 * クラウドリソースグループを作成します。。
	 * 
	 * @param request クラウドリソースグループ情報。
	 * @return 作成されたアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource addAccountResource(@WebParam(name = "request") CreateAccountResourceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドリソースグループを変更します。
	 * 
	 * @param request クラウドリソースグループ情報。
	 * @return 変更後のアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource modifyAccountResource(@WebParam(name = "request") ModifyAccountResourceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドリソースグループを変更します。
	 * 
	 * @param request クラウドリソースグループ情報。
	 * @return 変更後のアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource modifyAccountBillingDetail(@WebParam(name = "request") ModifyAccountBillingDetailRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドリソースグループを削除します。
	 * 
	 * @param cloudResourceGroupId クラウドリソースグループ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドリソースグループを取得します。
	 * 
	 * @param cloudResourceGroupId クラウドリソースグループ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource getAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * アカウントリソースの一覧を取得します。
	 * 
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudAccountResource> getAllAccountResources() throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドサービスに紐付くリージョンの一覧を取得します。
	 * 
	 * @param cloudServiceId クラウドサービス Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudRegion> getCloudRegionsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドサービスを取得します。
	 * 
	 * @param cloudServiceId クラウドサービス Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudService getCloudService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドサービスの一覧を取得します。
	 * 
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudService> getAllCloudServices() throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	@WebMethod
	List<String> getCloudServicesForBilling(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	@WebMethod
	void establishAgentConnection(@WebParam(name = "facilityId") String facilityId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	@WebMethod
	List<AvailableRole> getAvailableRoles() throws CloudManagerFault, InvalidUserPass, InvalidRole;
}