/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.JudgmentObjectConstant;
import com.clustercontrol.jobmanagement.model.JobEndMstEntity;
import com.clustercontrol.jobmanagement.model.JobMstEntity;
import com.clustercontrol.jobmanagement.model.JobNoticeMstEntity;
import com.clustercontrol.jobmanagement.model.JobParamMstEntity;
import com.clustercontrol.jobmanagement.model.JobStartJobMstEntity;
import com.clustercontrol.jobmanagement.util.QueryUtil;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.session.NotifyControllerBean;
import com.clustercontrol.repository.session.RepositoryControllerBean;

/**
 * ジョブ情報を検索するクラスです。
 *
 * @version 4.1.0
 * @since 1.0.0
 */
public class FullJob {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( FullJob.class );

	/**
	 * 高速化のためのキャッシュ
	 * <jobunitId, <jobId, jobInfo>>
	 */
	private static ConcurrentHashMap<String, ConcurrentHashMap<String, JobInfo>>fullJobCache =
			new ConcurrentHashMap<String, ConcurrentHashMap<String, JobInfo>>();
	private static Object fullJobObject = new Object();

	/**
	 * コンストラクタ
	 * <P>
	 * ジョブ管理プロパティファイルから、ファイル転送ジョブ展開表示設定を取得します。
	 */
	public FullJob(){

	}

	public static void clearCache() {
		synchronized(fullJobObject) {
			m_log.info("clearCache()");
			fullJobCache.clear();
		}
	}

	// deleteJobunitから呼ばれる
	public static void removeCache(String jobunitId) {
		synchronized(fullJobObject) {
			m_log.debug("removeCache " + jobunitId);
			fullJobCache.remove(jobunitId);
		}
	}

	// replaceJobunitから呼ばれる。
	public static void removeCache(String jobunitId, String jobId) {
		synchronized(fullJobObject) {
			ConcurrentHashMap<String, JobInfo> jobunitMap = fullJobCache.get(jobunitId);
			if (jobunitMap == null) {
				return;
			}
			m_log.debug("removeCache " + jobunitId + ", " + jobId);
			jobunitMap.remove(jobId);
		}
	}

	/**
	 * ジョブ情報{@link com.clustercontrol.jobmanagement.bean.JobInfo}を作成します。<BR>
	 * ジョブマスタを基に、ジョブ情報を作成します。
	 * 
	 * @param job ジョブマスタ
	 * @param treeOnly treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @return ジョブ情報
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws UserNotFound
	 * @throws InvalidRole
	 */
	public static JobInfo getJobFull(JobInfo info) throws HinemosUnknown, JobMasterNotFound, UserNotFound, InvalidRole {

		m_log.debug("createJobData() : " + info.getJobunitId() + ", " + info.getId() + "," + info.isPropertyFull());
		if (info.isPropertyFull()) {
			return info;
		}

		String jobunitId = info.getJobunitId();
		String jobId = info.getId();

		synchronized(fullJobObject) {
			//cacheを確認
			ConcurrentHashMap<String, JobInfo> jobunitMap = fullJobCache.get(jobunitId);
			if (jobunitMap != null) {
				JobInfo ret = jobunitMap.get(jobId);
				if (ret != null) {
					m_log.debug("cache hit " + jobunitId + "," + jobId +
							", hit=" + jobunitMap.size());
					return ret;
				}
			} else {
				jobunitMap = new ConcurrentHashMap<String, JobInfo>();
				fullJobCache.put(jobunitId, jobunitMap);
			}
			jobunitMap.put(jobId, info);

			//JobInfoを作成
			//JobInfo info = new JobInfo(jobInfo.getJobunitId(), jobInfo.getId(), job.getJob_name(), job.getJob_type());
			m_log.debug("createJobData() : " + info.getJobunitId() + ", " + info.getId());
			JobMstEntity job = QueryUtil.getJobMstPK(info.getJobunitId(), info.getId());

			info.setDescription(job.getDescription());
			info.setOwnerRoleId(job.getOwnerRoleId());

			if (job.getRegDate() != null) {
				info.setCreateTime(job.getRegDate().getTime());
			}
			if (job.getUpdateDate() != null) {
				info.setUpdateTime(job.getUpdateDate().getTime());
			}
			info.setCreateUser(job.getRegUser());
			info.setUpdateUser(job.getUpdateUser());

			//待ち条件を取得
			JobWaitRuleInfo waitRule = null;
			//待ち条件を取得
			waitRule = new JobWaitRuleInfo();
			//ジョブネット・ジョブ・ファイル転送ジョブの場合
			//待ち条件を設定
			if(job.getJobType() == JobConstant.TYPE_JOBNET ||
					job.getJobType() == JobConstant.TYPE_JOB||
					job.getJobType() == JobConstant.TYPE_FILEJOB ||
					job.getJobType() == JobConstant.TYPE_REFERJOB){
				waitRule.setSuspend(job.getSuspend());
				waitRule.setCondition(job.getConditionType());
				waitRule.setEndCondition(job.getUnmatchEndFlg());
				waitRule.setEndStatus(job.getUnmatchEndStatus());
				waitRule.setEndValue(job.getUnmatchEndValue());
				waitRule.setSkip(job.getSkip());
				waitRule.setSkipEndStatus(job.getSkipEndStatus());
				waitRule.setSkipEndValue(job.getSkipEndValue());
				waitRule.setCalendar(job.getCalendar());
				waitRule.setCalendarId(job.getCalendarId());
				waitRule.setCalendarEndStatus(job.getCalendarEndStatus());
				waitRule.setCalendarEndValue(job.getCalendarEndValue());

				waitRule.setStart_delay(job.getStartDelay());
				waitRule.setStart_delay_session(job.getStartDelaySession());
				waitRule.setStart_delay_session_value(job.getStartDelaySessionValue());
				waitRule.setStart_delay_time(job.getStartDelayTime());
				if (job.getStartDelayTimeValue() != null) {
					waitRule.setStart_delay_time_value(job.getStartDelayTimeValue().getTime());
				}
				waitRule.setStart_delay_condition_type(job.getStartDelayConditionType());
				waitRule.setStart_delay_notify(job.getStartDelayNotify());
				waitRule.setStart_delay_notify_priority(job.getStartDelayNotifyPriority());
				waitRule.setStart_delay_operation(job.getStartDelayOperation());
				waitRule.setStart_delay_operation_type(job.getStartDelayOperationType());
				waitRule.setStart_delay_operation_end_status(job.getStartDelayOperationEndStatus());
				waitRule.setStart_delay_operation_end_value(job.getStartDelayOperationEndValue());

				waitRule.setEnd_delay(job.getEndDelay());
				waitRule.setEnd_delay_session(job.getEndDelaySession());
				waitRule.setEnd_delay_session_value(job.getEndDelaySessionValue());
				waitRule.setEnd_delay_job(job.getEndDelayJob());
				waitRule.setEnd_delay_job_value(job.getEndDelayJobValue());
				waitRule.setEnd_delay_time(job.getEndDelayTime());
				if (job.getEndDelayTimeValue() != null) {
					waitRule.setEnd_delay_time_value(job.getEndDelayTimeValue().getTime());
				}
				waitRule.setEnd_delay_condition_type(job.getEndDelayConditionType());
				waitRule.setEnd_delay_notify(job.getEndDelayNotify());
				waitRule.setEnd_delay_notify_priority(job.getEndDelayNotifyPriority());
				waitRule.setEnd_delay_operation(job.getEndDelayOperation());
				waitRule.setEnd_delay_operation_type(job.getEndDelayOperationType());
				waitRule.setEnd_delay_operation_end_status(job.getEndDelayOperationEndStatus());
				waitRule.setEnd_delay_operation_end_value(job.getEndDelayOperationEndValue());
				waitRule.setMultiplicityNotify(job.getMultiplicityNotify());
				waitRule.setMultiplicityNotifyPriority(job.getMultiplicityNotifyPriority());
				waitRule.setMultiplicityOperation(job.getMultiplicityOperation());
				waitRule.setMultiplicityEndValue(job.getMultiplicityEndValue());
			}

			//待ち条件（ジョブ）を取得
			Collection<JobStartJobMstEntity> startJobList = job.getJobStartJobMstEntities();
			ArrayList<JobObjectInfo> objectList = new ArrayList<JobObjectInfo>();
			if(startJobList != null && startJobList.size() > 0){
				for (JobStartJobMstEntity startJob : startJobList){
					if(startJob != null){
						JobObjectInfo objectInfo = new JobObjectInfo();
						objectInfo.setJobId(startJob.getId().getTargetJobId());
						//対象ジョブを取得
						JobMstEntity targetJob = QueryUtil.getJobMstPK(startJob.getId().getTargetJobunitId(), startJob.getId().getTargetJobId());
						objectInfo.setJobName(targetJob.getJobName());
						objectInfo.setType(startJob.getId().getTargetJobType());
						objectInfo.setValue(startJob.getId().getTargetJobEndValue());
						objectList.add(objectInfo);
					}
				}
			}

			//待ち条件（時刻）を取得
			if (job.getStartTime() != null) {
				JobObjectInfo objectInfo = new JobObjectInfo();
				objectInfo.setType(JudgmentObjectConstant.TYPE_TIME);
				objectInfo.setTime(job.getStartTime().getTime());
				objectList.add(objectInfo);
			}
			/*
			 * ソート処理
			 */
			Collections.sort(objectList);
			waitRule.setObject(objectList);
			info.setWaitRule(waitRule);

			//実行コマンドを取得
			if(job.getJobType() == JobConstant.TYPE_JOB){
				JobCommandInfo commandInfo = new JobCommandInfo();
				commandInfo.setFacilityID(job.getFacilityId());
				commandInfo.setProcessingMethod(job.getProcessMode());
				commandInfo.setStartCommand(job.getStartCommand());
				commandInfo.setStopType(job.getStopType());
				commandInfo.setStopCommand(job.getStopCommand());
				commandInfo.setSpecifyUser(job.getSpecifyUser());
				commandInfo.setUser(job.getEffectiveUser());
				commandInfo.setMessageRetry(job.getMessageRetry());
				commandInfo.setErrorEndFlg(job.getErrorEndFlg());
				commandInfo.setErrorEndValue(job.getErrorEndValue());
				//ファシリティパスを取得
				commandInfo.setScope(new RepositoryControllerBean().getFacilityPath(job.getFacilityId(), null));
				info.setCommand(commandInfo);
			}

			//ファイル転送を取得
			if(job.getJobType() == JobConstant.TYPE_FILEJOB){
				JobFileInfo fileInfo = new JobFileInfo();
				fileInfo.setProcessingMethod(job.getProcessMode());
				fileInfo.setSrcFacilityID(job.getSrcFacilityId());
				fileInfo.setDestFacilityID(job.getDestFacilityId());
				fileInfo.setSrcFile(job.getSrcFile());
				fileInfo.setSrcWorkDir(job.getSrcWorkDir());
				fileInfo.setDestDirectory(job.getDestDirectory());
				fileInfo.setDestWorkDir(job.getDestWorkDir());
				fileInfo.setCompressionFlg(job.getCompressionFlg());
				fileInfo.setCheckFlg(job.getCheckFlg());
				fileInfo.setSpecifyUser(job.getSpecifyUser());
				fileInfo.setUser(job.getEffectiveUser());
				fileInfo.setMessageRetry(job.getMessageRetry());
				//ファシリティパスを取得
				fileInfo.setSrcScope(new RepositoryControllerBean().getFacilityPath(job.getSrcFacilityId(), null));
				fileInfo.setDestScope(new RepositoryControllerBean().getFacilityPath(job.getDestFacilityId(), null));
				info.setFile(fileInfo);
			}
			if (info.getType() != JobConstant.TYPE_REFERJOB) {
				//通知メッセージを取得
				ArrayList<JobNotificationsInfo> noticeList = new ArrayList<JobNotificationsInfo>();
				Collection<JobNoticeMstEntity> notices = job.getJobNoticeMstEntities();
				if(notices != null){
					for(JobNoticeMstEntity notice : notices) {
						JobNotificationsInfo notificationsInfo = new JobNotificationsInfo();
						notificationsInfo.setType(notice.getId().getNoticeType());
						notificationsInfo.setPriority(notice.getPriority());
						notificationsInfo.setNotifyGroupId(notice.getNotifyGroupId());

						//正常終了の場合のみ通知をセット
						if(notificationsInfo.getType() == EndStatusConstant.TYPE_NORMAL ){
							//通知情報の取得
							ArrayList<NotifyRelationInfo> nriList = new ArrayList<NotifyRelationInfo>();
							nriList = new NotifyControllerBean().getNotifyRelation(notice.getNotifyGroupId());
							if (nriList != null) {
								/*
								 * ソート処理
								 */
								Collections.sort(nriList);
								notificationsInfo.setNotifyId(nriList);
							}
							//notificationsInfo.setNotifyId(new NotifyControllerBean().getNotifyRelation(notice.getNotifyGroupId()));
						}
						noticeList.add(notificationsInfo);
					}
					/*
					 * ソート処理
					 */
					Collections.sort(noticeList);
				}
				info.setNotifications(noticeList);

				//終了状態を取得
				ArrayList<JobEndStatusInfo> endList = new ArrayList<JobEndStatusInfo>();
				Collection<JobEndMstEntity> ends = job.getJobEndMstEntities();
				if(ends != null){
					for (JobEndMstEntity end : ends) {
						JobEndStatusInfo endInfo = new JobEndStatusInfo();
						endInfo.setType(end.getId().getEndStatus());
						endInfo.setValue(end.getEndValue());
						endInfo.setStartRangeValue(end.getEndValueFrom());
						endInfo.setEndRangeValue(end.getEndValueTo());
						endList.add(endInfo);
					}
					/*
					 * ソート処理
					 */
					Collections.sort(endList);
				}
				info.setEndStatus(endList);
			}

			//パラメータを取得
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				ArrayList<JobParameterInfo> paramList = new ArrayList<JobParameterInfo>();
				Collection<JobParamMstEntity> params = job.getJobParamMstEntities();
				if(params != null){
					for (JobParamMstEntity param : params) {
						JobParameterInfo paramInfo = new JobParameterInfo();
						paramInfo.setParamId(param.getId().getParamId());
						paramInfo.setType(param.getParamType());
						paramInfo.setDescription(param.getDescription());
						paramInfo.setValue(param.getValue());
						paramList.add(paramInfo);
					}
					/*
					 * ソート処理
					 */
					Collections.sort(paramList);
				}
				info.setParam(paramList);
			}
			//参照ジョブ
			if(info.getType() == JobConstant.TYPE_REFERJOB){
				info.setReferJobUnitId(job.getReferJobUnitId());
				info.setReferJobId(job.getReferJobId());
			}

			info.setPropertyFull(true);
		}
		return info;
	}

	/**
	 * ジョブツリー配下のジョブを全てfullProperty=trueにする。
	 * @param item
	 * @throws JobMasterNotFound
	 * @throws NotifyNotFound
	 * @throws UserNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public static void setJobTreeFull (JobTreeItem item)
			throws JobMasterNotFound, NotifyNotFound, UserNotFound, InvalidRole, HinemosUnknown {
		JobInfo jobInfo = item.getData();
		if (!jobInfo.isPropertyFull()) {
			item.setData(getJobFull(item.getData()));
		}
		if (item.getChildren() == null) {
			return;
		}
		for (JobTreeItem child : item.getChildren()) {
			setJobTreeFull(child);
		}
	}
}
