package com.clustercontrol.notify.util;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.model.NotifyCommandInfoEntity;
import com.clustercontrol.notify.model.NotifyEventInfoEntity;
import com.clustercontrol.notify.model.NotifyInfoEntity;
import com.clustercontrol.notify.model.NotifyJobInfoEntity;
import com.clustercontrol.notify.model.NotifyLogEscalateInfoEntity;
import com.clustercontrol.notify.model.NotifyMailInfoEntity;
import com.clustercontrol.notify.model.NotifyStatusInfoEntity;

public class NotifyCache {
	private static Log m_log = LogFactory.getLog( NotifyCache.class );

	private static Object notifyMapLock = new Object();
	private static ConcurrentHashMap<String, NotifyInfo> notifyMap = null;
	private static ConcurrentHashMap<String, NotifyInfoDetail> notifyDetailMap = null;

	/**
	 * キャッシュをリフレッシュする。
	 * 通知の登録、変更、削除時に呼ぶ。
	 */
	public static void refresh() {
		synchronized (notifyMapLock) {
			long start = System.currentTimeMillis();
			notifyMap = new ConcurrentHashMap<String, NotifyInfo>();
			notifyDetailMap = new ConcurrentHashMap<String, NotifyInfoDetail>();
			try {
				List<NotifyInfoEntity> c = QueryUtil.getAllNotifyInfo();
				for (NotifyInfoEntity entity : c) {
					NotifyInfo info = new NotifyInfo();
					String notifyId = entity.getNotifyId();
					info.setNotifyId(notifyId);
					// info.setDescription(local.getDescription());
					info.setNotifyType(entity.getNotifyType());
					info.setInitialCount(entity.getInitialCount());
					info.setRenotifyType(entity.getRenotifyType());
					info.setRenotifyPeriod(entity.getRenotifyPeriod());
					// info.setRegDate(local.getRegDate().getTime());
					// info.setUpdateDate(local.getUpdateDate().getTime());
					// info.setRegUser(local.getRegUser());
					// info.setUpdateUser(local.getUpdateUser());
					info.setValidFlg(entity.getValidFlg());
					notifyMap.put(notifyId, info);

					m_log.debug("refresh() notifyInfo(notifyId=" + notifyId + ") cache is refreshed");

					for (int priority : PriorityConstant.PRIORITY_LIST) {
						NotifyInfoDetail notifyInfoDetail = new NotifyInfoDetail();
						notifyInfoDetail.setNotifyId(notifyId);
						notifyInfoDetail.setPriority(priority);
						notifyInfoDetail.setValidFlg(getValidFlg(entity, priority));
						notifyDetailMap.put(createKey(notifyId, priority), notifyInfoDetail);
					}
				}
			} catch (Exception e) {
				m_log.warn("refresh() create notifyCache failed : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			}
			m_log.info("refresh NotifyCache " + (System.currentTimeMillis() - start) + "ms. size=" + notifyDetailMap.size());
		}
	}

	private static void init () {
		if (notifyMap == null) {
			refresh();
		}
	}

	/**
	 * NotifyInfoをキャッシュから取得する。
	 * NotifyInfoは性能向上のため、メンバ変数の一部が欠けているため注意すること。
	 * @param notifyId
	 * @return
	 */
	public static NotifyInfo getNotifyInfo(String notifyId) {
		init ();
		NotifyInfo notifyInfo = null;
		synchronized (notifyMapLock) {
			notifyInfo = notifyMap.get(notifyId);
		}
		if (notifyInfo == null) {
			m_log.info("getNotifyInfo() notifyInfo (notifyId=" + notifyId + ") is null");
			m_log.debug("getNotifyInfo() notifyInfo map size=" + notifyMap.size());
		}
		return notifyInfo;
	}

	/**
	 * NotifyInfoDetailをキャッシュから取得する。
	 * @param notifyId
	 * @param priority
	 * @return
	 */
	public static NotifyInfoDetail getNotifyInfoDetail(String notifyId, int priority) {
		init ();
		NotifyInfoDetail notifyInfoDetail = null;
		synchronized (notifyMapLock) {
			notifyInfoDetail = notifyDetailMap.get(createKey(notifyId, priority));
		}
		if (notifyInfoDetail == null) {
			m_log.info("getNotifyInfoDetail() notifyInfoDetail is null");
		}
		return notifyInfoDetail;
	}

	private static String createKey (String notifyId, int priority) {
		return notifyId + "," + priority;
	}

	/**
	 * 重要度毎の有効無効フラグを返します。
	 * 
	 * @return Integer 有効無効フラグ
	 */
	private static Integer getValidFlg(NotifyInfoEntity entity, int priority) {
		m_log.debug("getNotifyDetail() " + entity.getNotifyId() + ",  priority = " + priority);

		switch (entity.getNotifyType()) {
		case NotifyTypeConstant.TYPE_EVENT:
			Collection<NotifyEventInfoEntity> events = entity.getNotifyEventInfoEntities();
			for(NotifyEventInfoEntity event : events){
				if(event.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return event.getEventNormalFlg();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + events.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		case NotifyTypeConstant.TYPE_STATUS:
			Collection<NotifyStatusInfoEntity> statuses = entity.getNotifyStatusInfoEntities();
			for(NotifyStatusInfoEntity status : statuses){
				if(status.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return status.getStatusFlg();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + statuses.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		case NotifyTypeConstant.TYPE_MAIL:
			Collection<NotifyMailInfoEntity> mails = entity.getNotifyMailInfoEntities();
			for(NotifyMailInfoEntity mail : mails){
				if(mail.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return mail.getMailFlg();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + mails.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		case NotifyTypeConstant.TYPE_JOB:
			Collection<NotifyJobInfoEntity> jobs = entity.getNotifyJobInfoEntities();
			for(NotifyJobInfoEntity job : jobs){
				if(job.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return job.getJobRun();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + jobs.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		case NotifyTypeConstant.TYPE_LOG_ESCALATE:
			Collection<NotifyLogEscalateInfoEntity> logs = entity.getNotifyLogEscalateInfoEntities();
			for(NotifyLogEscalateInfoEntity log : logs){
				if(log.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return log.getEscalateFlg();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + logs.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		case NotifyTypeConstant.TYPE_COMMAND:
			Collection<NotifyCommandInfoEntity> commands = entity.getNotifyCommandInfoEntities();
			for(NotifyCommandInfoEntity command : commands){
				if(command.getId().getPriority().intValue() == priority){
					m_log.debug(entity.getNotifyId() + ", notify type = " + entity.getNotifyType());
					return command.getValidFlg();
				}
			}
			m_log.info("getValidFlg() incorrect data. size = " + commands.size() + "  " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			break;
		default:
			m_log.info("getValidFlg() notify type mismatch. " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
			return null;
		}
		m_log.info("getValidFlg() incorrect data. " + entity.getNotifyId() + "  type = " + entity.getNotifyType());
		return null;
	}
}
