/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.plugin.impl;

import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.plugin.api.HinemosPlugin;
import com.clustercontrol.systemlog.bean.SyslogMessage;
import com.clustercontrol.systemlog.service.SystemLogMonitor;
import com.clustercontrol.systemlog.util.SyslogReceiver;

/**
 * システムログ監視の初期化・終了処理(udp:24514の待ち受け開始)を制御するプラグイン.
 * 
 * @author takahatat
 */
public class SystemLogPlugin implements HinemosPlugin {

	public static final Log log = LogFactory.getLog(SystemLogPlugin.class);

	/** syslogの待ち受けポートを設けるかどうかを制御するパラメータ(クラスタ構成用) */
	public static final String _keyNoListen = "monitor.systemlog.nolisten";
	public static final boolean _noListenDefault = false;
	public static final boolean _noListen;

	/** syslogの待ち受けアドレス */
	public static final String _keyListenAddress = "monitor.systemlog.listen.address";
	public static final String _listenAddressDefault = "0.0.0.0";
	public static final String _listenAddress;

	/** syslogの待ち受けポート番号 */
	public static final String _keyListenPort = "monitor.systemlog.listen.port";
	public static final int _listenPortDefault = 24514;
	public static final int _listenPort;

	/** syslogのデフォルト文字コード */
	public static final String _keyCharset = "monitor.systemlog.charset";
	public static final Charset _charsetDefault = Charset.forName("UTF-8");
	public static final Charset _charset;

	/** 受信処理とフィルタリング処理の間に存在するsyslog処理待ちキューの最大サイズ*/
	public static final String _keyTaskQueueSize = "monitor.systemlog.filter.queue.size";
	public static final int _taskQueueSizeDefault = 15 * 60 * 30;	// 15[min] * 30[msg/sec] (about 27mbyte)
	public static final int _taskQueueSize;

	/** フィルタリング処理のスレッド数 */
	public static final String _keyTaskThreadSize = "monitor.systemlog.filter.thread.size";
	public static final int _taskThreadSizeDefault = 1;
	public static final int _taskThreadSize;

	/** syslog監視の受信クラス */
	private static SyslogReceiver _receiver;

	/** syslog監視のフィルタリングクラス */
	private static SystemLogMonitor _handler;

	static {
		_listenAddress = HinemosProperties.getProperty(_keyListenAddress, _listenAddressDefault);

		boolean noListen = _noListenDefault;
		String noListenStr = HinemosProperties.getProperty(_keyNoListen);
		try {
			noListen = Boolean.parseBoolean(noListenStr);
		} catch (Exception e) { }
		_noListen = noListen;

		int listenPort = _listenPortDefault;
		String listenPortStr = HinemosProperties.getProperty(_keyListenPort);
		try {
			listenPort = Integer.parseInt(listenPortStr);
		} catch (Exception e) { }
		_listenPort = listenPort;

		String charsetAll = "";
		for (String c : Charset.availableCharsets().keySet()) {
			charsetAll += c + ", ";
		}
		log.info("supported charset : " + charsetAll);

		Charset charset = _charsetDefault;
		String charsetStr = HinemosProperties.getProperty(_keyCharset);
		try {
			charset = Charset.forName(charsetStr);
		} catch (Exception e) { }
		_charset = charset;

		String taskQueueSizeStr = HinemosProperties.getProperty(_keyTaskQueueSize);
		int taskQueueSize = _taskQueueSizeDefault;
		try {
			if (taskQueueSizeStr != null) {
				taskQueueSize = Integer.parseInt(taskQueueSizeStr);
			}
		} catch (Exception e) { }
		_taskQueueSize = taskQueueSize;

		String taskThreadSizeStr = HinemosProperties.getProperty(_keyTaskThreadSize);
		int taskThreadSize = _taskThreadSizeDefault;
		try {
			if (taskThreadSizeStr != null) {
				taskThreadSize = Integer.parseInt(taskThreadSizeStr);
			}
		} catch (Exception e) { }
		_taskThreadSize = taskThreadSize;
	}

	@Override
	public Set<String> getDependency() {
		Set<String> dependency = new HashSet<String>();
		dependency.add(AsyncWorkerPlugin.class.getName());
		dependency.add(SharedTablePlugin.class.getName());
		return dependency;
	}

	@Override
	public void create() {
		log.info(String.format("starting SystemLogPlugin : listenAddress = %s, listenPort = %d, charset = %s, queueSize = %d, threads = %d",
				_listenAddress, _listenPort, _charset.name(), _taskQueueSize, _taskThreadSize));

		_handler = new SystemLogMonitor(_taskThreadSize, _taskQueueSize);
		_receiver = new SyslogReceiver(_listenAddress, _listenPort, _charset, _handler);

	}

	@Override
	public void activate() {
		try {
			if (_noListen) {
				log.info("activating SystemLogPlugin without receiver.");
				_handler.start();
			} else {
				_receiver.start();
			}
		} catch (Exception e) {
			log.warn("SystemLogPlugin activation failure.", e);
		}
	}

	@Override
	public void deactivate() {
		if (_noListen) {
			if (_handler != null) {
				_handler.shutdown();
			}
		} else {
			if (_receiver != null) {
				_receiver.shutdown();
			}
		}
	}

	@Override
	public void destroy() {

	}

	public static void syslogReceived(String receiverId, SyslogMessage syslog) {
		_handler.syslogReceived(receiverId, syslog);
	}

	public static long getReceivedCount() {
		return _handler.getReceivedCount();
	}

	public static long getNotifiedCount() {
		return _handler.getNotifiedCount();
	}

	public static long getDiscardedCount() {
		return _handler.getDiscardedCount();
	}

	public static int getQueuedCount() {
		return _handler.getQueuedCount();
	}



}
