/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck;

import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.bean.ThreadInfo;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.MonitoredThreadPoolExecutor;
import com.clustercontrol.plugin.impl.SchedulerInfo;
import com.clustercontrol.plugin.impl.SchedulerPlugin;
import com.clustercontrol.plugin.impl.SchedulerPlugin.SchedulerType;
import com.clustercontrol.selfcheck.monitor.AsyncTaskQueueMonitor;
import com.clustercontrol.selfcheck.monitor.TableSizeMonitor;
import com.clustercontrol.selfcheck.monitor.DatabaseMonitor;
import com.clustercontrol.selfcheck.monitor.FileSystemMonitor;
import com.clustercontrol.selfcheck.monitor.JVMHeapMonitor;
import com.clustercontrol.selfcheck.monitor.JobRunSessionMonitor;
import com.clustercontrol.selfcheck.monitor.SchedulerMonitor;
import com.clustercontrol.selfcheck.monitor.RAMSwapOutMonitor;
import com.clustercontrol.selfcheck.monitor.SnmpTrapQueueMonitor;
import com.clustercontrol.selfcheck.monitor.SyslogQueueMonitor;
import com.clustercontrol.selfcheck.monitor.ThreadActivityMonitor;
import com.clustercontrol.selfcheck.monitor.WebServiceQueueMonitor;

/**
 * セルフチェック機能の定期実行制御クラス
 * @author takahatat
 */
public class SelfCheckTaskSubmitter implements Runnable {

	private static Log log = LogFactory.getLog( SelfCheckTaskSubmitter.class );

	private final ScheduledExecutorService _scheduler;
	private final ExecutorService _executorService;
	private static final long _shutdownTimeoutMsec;

	static {
		_shutdownTimeoutMsec = Long.parseLong(HinemosProperties.getProperty("hinemos.selfcheck.shutdown.timeoue", "15000"));
	}

	public SelfCheckTaskSubmitter() {
		_scheduler = Executors.newScheduledThreadPool(1, new ThreadFactory() {

			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, "SelfCheckScheduler");
			}
		});

		_executorService = Executors.newFixedThreadPool(
				SelfCheckConfig.threadPoolSize,
				new ThreadFactory() {
					private volatile int _count = 0;

					@Override
					public Thread newThread(Runnable r) {
						return new Thread(r, "SelfCheckWorker-" + _count++);
					}
				}
				);
	}

	/**
	 * セルフチェック機能を活性化させるメソッド
	 */
	public void start() {
		_scheduler.scheduleWithFixedDelay(this, SelfCheckConfig.startupDelaySec, SelfCheckConfig.intervalSec, TimeUnit.SECONDS);
	}

	/**
	 * セルフチェック機能を非活性化させるメソッド
	 */
	public void shutdown() {
		// キック元となるスケジューラから停止していく
		_scheduler.shutdown();
		try {
			if (! _scheduler.awaitTermination(_shutdownTimeoutMsec, TimeUnit.MILLISECONDS)) {
				List<Runnable> remained = _scheduler.shutdownNow();
				if (remained != null) {
					log.info("shutdown timeout. runnable remained. (size = " + remained.size() + ")");
				}
			}
		} catch (InterruptedException e) {
			_scheduler.shutdownNow();
		}

		_executorService.shutdown();
		try {
			if (! _executorService.awaitTermination(_shutdownTimeoutMsec, TimeUnit.MILLISECONDS)) {
				_executorService.shutdownNow();
			}
		} catch (InterruptedException e) {
			_executorService.shutdownNow();
		}
	}

	/**
	 * 定期実行間隔(interval)に基づいて、定期的に実行されるメソッド
	 */
	@Override
	public void run() {
		/** メイン処理 */
		// Java VM Heap
		if (SelfCheckConfig.jvmHeapMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new JVMHeapMonitor(SelfCheckConfig.jvmHeapThresholdMByte)
							)
					);
		}

		// FileSystem
		if (SelfCheckConfig.fsUsageMonitoring) {
			for (FileSystemUsageConfig config : SelfCheckConfig.fsUsageList) {
				_executorService.submit(
						new SelfCheckTask(
								new FileSystemMonitor(
										config.mountPoint,
										config.percentThreshold,
										SelfCheckConfig.snmpPort,
										SelfCheckConfig.snmpVersion,
										SelfCheckConfig.snmpCommunity,
										SelfCheckConfig.snmpRetries,
										SelfCheckConfig.snmpTimeoutMSec
										)
								)
						);
			}
		}

		// swap-out
		if (SelfCheckConfig.swapOutMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new RAMSwapOutMonitor(
									SelfCheckConfig.intervalSec,
									SelfCheckConfig.snmpPort,
									SelfCheckConfig.snmpVersion,
									SelfCheckConfig.snmpCommunity,
									SelfCheckConfig.snmpRetries,
									SelfCheckConfig.snmpTimeoutMSec
									)
							)
					);
		}

		// Database
		if (SelfCheckConfig.dbMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new DatabaseMonitor(SelfCheckConfig.dbValidationQuery)
							)
					);
		}

		// Scheduler
		if (SelfCheckConfig.schedulerMonitoring) {
			List<SchedulerInfo> triggerList = null;
			try {
				triggerList = SchedulerPlugin.getSchedulerList(SchedulerType.DBMS);
			} catch (Exception e) {
				log.warn("quartz scheduler access failure. (" + SchedulerType.DBMS + ")", e);
			}

			for (SchedulerInfo trigger : triggerList) {
				if (! trigger.isPaused) {
					_executorService.submit(
							new SelfCheckTask(
									new SchedulerMonitor(
											SchedulerType.DBMS,
											trigger,
											SelfCheckConfig.schedulerDelayThresholdSec
											)
									)
							);
				}
			}

			try {
				triggerList = SchedulerPlugin.getSchedulerList(SchedulerType.RAM);
			} catch (Exception e) {
				log.warn("quartz scheduler access failure. (" + SchedulerType.RAM + ")", e);
			}

			for (SchedulerInfo trigger : triggerList) {
				if (! trigger.isPaused) {
					_executorService.submit(
							new SelfCheckTask(
									new SchedulerMonitor(
											SchedulerType.RAM,
											trigger,
											SelfCheckConfig.schedulerDelayThresholdSec
											)
									)
							);
				}
			}

		}

		// Web Service
		if (SelfCheckConfig.wsQueueMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new WebServiceQueueMonitor(
									SelfCheckConfig.wsQueueThreshold
									)
							)
					);
		}

		// syslog queue
		if (SelfCheckConfig.syslogQueueMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new SyslogQueueMonitor(
									SelfCheckConfig.syslogQueueThreshold
									)
							)
					);
		}

		// snmptrap queue
		if (SelfCheckConfig.snmptrapQueueMonitoring) {
			_executorService.submit(
					new SelfCheckTask(
							new SnmpTrapQueueMonitor(
									SelfCheckConfig.snmptrapQueueThreshold
									)
							)
					);
		}

		// asynchronous task queue
		if (SelfCheckConfig.asyncTaskMonitoring) {
			for (AsyncTaskQueueConfig config : SelfCheckConfig.asyncTaskList) {
				_executorService.submit(
						new SelfCheckTask(
								new AsyncTaskQueueMonitor(
										config.worker,
										config.queueThreshold
										)
								)
						);
			}
		}

		// thread activity
		if (SelfCheckConfig.threadActivityMonitoring) {
			for (ThreadInfo threadInfo : MonitoredThreadPoolExecutor.getRunningThreadMap().values()) {
				_executorService.submit(
						new SelfCheckTask(
								new ThreadActivityMonitor(
										threadInfo,
										SelfCheckConfig.threadRunningTimeThreshold
										)
								)
						);
			}

		}

		// log table
		if (SelfCheckConfig.tableSizeMonitoring) {
			for (TableSizeConfig config : SelfCheckConfig.tableList) {
				_executorService.submit(
						new SelfCheckTask(
								new TableSizeMonitor(
										config.tableName,
										config.threshold,
										config.thresdholdType
										)
								)
						);
			}
		}

		// job
		if(SelfCheckConfig.jobRunningSessionMonitoring){
			_executorService.submit(
					new SelfCheckTask(
							new JobRunSessionMonitor(
									SelfCheckConfig.jobRunningSessionThreshold
									)
							)
					);
		}

		// set timestamp of last monitoring
		SelfCheckConfig.setLastMonitorDate(new Date());
	}

}
