/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.dialog;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.accesscontrol.action.CheckPermission;
import com.clustercontrol.accesscontrol.bean.FunctionConstant;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.bean.PropertyDefineConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.FacilityDuplicate_Exception;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.RoleDuplicate_Exception;
import com.clustercontrol.ws.access.RoleInfo;
import com.clustercontrol.ws.access.UnEditableRole_Exception;

/**
 * アカウント[ロールの作成・変更]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 2.0.0
 */
public class RoleDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( RoleDialog.class );

	/** ロールID */
	private String roleId = "";

	/** モード */
	private int mode = 0;

	/** 変更用ダイアログ判別フラグ */
	private boolean isModifyDialog = false;

	/** ロールID　テキストボックス */
	private Text textRoleId = null;

	/** ロール名　テキストボックス */
	private Text textRoleName = null;

	/** 説明　テキストボックス */
	private Text textDescription = null;

	/** カラム数 */
	public static final int WIDTH	 = 15;

	/** カラム数（ラベル）。 */
	public static final int WIDTH_LABEL = 4;

	/** カラム数（テキスト）。 */
	public static final int WIDTH_TEXT = 10;

	/** 入力値を保持するオブジェクト。 */
	private RoleInfo inputData = null;

	private boolean permission = false;		// 現在のユーザが変更権限をもつか否か

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 * @param uid ユーザID
	 * @param isModifyDialog 変更用ダイアログとして利用する場合は、true
	 */
	public RoleDialog(Shell parent, String roleId, boolean isModifyDialog) {
		super(parent);

		this.roleId = roleId;
		this.isModifyDialog = isModifyDialog;

		// 現在のユーザがアカウント機能の設定権限をもつか否かを取得する
		permission = new CheckPermission().check(FunctionConstant.ACCESSCONTROL, SystemPrivilegeMode.WRITE);

	}

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 * 
	 * @see org.eclipse.jface.window.Window#getInitialSize()
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(600, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#customizeDialog(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages
				.getString("dialog.accesscontrol.role.create.modify"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * ロールID
		 */
		// ラベル
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = WIDTH_LABEL;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("role.id") + " : ");
		// テキスト
		this.textRoleId = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textRoleId.setLayoutData(gridData);
		this.textRoleId.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// プロパティの取得及び設定
		if (this.isModifyDialog) {
			if (permission) {
				this.mode = PropertyDefineConstant.MODE_MODIFY;
			}else{
				this.mode = PropertyDefineConstant.MODE_SHOW;
			}
		} else {
			this.mode = PropertyDefineConstant.MODE_ADD;
		}

		/*
		 * ロール名
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_LABEL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("role.name") + " : ");
		// テキスト
		this.textRoleName = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textRoleName.setLayoutData(gridData);
		this.textRoleName.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		/*
		 * 説明
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_LABEL;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textDescription.setLayoutData(gridData);
		this.textDescription.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(550, shell.getSize().y));

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		// ロールIDが指定されている場合、その情報を初期化する。
		RoleInfo info = null;
		if (this.roleId != null) {
			try {
				info = AccessEndpointWrapper.getRoleInfo(this.roleId);
			} catch (InvalidRole_Exception e) {
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} catch (Exception e) {
				m_log.warn("customizeDialog(), " + e.getMessage(), e);
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
			}
		} else {
			info = new RoleInfo();
		}
		this.setInputData(info);
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return null;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		RoleInfo roleInfo = this.inputData;
		if(roleInfo == null){
			return result;
		}

		if(!this.isModifyDialog){
			// 作成の場合
			try {
				AccessEndpointWrapper.addRoleInfo(roleInfo);
				result = true;

				// 完了メッセージ
				MessageDialog.openInformation(
						null,
						Messages.getString("successful"),
						Messages.getString("message.accesscontrol.26"));

			} catch (RoleDuplicate_Exception e) {
				//ロールID取得
				String args[] = { roleInfo.getId() };

				// ロールIDが重複している場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.accesscontrol.33", args));
			} catch (FacilityDuplicate_Exception e) {
				//ロールID取得
				String args[] = { roleInfo.getId() };

				// ロールIDが重複している場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.repository.26", args));
			} catch (Exception e) {
				String errMessage = "";
				if (e instanceof InvalidRole_Exception) {
					// 権限なし
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} else {
					errMessage = ", " + e.getMessage();
				}
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.27") + errMessage);

			}
		} else{
			// 変更の場合
			try {
				AccessEndpointWrapper.modifyRoleInfo(roleInfo);
				result = true;

				// 完了メッセージ
				MessageDialog.openInformation(
						null,
						Messages.getString("successful"),
						Messages.getString("message.accesscontrol.28"));

			} catch (Exception e) {
				String errMessage = "";
				if (e instanceof InvalidRole_Exception) {
					// 権限なし
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} else if (e instanceof UnEditableRole_Exception) {
					//　変更不可なロールの場合はエラー（システムロール、内部モジュール用ロール）
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.40"));
				} else {
					errMessage = ", " + e.getMessage();
				}
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.29") + errMessage);

			}
		}

		return result;
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// ロールID
		if("".equals(this.textRoleId.getText())){
			this.textRoleId.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.textRoleId.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		// ロール名
		if("".equals(this.textRoleName.getText())){
			this.textRoleName.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.textRoleName.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 指定されたロール情報を元に、ダイアログのテキストボックスにロールID、ロール名、説明をセットする
	 * 
	 * @param roleInfo 設定値として用いるロール情報
	 */
	protected void setInputData(RoleInfo roleInfo) {

		this.inputData = roleInfo;

		// 各項目に反映
		// ロールID
		if (roleInfo.getId() != null) {
			this.textRoleId.setText(roleInfo.getId());
		}
		// ロール名
		if (roleInfo.getName() != null) {
			this.textRoleName.setText(roleInfo.getName());
		}
		// 説明
		if (roleInfo.getDescription() != null) {
			this.textDescription.setText(roleInfo.getDescription());
		}
		// 入力制御 
		if(this.mode == PropertyDefineConstant.MODE_SHOW
				|| this.mode == PropertyDefineConstant.MODE_MODIFY){
			this.textRoleId.setEnabled(false);
		} else {
			this.textRoleId.setEnabled(true);
		}

		// 必須入力項目を可視化
		this.update();
	}

	/**
	 * ダイアログ中で入力されたRoleID・Role名・説明を読みだし、ロール情報として返却する。<BR>
	 * 入力値チェックは行わないため、null文字などが入っている場合もありうる。
	 * 
	 * @return ロール情報
	 */
	private RoleInfo createInputData() {
		final RoleInfo info = new RoleInfo();
		info.setId(this.textRoleId.getText());
		info.setName(this.textRoleName.getText());
		info.setDescription(this.textDescription.getText());
		return info;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		if (this.isModifyDialog) {
			return Messages.getString("modify"); //$NON-NLS-1$
		} else {
			return Messages.getString("register"); //$NON-NLS-1$
		}
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel"); //$NON-NLS-1$
	}
}