/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.composite;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.bean.Property;
import com.clustercontrol.collectiverun.action.GetHistoryTableDefine;
import com.clustercontrol.collectiverun.composite.action.HistorySelectionChangedListener;
import com.clustercontrol.collectiverun.preference.CollectiveRunPreferencePage;
import com.clustercontrol.collectiverun.util.CollectiveRunEndpointWrapper;
import com.clustercontrol.collectiverun.util.HistoryFilterPropertyUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.viewer.CommonTableViewer;
import com.clustercontrol.ws.collectiverun.HistoryFilterInfo;
import com.clustercontrol.ws.collectiverun.HistoryInfo;
import com.clustercontrol.ws.collectiverun.InvalidRole_Exception;
import com.clustercontrol.ws.collectiverun.ViewListInfo;
import com.sun.xml.internal.ws.client.ClientTransportException;

/**
 * 一括制御[履歴]コンポジット
 * 
 * 
 * @version 2.1.1
 * @since 1.0.0
 * 
 */
public class HistoryComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( HistoryComposite.class );

	private CommonTableViewer m_viewer = null;

	private Label m_labelType = null;

	private Label m_labelCount = null;

	private String m_sessionId = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent
	 * @param style
	 */
	public HistoryComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	/**
	 * 初期化処理
	 * 
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		m_labelType = new Label(this, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		m_labelType.setLayoutData(gridData);

		Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.SINGLE);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);

		m_labelCount = new Label(this, SWT.RIGHT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		m_labelCount.setLayoutData(gridData);

		m_viewer = new CommonTableViewer(table);
		m_viewer.createTableColumn(GetHistoryTableDefine.get(),
				GetHistoryTableDefine.SORT_COLUMN_INDEX,
				GetHistoryTableDefine.SORT_ORDER);

		this.update();

		m_viewer.addSelectionChangedListener(
				new HistorySelectionChangedListener(this));
	}

	/**
	 * 更新処理
	 * 
	 * @since 1.0.0
	 */
	@Override
	public void update() {
		ViewListInfo historyInfo = null;

		//履歴情報取得
		try {
			int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
					CollectiveRunPreferencePage.P_HISTORY_MAX_HISTORIES);
			historyInfo = CollectiveRunEndpointWrapper.getHistoryListAll(histories);
		} catch (InvalidRole_Exception e) {
			if(LocalSession.isDialogFlag()){
				LocalSession.setDialogFlag(false);
				m_log.info("update(), " + e.getMessage());
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
				LocalSession.setDialogFlag(true);
			}
		} catch (Exception e) {
			if (LocalSession.isDialogFlag()) {
				LocalSession.setDialogFlag(false);
				if (e instanceof ClientTransportException) {
					m_log.info("update() : " + e.getMessage());
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.transfer") + ", " + e.getMessage());
				} else {
					m_log.warn("update() : " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}
				LocalSession.setDialogFlag(true);
			}
		}

		if (historyInfo == null) {
			historyInfo = new ViewListInfo();
			historyInfo.setTotal(0);
		}

		if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
				CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG)){
			if(historyInfo.getTotal() > historyInfo.getCollectiverunHistoryList().size()){
				if(LocalSession.isDialogFlag()){
					LocalSession.setDialogFlag(false);
					// 最大表示件数を超える場合、エラーダイアログを表示する
					MessageDialogWithToggle.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.job.33"),
							Messages.getString("message.will.not.be.displayed"),
							false,
							ClusterControlPlugin.getDefault().getPreferenceStore(),
							CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG);
					LocalSession.setDialogFlag(true);
				}
			}
		}

		ArrayList<Object> listInput = new ArrayList<Object>();
		for(HistoryInfo info : historyInfo.getCollectiverunHistoryList()){
			ArrayList<Object> line = new ArrayList<Object>();
			line.add(info.getStatus());
			line.add(info.getSessionId());
			line.add(new Date(info.getStartDate()));
			if(info.getEndDate() == null){
				line.add("");
			}
			else{
				line.add(new Date(info.getEndDate()));
			}
			line.add(info.getType());
			line.add(info.getName());
			line.add(info.getFacilityId());
			line.add(info.getScopeText());
			line.add(info.getNodeCount());
			line.add(info.getEndNormalCount());
			line.add(info.getEndWarningCount());
			line.add(info.getEndAbnormalCount());
			line.add(info.getWaitCount());
			line.add(info.getRunningCount());
			listInput.add(line);
		}

		m_viewer.setInput(listInput);

		selectHistory(historyInfo.getCollectiverunHistoryList());

		m_labelType.setText("");
		Object[] args = { new Integer(historyInfo.getTotal()) };
		m_labelCount.setText(Messages.getString("records", args));
	}

	/**
	 * 更新処理
	 * 
	 * @param condition
	 *            フィルタ条件
	 * @since 1.0.0
	 */
	public void update(Property condition) {
		ViewListInfo historyInfo = null;

		//履歴情報取得
		try {
			int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
					CollectiveRunPreferencePage.P_HISTORY_MAX_HISTORIES);
			PropertyUtil.deletePropertyDefine(condition);
			HistoryFilterInfo filter = HistoryFilterPropertyUtil.property2dto(condition);
			historyInfo = CollectiveRunEndpointWrapper.getHistoryList(filter, histories);
		} catch (InvalidRole_Exception e) {
			if(LocalSession.isDialogFlag()){
				LocalSession.setDialogFlag(false);
				m_log.info("update(), " + e.getMessage());
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
				LocalSession.setDialogFlag(true);
			}
		} catch (Exception e) {
			if (LocalSession.isDialogFlag()) {
				LocalSession.setDialogFlag(false);
				if (e instanceof ClientTransportException) {
					m_log.info("update() : " + e.getMessage());
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.transfer") + ", " + e.getMessage());
				} else {
					m_log.warn("update() : " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}
				LocalSession.setDialogFlag(true);
			}
		}

		if (historyInfo == null) {
			historyInfo = new ViewListInfo();
		}

		if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
				CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG)){
			if(historyInfo.getTotal() > historyInfo.getCollectiverunHistoryList().size()){
				if(LocalSession.isDialogFlag()){
					LocalSession.setDialogFlag(false);
					// 最大表示件数を超える場合、エラーダイアログを表示する
					MessageDialogWithToggle.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.job.33"),
							Messages.getString("message.will.not.be.displayed"),
							false,
							ClusterControlPlugin.getDefault().getPreferenceStore(),
							CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG);
					LocalSession.setDialogFlag(true);
				}
			}
		}
		ArrayList<Object> listInput = new ArrayList<Object>();
		for(HistoryInfo info : historyInfo.getCollectiverunHistoryList()){

			ArrayList<Object> line = new ArrayList<Object>();
			line.add(info.getStatus());
			line.add(info.getSessionId());
			if(info.getStartDate() != null)
				line.add(new Date(info.getStartDate()));
			else
				line.add("");
			if(info.getEndDate() != null)
				line.add(new Date(info.getEndDate()));
			else
				line.add("");
			line.add(info.getType());
			line.add(info.getName());
			line.add(info.getFacilityId());
			line.add(info.getScopeText());
			line.add(info.getNodeCount());
			line.add(info.getEndNormalCount());
			line.add(info.getEndWarningCount());
			line.add(info.getEndAbnormalCount());
			line.add(info.getWaitCount());
			line.add(info.getRunningCount());
			listInput.add(line);
		}

		m_viewer.setInput(listInput);

		selectHistory(historyInfo.getCollectiverunHistoryList());

		m_labelType.setText(Messages.getString("filtered.list"));
		Object[] args = { new Integer(historyInfo.getTotal()) };
		m_labelCount.setText(Messages.getString("filtered.records", args));
	}

	/**
	 * 選択処理
	 * 
	 * @param historyInfo
	 *            履歴情報
	 * @since 1.0.0
	 */
	public void selectHistory(List<HistoryInfo> list) {
		if (m_sessionId != null && m_sessionId.length() > 0) {
			int index = -1;
			HistoryInfo info = null;
			for (int i = 0; i < list.size(); i++) {
				info = list.get(i);
				String sessionId = info.getSessionId();
				if (m_sessionId.compareTo(sessionId) == 0) {
					index = i;
					break;
				}
			}
			if (index == -1) {
				m_sessionId = null;
			} else {
				ArrayList<Object> line = new ArrayList<Object>();
				line.add(info.getStatus());
				line.add(info.getSessionId());
				if (info.getStartDate() != null) {
					line.add(new Date(info.getStartDate()));
				} else {
					line.add("");
				}
				if (info.getEndDate() != null) {
					line.add(new Date(info.getEndDate()));
				} else {
					line.add("");
				}
				line.add(info.getType());
				line.add(info.getName());
				line.add(info.getFacilityId());
				line.add(info.getScopeText());
				line.add(info.getNodeCount());
				line.add(info.getEndNormalCount());
				line.add(info.getEndWarningCount());
				line.add(info.getEndAbnormalCount());
				line.add(info.getWaitCount());
				line.add(info.getRunningCount());

				m_viewer.setSelection(new StructuredSelection(line), true);
			}
		}
	}

	/**
	 * テーブルビューワ取得
	 * 
	 * @return m_viewer
	 */
	public CommonTableViewer getTableViewer() {
		return this.m_viewer;
	}

	/**
	 * @return Returns the sessionId.
	 */
	public String getSessionId() {
		return m_sessionId;
	}

	/**
	 * @param sessionId
	 *            The sessionId to set.
	 */
	public void setSessionId(String sessionId) {
		m_sessionId = sessionId;
	}
}