/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;

import com.clustercontrol.accesscontrol.util.ObjectBean;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.composite.JobListComposite;
import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.composite.action.JobListSelectionChangedListener;
import com.clustercontrol.jobmanagement.composite.action.JobTreeSelectionChangedListener;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.view.action.CopyJobAction;
import com.clustercontrol.jobmanagement.view.action.CreateFileJobAction;
import com.clustercontrol.jobmanagement.view.action.CreateJobAction;
import com.clustercontrol.jobmanagement.view.action.CreateJobNetAction;
import com.clustercontrol.jobmanagement.view.action.CreateJobUnitAction;
import com.clustercontrol.jobmanagement.view.action.CreateReferJobAction;
import com.clustercontrol.jobmanagement.view.action.DeleteJobAction;
import com.clustercontrol.jobmanagement.view.action.EditModeAction;
import com.clustercontrol.jobmanagement.view.action.JobObjectPrivilegeAction;
import com.clustercontrol.jobmanagement.view.action.ModifyJobAction;
import com.clustercontrol.jobmanagement.view.action.PasteJobAction;
import com.clustercontrol.jobmanagement.view.action.RunJobAction;
import com.clustercontrol.view.CommonViewPart;
import com.clustercontrol.view.ObjectPrivilegeTargetListView;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[一覧]ビュークラスです。
 * 
 * @version 4.1.0
 * @since 1.0.0
 */
public class JobListView extends CommonViewPart implements ObjectPrivilegeTargetListView {
	// ログ
	private static Log m_log = LogFactory.getLog( JobListView.class );

	/** ビューID */
	public static final String ID = "com.clustercontrol.jobmanagement.view.JobListView";
	/** サッシュ */
	private SashForm m_sash = null;
	/** ジョブツリー用コンポジット */
	private JobTreeComposite m_jobTree = null;
	/** ジョブ[一覧]ビュー用のコンポジット */
	private JobListComposite m_jobList = null;
	/** ジョブツリーアイテム */
	private JobTreeItem m_copyJobTreeItem = null;

	/**
	 * コンストラクタ
	 */
	public JobListView() {
		super();
	}

	/**
	 * ビューを構築します。
	 * 
	 * @param parent 親コンポジット
	 * 
	 * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 * @see #createContextMenu()
	 * @see #update()
	 */
	@Override
	public void createPartControl(Composite parent) {
		GridLayout layout = new GridLayout(1, true);
		parent.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		m_sash = new SashForm(parent, SWT.HORIZONTAL);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		m_sash.setLayoutData(gridData);

		Long start = System.currentTimeMillis();
		//ジョブ階層ツリー作成
		m_jobTree = new JobTreeComposite(this, m_sash, SWT.NONE, null);

		//ジョブ一覧作成
		m_jobList = new JobListComposite(m_sash, SWT.NONE);

		//Sashの境界を調整 左部30% 右部70%
		m_sash.setWeights(new int[] { 30, 70 });

		m_jobTree.getTreeViewer().addSelectionChangedListener(
				new JobTreeSelectionChangedListener(m_jobTree, m_jobList));

		m_jobList.getTableViewer().addSelectionChangedListener(
				new JobListSelectionChangedListener(m_jobList));

		//ポップアップメニュー作成
		createContextMenu();

		Long end = System.currentTimeMillis();

		m_log.info("init() : " + (end - start) + "ms");
		
		m_jobTree.addToTreeViewerList();
		
		//ツリーを開く
		m_jobTree.getTreeViewer().expandToLevel(2);
	}

	/**
	 * コンテキストメニューを作成します。
	 * 
	 * @see org.eclipse.jface.action.MenuManager
	 * @see org.eclipse.swt.widgets.Menu
	 */
	private void createContextMenu() {
		MenuManager menuManager = new MenuManager();
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
			 */
			@Override
			public void menuAboutToShow(IMenuManager manager) {

				//メニュー項目設定
				fillContextMenu(manager);
			}
		});
		Menu treeMenu = menuManager.createContextMenu(m_jobTree.getTree());
		m_jobTree.getTree().setMenu(treeMenu);

		Menu listMenu = menuManager.createContextMenu(m_jobList.getTable());
		m_jobList.getTable().setMenu(listMenu);
	}

	/**
	 * コンテキストメニューに項目を追加します。
	 * 
	 * @param manager メニューマネージャ
	 * 
	 * @see org.eclipse.jface.action.IToolBarManager
	 * @see org.eclipse.jface.action.IMenuManager
	 */
	private void fillContextMenu(IMenuManager manager) {
		//ビューのアクションのインスタンスを取得
		IToolBarManager toolbar = this.getViewSite().getActionBars()
				.getToolBarManager();
		ActionContributionItem createJobUnitAction = (ActionContributionItem) toolbar
				.find(CreateJobUnitAction.ID);
		ActionContributionItem createJobNetAction = (ActionContributionItem) toolbar
				.find(CreateJobNetAction.ID);
		ActionContributionItem createJobAction = (ActionContributionItem) toolbar
				.find(CreateJobAction.ID);
		ActionContributionItem createFileJobAction = (ActionContributionItem) toolbar
				.find(CreateFileJobAction.ID);

		ActionContributionItem createReferJobAction = (ActionContributionItem) toolbar
				.find(CreateReferJobAction.ID);

		ActionContributionItem deleteJobAction = (ActionContributionItem) toolbar
				.find(DeleteJobAction.ID);
		ActionContributionItem modifyJobAction = (ActionContributionItem) toolbar
				.find(ModifyJobAction.ID);
		ActionContributionItem runJobAction = (ActionContributionItem) toolbar
				.find(RunJobAction.ID);
		ActionContributionItem editModeAction = (ActionContributionItem) toolbar
				.find(EditModeAction.ID);
		ActionContributionItem objPrivilege =
				(ActionContributionItem) toolbar.find(JobObjectPrivilegeAction.ID);

		//ポップアップメニューにアクションを追加
		//Ver2.0.0 コピーを追加
		manager.add(editModeAction.getAction());
		manager.add(new Separator());
		CopyJobAction copyJobAction = new CopyJobAction(this);
		manager.add(copyJobAction);
		//Ver2.0.0 貼り付けを追加
		PasteJobAction pasteJobAction = new PasteJobAction(this);
		manager.add(pasteJobAction);
		manager.add(new Separator());
		manager.add(createJobUnitAction.getAction());
		manager.add(createJobNetAction.getAction());
		manager.add(createJobAction.getAction());
		manager.add(createFileJobAction.getAction());
		manager.add(createReferJobAction.getAction());
		manager.add(modifyJobAction.getAction());
		manager.add(deleteJobAction.getAction());
		manager.add(runJobAction.getAction());
		manager.add(new Separator());
		manager.add(objPrivilege.getAction());

		//コピー・貼り付けアクションの使用可/不可を設定
		JobTreeItem select = getSelectJobTreeItem();
		if (select instanceof JobTreeItem) {
			//コピーアクションの使用可/不可を設定
			if(select.getData().getType() == JobConstant.TYPE_COMPOSITE){
				copyJobAction.setEnabled(false);
			} else{
				copyJobAction.setEnabled(true);
			}
			//貼り付けアクションの使用可/不可を設定
			if(m_copyJobTreeItem != null && m_copyJobTreeItem instanceof JobTreeItem){
				// アイテムがコピーされている場合
				if (select.getData().getType() == JobConstant.TYPE_COMPOSITE) {
					// ツリーのトップでは貼り付けできる
					pasteJobAction.setEnabled(true);
				} else {
					// ツリーのトップ以外の場合
					if (JobEditStateUtil.isLockedJobunitId(select.getData().getJobunitId())) {
						// 編集モードに入っている
						if (select.getData().getType() == JobConstant.TYPE_JOBUNIT) {
							// 編集モードの参照可能ジョブユニットでは貼り付けできる
							pasteJobAction.setEnabled(true);
						} else if (select.getData().getType() == JobConstant.TYPE_JOBNET) {
							// 編集モードのジョブネットも貼り付けできる
							pasteJobAction.setEnabled(true);
						} else {
							// それ以外(ジョブ・ファイル転送ジョブ・参照ジョブ)では貼り付けできない
							pasteJobAction.setEnabled(false);
						}
					} else {
						// 編集モードでないときは貼り付けできない
						pasteJobAction.setEnabled(false);
					}
				}
			}else {
				// アイテムがコピーされていない場合
				pasteJobAction.setEnabled(false);
			}
		} else {
			// アイテムが選択されていない場合
			pasteJobAction.setEnabled(false);
			copyJobAction.setEnabled(false);
		}
	}

	/**
	 * ビューを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.composite.JobTreeComposite#update()
	 * @see com.clustercontrol.jobmanagement.composite.JobTreeComposite#getSelectItem()
	 * @see com.clustercontrol.jobmanagement.composite.JobListComposite#update(JobTreeItem)
	 */
	@Override
	public void update() {
		Long start = System.currentTimeMillis();
		JobEditStateUtil.clearEditStateAll();

		m_jobTree.update();
		JobTreeItem item = this.getJobTreeComposite().getSelectItem();
		m_jobList.update(item);
		Long end = System.currentTimeMillis();
		m_log.info("update () : " + (end - start) + "ms");
	}

	/**
	 * ビューを更新します
	 * 指定されたジョブIDに該当するジョブツリーをフォーカスする
	 * @param jobId
	 */
	public void update(String jobId){
		JobEditStateUtil.clearEditStateAll();

		m_jobTree.update(jobId);
		JobTreeItem item = this.getJobTreeComposite().getSelectItem();
		m_jobList.update(item);
	}
	/**
	 * ビューのアクションの有効/無効を設定します。
	 * 
	 * @param type ジョブ種別
	 * @param selection ボタン（アクション）を有効にするための情報
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	public void setEnabledAction(int type, ISelection selection) {
		setEnabledAction(type, "", selection);
	}

	/**
	 * ビューのアクションの有効/無効を設定します。
	 * 
	 * @param type ジョブ種別
	 * @param selection ボタン（アクション）を有効にするための情報
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	public void setEnabledAction(int type, String jobunitId, ISelection selection) {

		super.setEnabledAction(CreateJobUnitAction.ID, selection);
		super.setEnabledAction(CreateJobNetAction.ID, selection);
		super.setEnabledAction(CreateJobAction.ID, selection);
		super.setEnabledAction(CreateFileJobAction.ID, selection);
		super.setEnabledAction(CreateReferJobAction.ID, selection);
		super.setEnabledAction(DeleteJobAction.ID, selection);
		super.setEnabledAction(ModifyJobAction.ID, selection);
		super.setEnabledAction(JobObjectPrivilegeAction.ID, selection);
		super.setEnabledAction(RunJobAction.ID, selection);
		super.setEnabledAction(EditModeAction.ID, selection);

		// 編集モードボタンの押下状態を変更する
		boolean enable = JobEditStateUtil.isLockedJobunitId(jobunitId);
		IToolBarManager toolbar = this.getViewSite().getActionBars().getToolBarManager();
		ActionContributionItem editModeAction = (ActionContributionItem) toolbar.find(EditModeAction.ID);
		editModeAction.getAction().setChecked(enable);

		if (type == JobConstant.TYPE_COMPOSITE) {
			super.setEnabledAction(CreateJobUnitAction.ID, true);
			super.setEnabledAction(CreateJobNetAction.ID, false);
			super.setEnabledAction(CreateJobAction.ID, false);
			super.setEnabledAction(CreateFileJobAction.ID, false);
			super.setEnabledAction(CreateReferJobAction.ID, false);
			super.setEnabledAction(DeleteJobAction.ID, false);
			super.setEnabledAction(ModifyJobAction.ID, false);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, false);
			super.setEnabledAction(EditModeAction.ID, false);
		} else if (type == JobConstant.TYPE_JOBUNIT) {
			super.setEnabledAction(CreateJobUnitAction.ID, false & enable);
			super.setEnabledAction(CreateJobNetAction.ID, true & enable);
			super.setEnabledAction(CreateJobAction.ID, true & enable);
			super.setEnabledAction(CreateFileJobAction.ID, true & enable);
			super.setEnabledAction(CreateReferJobAction.ID, true & enable);
			super.setEnabledAction(DeleteJobAction.ID, true & enable);
			super.setEnabledAction(ModifyJobAction.ID, true);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, true & !enable);
			super.setEnabledAction(RunJobAction.ID, true);
			super.setEnabledAction(EditModeAction.ID, true);
		} else if (type == JobConstant.TYPE_JOBNET) {
			super.setEnabledAction(CreateJobUnitAction.ID, false & enable);
			super.setEnabledAction(CreateJobNetAction.ID, true & enable);
			super.setEnabledAction(CreateJobAction.ID, true & enable);
			super.setEnabledAction(CreateFileJobAction.ID, true & enable);
			super.setEnabledAction(CreateReferJobAction.ID, true & enable);
			super.setEnabledAction(DeleteJobAction.ID, true & enable);
			super.setEnabledAction(ModifyJobAction.ID, true);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, true);
			super.setEnabledAction(EditModeAction.ID, true);
		} else if (type == JobConstant.TYPE_JOB) {
			super.setEnabledAction(CreateJobUnitAction.ID, false);
			super.setEnabledAction(CreateJobNetAction.ID, false);
			super.setEnabledAction(CreateJobAction.ID, false);
			super.setEnabledAction(CreateFileJobAction.ID, false);
			super.setEnabledAction(CreateReferJobAction.ID, false);
			super.setEnabledAction(DeleteJobAction.ID, true & enable);
			super.setEnabledAction(ModifyJobAction.ID, true);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, true);
			super.setEnabledAction(EditModeAction.ID, true);
		} else if (type == JobConstant.TYPE_FILEJOB) {
			super.setEnabledAction(CreateJobUnitAction.ID, false);
			super.setEnabledAction(CreateJobNetAction.ID, false);
			super.setEnabledAction(CreateJobAction.ID, false);
			super.setEnabledAction(CreateFileJobAction.ID, false);
			super.setEnabledAction(CreateReferJobAction.ID, false);
			super.setEnabledAction(DeleteJobAction.ID, true & enable);
			super.setEnabledAction(ModifyJobAction.ID, true);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, true);
			super.setEnabledAction(EditModeAction.ID, true);
		} else if (type == JobConstant.TYPE_REFERJOB) {
			super.setEnabledAction(CreateJobUnitAction.ID, false);
			super.setEnabledAction(CreateJobNetAction.ID, false);
			super.setEnabledAction(CreateJobAction.ID, false);
			super.setEnabledAction(CreateFileJobAction.ID, false);
			super.setEnabledAction(CreateReferJobAction.ID, false);
			super.setEnabledAction(DeleteJobAction.ID, true & enable);
			super.setEnabledAction(ModifyJobAction.ID, true);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, true);
			super.setEnabledAction(EditModeAction.ID, true);
		} else {
			super.setEnabledAction(CreateJobUnitAction.ID, false);
			super.setEnabledAction(CreateJobNetAction.ID, false);
			super.setEnabledAction(CreateJobAction.ID, false);
			super.setEnabledAction(CreateFileJobAction.ID, false);
			super.setEnabledAction(CreateReferJobAction.ID, false);
			super.setEnabledAction(DeleteJobAction.ID, false);
			super.setEnabledAction(ModifyJobAction.ID, false);
			super.setEnabledAction(JobObjectPrivilegeAction.ID, false);
			super.setEnabledAction(RunJobAction.ID, false);
			super.setEnabledAction(EditModeAction.ID, false);
		}

	}

	/**
	 * ジョブツリー用のコンポジットを返します。
	 * 
	 * @return ジョブツリー用のコンポジット
	 */
	public JobTreeComposite getJobTreeComposite() {
		return m_jobTree;
	}

	/**
	 * ジョブ[一覧]ビュー用のコンポジットを返します。
	 * 
	 * @return ジョブ[一覧]ビュー用のコンポジット
	 */
	public JobListComposite getJobListComposite() {
		return m_jobList;
	}

	/**
	 * ジョブツリーを表示します。
	 */
	public void show() {
		m_sash.setMaximizedControl(null);
	}

	/**
	 * ジョブツリーを非表示にします。
	 */
	public void hide() {
		m_sash.setMaximizedControl(m_jobList);
	}

	/**
	 * 選択ジョブツリーアイテムを返します。
	 * 
	 * @return JobTreeItem 選択されたジョブツリーアイテム
	 */
	public JobTreeItem getSelectJobTreeItem() {
		JobTreeItem select = null;
		if (m_jobTree.getTree().isFocusControl()) {
			select = m_jobTree.getSelectItem();
		} else if (m_jobList.getTable().isFocusControl()) {
			select = m_jobList.getSelectJobTreeItem();
		}
		return select;
	}

	/**
	 * コピー元ジョブツリーアイテムを取得します。
	 * 
	 * @return コピー元ジョブツリーアイテム
	 */
	public JobTreeItem getCopyJobTreeItem() {
		return m_copyJobTreeItem;
	}

	/**
	 * コピー元ジョブツリーアイテムを設定します。
	 * 
	 * @param copy コピー元ジョブツリーアイテム
	 */
	public void setCopyJobTreeItem(JobTreeItem copy) {
		m_copyJobTreeItem = copy;
	}

	@Override
	public void dispose() {
		m_jobTree.removeFromTreeViewerList();
		super.dispose();
	}

	@Override
	public List<ObjectBean> getSelectedObjectBeans() {

		// 選択されているスコープを取得する
		JobTreeItem item = getSelectedJobunitItem();

		// 選択されており、スコープの場合は値を返す
		List<ObjectBean> objectBeans = new ArrayList<ObjectBean>();
		if (item != null) {
			String objectId = item.getData().getJobunitId();
			String objectType = HinemosModuleConstant.JOB;
			ObjectBean objectBean = new ObjectBean(objectType, objectId);
			objectBeans.add(objectBean);
		}
		return objectBeans;
	}

	@Override
	public String getSelectedOwnerRoleId() {

		// 選択されているスコープを取得する
		JobTreeItem item = getSelectedJobunitItem();

		// 選択されており、スコープの場合は値を返す
		String ownerRoleId = null;
		if (item != null) {
			ownerRoleId = item.getData().getOwnerRoleId();
		}
		return ownerRoleId;
	}

	private JobTreeItem getSelectedJobunitItem() {

		JobTreeComposite tree = getJobTreeComposite();
		JobListComposite list = getJobListComposite();

		JobTreeItem item = null;
		if(tree.getTree().isFocusControl()){
			item = tree.getSelectItem();
		}
		else if(list.getTableViewer().getTable().isFocusControl()){
			item = list.getSelectItem();
		}
		else{
			// 該当項目なし
		}

		// 選択されており、ジョブユニットの場合は値を返す
		if (item != null && item.getData().getType() == JobConstant.TYPE_JOBUNIT) {
			return item;
		} else {
			return null;
		}
	}
}